package ru.yandex.intranet.d.model.folders;

import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Folder.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class FolderModel {
    private final String id;
    private final TenantId tenantId;
    private final long serviceId;
    private final long version;
    private final String displayName;
    private final String description;
    private final boolean deleted;
    private final FolderType folderType;
    private final Set<String> tags;
    private final long nextOpLogOrder;

    @SuppressWarnings("ParameterNumber")
    public FolderModel(
            String id,
            TenantId tenantId,
            long serviceId,
            long version,
            String displayName,
            String description,
            boolean deleted,
            FolderType folderType,
            Set<String> tags,
            long nextOpLogOrder
    ) {
        this.id = id;
        this.tenantId = tenantId;
        this.serviceId = serviceId;
        this.version = version;
        this.displayName = displayName;
        this.description = description;
        this.deleted = deleted;
        this.folderType = folderType;
        this.tags = Set.of(tags.toArray(String[]::new));
        this.nextOpLogOrder = nextOpLogOrder;
    }

    public Builder toBuilder() {
        return new Builder(
                id,
                tenantId,
                serviceId,
                version,
                displayName,
                description,
                deleted,
                folderType,
                new HashSet<>(tags),
                nextOpLogOrder
        );
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String getId() {
        return id;
    }

    public long getServiceId() {
        return serviceId;
    }

    public String getDisplayName() {
        return displayName;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public long getVersion() {
        return version;
    }

    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    public FolderType getFolderType() {
        return folderType;
    }

    public Set<String> getTags() {
        return tags;
    }

    public long getNextOpLogOrder() {
        return nextOpLogOrder;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FolderModel that = (FolderModel) o;
        return serviceId == that.serviceId &&
                version == that.version &&
                deleted == that.deleted &&
                nextOpLogOrder == that.nextOpLogOrder &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(displayName, that.displayName) &&
                Objects.equals(description, that.description) &&
                folderType == that.folderType &&
                Objects.equals(tags, that.tags);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, serviceId, version, displayName, description, deleted, folderType, tags,
                nextOpLogOrder);
    }

    @Override
    public String toString() {
        return "FolderModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", serviceId=" + serviceId +
                ", version=" + version +
                ", displayName='" + displayName + '\'' +
                ", description='" + description + '\'' +
                ", deleted=" + deleted +
                ", folderType=" + folderType +
                ", tags=" + tags +
                ", nextOpLogOrder=" + nextOpLogOrder +
                '}';
    }

    public static class Builder {
        private String id;
        private TenantId tenantId;
        private long serviceId;
        private long version = 0;
        private String displayName;
        private String description;
        private boolean deleted = false;
        private FolderType folderType = FolderType.COMMON;
        private Set<String> tags = Set.of();
        private long nextOpLogOrder = 0L;

        public Builder() {
        }

        @SuppressWarnings("checkstyle:ParameterNumber")
        public Builder(
                String id,
                TenantId tenantId,
                long serviceId,
                long version,
                String displayName,
                String description,
                boolean deleted,
                FolderType folderType,
                Set<String> tags,
                long nextOpLogOrder
        ) {
            this.id = id;
            this.tenantId = tenantId;
            this.serviceId = serviceId;
            this.version = version;
            this.displayName = displayName;
            this.description = description;
            this.deleted = deleted;
            this.folderType = folderType;
            this.tags = Set.of(tags.toArray(String[]::new));
            this.nextOpLogOrder = nextOpLogOrder;
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public Builder setTenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder setServiceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder setVersion(long version) {
            this.version = version;
            return this;
        }

        public Builder setDisplayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public Builder setDeleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder setFolderType(FolderType folderType) {
            this.folderType = folderType;
            return this;
        }

        public Builder setTags(Set<String> tags) {
            this.tags = tags;
            return this;
        }

        public boolean addTag(String s) {
            return tags.add(s);
        }

        public boolean removeTag(String s) {
            return tags.remove(s);
        }

        public boolean addTags(Collection<String> c) {
            return tags.addAll(c);
        }

        public boolean removeTags(Collection<String> c) {
            return tags.removeAll(c);
        }

        public Builder setNextOpLogOrder(long nextOpLogOrder) {
            this.nextOpLogOrder = nextOpLogOrder;
            return this;
        }

        public FolderModel build() {
            return new FolderModel(
                    id,
                    tenantId,
                    serviceId,
                    version,
                    displayName,
                    description,
                    deleted,
                    folderType,
                    tags,
                    nextOpLogOrder
            );
        }
    }
}
