package ru.yandex.intranet.d.model.folders;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaHistoryModel;

import static java.util.Objects.requireNonNull;

/**
 * FolderOperationLogModel.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 23.10.2020
 */
public final class FolderOperationLogModel {
    private final TenantId tenantId;
    private final Identity identity;
    private final String providerRequestId;
    private final FolderOperationType operationType;
    private final String authorUserId;
    private final String authorUserUid;
    private final String authorProviderId;
    private final String sourceFolderOperationsLogId; // null for source row
    private final String destinationFolderOperationsLogId; // null for destination row
    private final FolderHistoryFieldsModel oldFolderFields;
    private final QuotasByResource oldQuotas; // assured quotas map by resources
    private final QuotasByResource oldBalance; // old values of changed balance by resource
    private final QuotasByAccount oldProvisions; // provided quotas map by accounts and by resources
    private final AccountsHistoryModel oldAccounts;
    private final FolderHistoryFieldsModel newFolderFields;
    private final QuotasByResource newQuotas; // assured quotas map by resources
    private final QuotasByResource newBalance; // new values of changed balance by resource
    private final QuotasByAccount newProvisions; // provided quotas map by accounts and by resources
    private final QuotasByAccount actuallyAppliedProvisions; // provided returned by provider
    private final AccountsHistoryModel newAccounts;
    private final String accountsQuotasOperationsId;
    private final String quotasDemandsId;
    private final OperationPhase operationPhase;
    private final String commentId;
    private final long order;
    private final DeliverableMetaHistoryModel deliveryMeta;
    private final TransferMetaHistoryModel transferMeta;

    public FolderOperationLogModel(Builder builder) {
        this.tenantId = requireNonNull(builder.tenantId);
        this.identity = new Identity(
                builder.folderId,
                builder.operationDateTime,
                builder.id
        );
        this.providerRequestId = builder.providerRequestId;
        this.operationType = builder.operationType;
        this.authorUserId = builder.authorUserId;
        this.authorUserUid = builder.authorUserUid;
        this.authorProviderId = builder.authorProviderId;
        this.sourceFolderOperationsLogId = builder.sourceFolderOperationsLogId;
        this.destinationFolderOperationsLogId = builder.destinationFolderOperationsLogId;
        this.oldFolderFields = builder.oldFolderFields;
        this.oldQuotas = builder.oldQuotas;
        this.oldBalance = builder.oldBalance;
        this.oldProvisions = builder.oldProvisions;
        this.oldAccounts = builder.oldAccounts;
        this.newFolderFields = builder.newFolderFields;
        this.newQuotas = builder.newQuotas;
        this.newBalance = builder.newBalance;
        this.newProvisions = builder.newProvisions;
        this.actuallyAppliedProvisions = builder.actuallyAppliedProvisions;
        this.newAccounts = builder.newAccounts;
        this.accountsQuotasOperationsId = builder.accountsQuotasOperationsId;
        this.quotasDemandsId = builder.quotasDemandsId;
        this.operationPhase = builder.operationPhase;
        this.order = builder.order;
        this.commentId = builder.commentId;
        this.deliveryMeta = builder.deliveryMeta;
        this.transferMeta = builder.transferMeta;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getFolderId() {
        return identity.folderId;
    }

    public Instant getOperationDateTime() {
        return identity.operationDateTime;
    }

    public String getId() {
        return identity.id;
    }

    public Optional<String> getProviderRequestId() {
        return Optional.ofNullable(providerRequestId);
    }

    public FolderOperationType getOperationType() {
        return operationType;
    }

    public Optional<String> getAuthorUserId() {
        return Optional.ofNullable(authorUserId);
    }

    public Optional<String> getAuthorUserUid() {
        return Optional.ofNullable(authorUserUid);
    }

    public Optional<String> getAuthorProviderId() {
        return Optional.ofNullable(authorProviderId);
    }

    public Optional<String> getSourceFolderOperationsLogId() {
        return Optional.ofNullable(sourceFolderOperationsLogId);
    }

    public Optional<String> getDestinationFolderOperationsLogId() {
        return Optional.ofNullable(destinationFolderOperationsLogId);
    }

    public Optional<FolderHistoryFieldsModel> getOldFolderFields() {
        return Optional.ofNullable(oldFolderFields);
    }

    public QuotasByResource getOldQuotas() {
        return oldQuotas;
    }

    public QuotasByResource getOldBalance() {
        return oldBalance;
    }

    public QuotasByAccount getOldProvisions() {
        return oldProvisions;
    }

    public Optional<AccountsHistoryModel> getOldAccounts() {
        return Optional.ofNullable(oldAccounts);
    }

    public Optional<FolderHistoryFieldsModel> getNewFolderFields() {
        return Optional.ofNullable(newFolderFields);
    }

    public QuotasByResource getNewQuotas() {
        return newQuotas;
    }

    public Optional<QuotasByAccount> getActuallyAppliedProvisions() {
        return Optional.ofNullable(actuallyAppliedProvisions);
    }

    public QuotasByResource getNewBalance() {
        return newBalance;
    }

    public QuotasByAccount getNewProvisions() {
        return newProvisions;
    }

    public Optional<AccountsHistoryModel> getNewAccounts() {
        return Optional.ofNullable(newAccounts);
    }

    public Optional<String> getAccountsQuotasOperationsId() {
        return Optional.ofNullable(accountsQuotasOperationsId);
    }

    public Optional<String> getQuotasDemandsId() {
        return Optional.ofNullable(quotasDemandsId);
    }

    public Optional<OperationPhase> getOperationPhase() {
        return Optional.ofNullable(operationPhase);
    }

    public long getOrder() {
        return order;
    }

    public Optional<String> getCommentId() {
        return Optional.ofNullable(commentId);
    }

    public Optional<DeliverableMetaHistoryModel> getDeliveryMeta() {
        return Optional.ofNullable(deliveryMeta);
    }

    public Identity getIdentity() {
        return identity;
    }

    public Optional<TransferMetaHistoryModel> getTransferMeta() {
        return Optional.ofNullable(transferMeta);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FolderOperationLogModel that = (FolderOperationLogModel) o;
        return order == that.order &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(identity, that.identity) &&
                Objects.equals(providerRequestId, that.providerRequestId) &&
                operationType == that.operationType &&
                Objects.equals(authorUserId, that.authorUserId) &&
                Objects.equals(authorUserUid, that.authorUserUid) &&
                Objects.equals(authorProviderId, that.authorProviderId) &&
                Objects.equals(sourceFolderOperationsLogId, that.sourceFolderOperationsLogId) &&
                Objects.equals(destinationFolderOperationsLogId, that.destinationFolderOperationsLogId) &&
                Objects.equals(oldFolderFields, that.oldFolderFields) &&
                Objects.equals(oldQuotas, that.oldQuotas) &&
                Objects.equals(oldBalance, that.oldBalance) &&
                Objects.equals(oldProvisions, that.oldProvisions) &&
                Objects.equals(oldAccounts, that.oldAccounts) &&
                Objects.equals(newFolderFields, that.newFolderFields) &&
                Objects.equals(newQuotas, that.newQuotas) &&
                Objects.equals(newBalance, that.newBalance) &&
                Objects.equals(newProvisions, that.newProvisions) &&
                Objects.equals(actuallyAppliedProvisions, that.actuallyAppliedProvisions) &&
                Objects.equals(newAccounts, that.newAccounts) &&
                Objects.equals(accountsQuotasOperationsId, that.accountsQuotasOperationsId) &&
                Objects.equals(quotasDemandsId, that.quotasDemandsId) &&
                operationPhase == that.operationPhase &&
                Objects.equals(commentId, that.commentId) &&
                Objects.equals(deliveryMeta, that.deliveryMeta) &&
                Objects.equals(transferMeta, that.transferMeta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, identity, providerRequestId, operationType, authorUserId, authorUserUid,
                authorProviderId, sourceFolderOperationsLogId, destinationFolderOperationsLogId, oldFolderFields,
                oldQuotas, oldBalance, oldProvisions, oldAccounts, newFolderFields, newQuotas, newBalance,
                newProvisions, actuallyAppliedProvisions, newAccounts, accountsQuotasOperationsId, quotasDemandsId,
                operationPhase, commentId, order, deliveryMeta, transferMeta);
    }

    @Override
    public String toString() {
        return "FolderOperationLogModel{" +
                "tenantId=" + tenantId +
                ", identity=" + identity +
                ", providerRequestId='" + providerRequestId + '\'' +
                ", operationType=" + operationType +
                ", authorUserId='" + authorUserId + '\'' +
                ", authorUserUid='" + authorUserUid + '\'' +
                ", authorProviderId='" + authorProviderId + '\'' +
                ", sourceFolderOperationsLogId='" + sourceFolderOperationsLogId + '\'' +
                ", destinationFolderOperationsLogId='" + destinationFolderOperationsLogId + '\'' +
                ", oldFolderFields=" + oldFolderFields +
                ", oldQuotas=" + oldQuotas +
                ", oldBalance=" + oldBalance +
                ", oldProvisions=" + oldProvisions +
                ", oldAccounts=" + oldAccounts +
                ", newFolderFields=" + newFolderFields +
                ", newQuotas=" + newQuotas +
                ", newBalance=" + newBalance +
                ", newProvisions=" + newProvisions +
                ", actuallyAppliedProvisions=" + actuallyAppliedProvisions +
                ", newAccounts=" + newAccounts +
                ", accountsQuotasOperationsId='" + accountsQuotasOperationsId + '\'' +
                ", quotasDemandsId='" + quotasDemandsId + '\'' +
                ", operationPhase=" + operationPhase +
                ", commentId='" + commentId + '\'' +
                ", order=" + order +
                ", deliveryMeta=" + deliveryMeta +
                ", transferMeta=" + transferMeta +
                '}';
    }

    public static final class Identity {
        private final String folderId;
        private final Instant operationDateTime;
        private final String id;

        public Identity(String folderId, Instant operationDateTime, String id) {
            this.folderId = requireNonNull(folderId);
            this.operationDateTime = requireNonNull(operationDateTime);
            this.id = requireNonNull(id);
        }

        public String getFolderId() {
            return folderId;
        }

        public Instant getOperationDateTime() {
            return operationDateTime;
        }

        public String getId() {
            return id;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Identity identity = (Identity) o;
            return folderId.equals(identity.folderId) &&
                    operationDateTime.equals(identity.operationDateTime) &&
                    id.equals(identity.id);
        }

        @Override
        public int hashCode() {
            return Objects.hash(folderId, operationDateTime, id);
        }

        @Override
        public String toString() {
            return "FolderOperationLogModel.Identity{" +
                    "folderId='" + folderId + '\'' +
                    ", operationDateTime=" + operationDateTime +
                    ", id='" + id + '\'' +
                    '}';
        }
    }

    public Builder copyBuilder() {
        return builder()
                .setTenantId(tenantId)
                .setFolderId(identity.getFolderId())
                .setOperationDateTime(getOperationDateTime())
                .setId(getId())
                .setProviderRequestId(providerRequestId)
                .setOperationType(operationType)
                .setAuthorUserId(authorUserId)
                .setAuthorUserUid(authorUserUid)
                .setAuthorProviderId(authorProviderId)
                .setSourceFolderOperationsLogId(sourceFolderOperationsLogId)
                .setDestinationFolderOperationsLogId(destinationFolderOperationsLogId)
                .setOldFolderFields(oldFolderFields)
                .setOldQuotas(oldQuotas)
                .setOldBalance(oldBalance)
                .setOldProvisions(oldProvisions)
                .setNewFolderFields(newFolderFields)
                .setNewQuotas(newQuotas)
                .setNewBalance(newBalance)
                .setNewProvisions(newProvisions)
                .setActuallyAppliedProvisions(actuallyAppliedProvisions)
                .setNewAccounts(newAccounts)
                .setAccountsQuotasOperationsId(accountsQuotasOperationsId)
                .setQuotasDemandsId(quotasDemandsId)
                .setOperationPhase(operationPhase)
                .setOrder(order)
                .setCommentId(commentId)
                .setDeliveryMeta(deliveryMeta);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {

        private TenantId tenantId;
        private String folderId;
        private Instant operationDateTime;
        private String id;
        private String providerRequestId;
        private FolderOperationType operationType;
        private String authorUserId;
        private String authorUserUid;
        private String authorProviderId;
        private String sourceFolderOperationsLogId;
        private String destinationFolderOperationsLogId;
        private FolderHistoryFieldsModel oldFolderFields;
        private QuotasByResource oldQuotas;
        private QuotasByResource oldBalance;
        private QuotasByAccount oldProvisions = new QuotasByAccount(Map.of());
        private AccountsHistoryModel oldAccounts;
        private FolderHistoryFieldsModel newFolderFields;
        private QuotasByResource newQuotas;
        private QuotasByResource newBalance;
        private QuotasByAccount newProvisions = new QuotasByAccount(Map.of());
        private QuotasByAccount actuallyAppliedProvisions;
        private AccountsHistoryModel newAccounts;
        private String accountsQuotasOperationsId;
        private String quotasDemandsId;
        private OperationPhase operationPhase;
        private long order;
        private String commentId;
        private DeliverableMetaHistoryModel deliveryMeta;
        private Map<String, Map<String, ProvisionHistoryModel>> defaultQuotasByAccountIdByResourceId;
        private TransferMetaHistoryModel transferMeta;

        public Builder setTenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder setFolderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder setOperationDateTime(Instant operationDateTime) {
            this.operationDateTime = operationDateTime;
            return this;
        }

        public Builder setId(String id) {
            this.id = id;
            return this;
        }

        public Builder setProviderRequestId(String providerRequestId) {
            this.providerRequestId = providerRequestId;
            return this;
        }

        public Builder setOperationType(FolderOperationType operationType) {
            this.operationType = operationType;
            return this;
        }

        public Builder setAuthorUserId(String authorUserId) {
            this.authorUserId = authorUserId;
            return this;
        }

        public Builder setAuthorUserUid(String authorUserUid) {
            this.authorUserUid = authorUserUid;
            return this;
        }

        public Builder setAuthorProviderId(String authorProviderId) {
            this.authorProviderId = authorProviderId;
            return this;
        }

        public Builder setSourceFolderOperationsLogId(String sourceFolderOperationsLogId) {
            this.sourceFolderOperationsLogId = sourceFolderOperationsLogId;
            return this;
        }

        public Builder setDestinationFolderOperationsLogId(String destinationFolderOperationsLogId) {
            this.destinationFolderOperationsLogId = destinationFolderOperationsLogId;
            return this;
        }

        public Builder setOldFolderFields(FolderHistoryFieldsModel oldFolderFields) {
            this.oldFolderFields = oldFolderFields;
            return this;
        }

        public Builder setOldQuotas(QuotasByResource oldQuotas) {
            this.oldQuotas = oldQuotas;
            return this;
        }

        public Builder setOldBalance(QuotasByResource oldBalance) {
            this.oldBalance = oldBalance;
            return this;
        }

        public Builder setOldProvisions(QuotasByAccount oldProvisions) {
            this.oldProvisions = oldProvisions;
            return this;
        }

        public Builder setOldAccounts(AccountsHistoryModel oldAccounts) {
            this.oldAccounts = oldAccounts;
            return this;
        }

        public Builder setNewFolderFields(FolderHistoryFieldsModel newFolderFields) {
            this.newFolderFields = newFolderFields;
            return this;
        }

        public Builder setNewQuotas(QuotasByResource newQuotas) {
            this.newQuotas = newQuotas;
            return this;
        }

        public Builder setNewBalance(QuotasByResource newBalance) {
            this.newBalance = newBalance;
            return this;
        }

        public Builder setNewProvisions(QuotasByAccount newProvisions) {
            this.newProvisions = newProvisions;
            return this;
        }

        public Builder setActuallyAppliedProvisions(QuotasByAccount actuallyAppliedProvisions) {
            this.actuallyAppliedProvisions = actuallyAppliedProvisions;
            return this;
        }

        public Builder setNewAccounts(AccountsHistoryModel newAccounts) {
            this.newAccounts = newAccounts;
            return this;
        }

        public Builder setAccountsQuotasOperationsId(String accountsQuotasOperationsId) {
            this.accountsQuotasOperationsId = accountsQuotasOperationsId;
            return this;
        }

        public Builder setQuotasDemandsId(String quotasDemandsId) {
            this.quotasDemandsId = quotasDemandsId;
            return this;
        }

        public Builder setOperationPhase(OperationPhase operationPhase) {
            this.operationPhase = operationPhase;
            return this;
        }

        public Builder setOrder(long order) {
            this.order = order;
            return this;
        }

        public Builder setCommentId(String commentId) {
            this.commentId = commentId;
            return this;
        }

        public Builder setDeliveryMeta(DeliverableMetaHistoryModel deliveryMeta) {
            this.deliveryMeta = deliveryMeta;
            return this;
        }

        public Builder setTransferMeta(TransferMetaHistoryModel transferMeta) {
            this.transferMeta = transferMeta;
            return this;
        }

        public Optional<String> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<String> getFolderId() {
            return Optional.ofNullable(folderId);
        }

        public Optional<String> getAccountsQuotasOperationsId() {
            return Optional.ofNullable(accountsQuotasOperationsId);
        }

        public Optional<AccountsHistoryModel> getOldAccounts() {
            return Optional.ofNullable(oldAccounts);
        }

        public Optional<AccountsHistoryModel> getNewAccounts() {
            return Optional.ofNullable(newAccounts);
        }

        public Optional<QuotasByAccount> getOldProvisions() {
            return Optional.ofNullable(oldProvisions);
        }

        public Optional<QuotasByAccount> getNewProvisions() {
            return Optional.ofNullable(newProvisions);
        }

        public Map<String, Map<String, ProvisionHistoryModel>> getDefaultQuotasByAccountIdByResourceId() {
            if (defaultQuotasByAccountIdByResourceId == null) {
                defaultQuotasByAccountIdByResourceId = new HashMap<>();
            }
            return defaultQuotasByAccountIdByResourceId;
        }

        public FolderOperationLogModel build() {
            return new FolderOperationLogModel(this);
        }
    }

}
