package ru.yandex.intranet.d.model.providers;

import java.util.Collections;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;

/**
 * Accounts settings.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AccountsSettingsModel {

    public static final long DEFAULT_ACCOUNTS_SYNC_PAGE_SIZE = 1000L;

    private final boolean displayNameSupported;
    private final boolean keySupported;
    private final boolean deleteSupported;
    private final boolean softDeleteSupported;
    private final boolean moveSupported;
    private final boolean renameSupported;
    private final boolean perAccountVersionSupported;
    private final boolean perProvisionVersionSupported;
    private final boolean perAccountLastUpdateSupported;
    private final boolean perProvisionLastUpdateSupported;
    private final boolean operationIdDeduplicationSupported;
    private final boolean syncCoolDownDisabled;
    private final boolean retryCoolDownDisabled;
    private final long accountsSyncPageSize;
    private final boolean moveProvisionSupported;
    private final Set<ExternalAccountUrlTemplate> externalAccountUrlTemplates;
    private final Boolean multipleReservesAllowed;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public AccountsSettingsModel(
            boolean displayNameSupported,
            boolean keySupported,
            boolean deleteSupported,
            boolean softDeleteSupported,
            boolean moveSupported,
            boolean renameSupported,
            boolean perAccountVersionSupported,
            boolean perProvisionVersionSupported,
            boolean perAccountLastUpdateSupported,
            boolean perProvisionLastUpdateSupported,
            boolean operationIdDeduplicationSupported,
            boolean syncCoolDownDisabled,
            boolean retryCoolDownDisabled,
            long accountsSyncPageSize,
            boolean moveProvisionSupported,
            Set<ExternalAccountUrlTemplate> externalAccountUrlTemplates,
            Boolean multipleReservesAllowed
    ) {
        this.displayNameSupported = displayNameSupported;
        this.keySupported = keySupported;
        this.deleteSupported = deleteSupported;
        this.softDeleteSupported = softDeleteSupported;
        this.moveSupported = moveSupported;
        this.renameSupported = renameSupported;
        this.perAccountVersionSupported = perAccountVersionSupported;
        this.perProvisionVersionSupported = perProvisionVersionSupported;
        this.perAccountLastUpdateSupported = perAccountLastUpdateSupported;
        this.perProvisionLastUpdateSupported = perProvisionLastUpdateSupported;
        this.operationIdDeduplicationSupported = operationIdDeduplicationSupported;
        this.syncCoolDownDisabled = syncCoolDownDisabled;
        this.retryCoolDownDisabled = retryCoolDownDisabled;
        this.accountsSyncPageSize = accountsSyncPageSize;
        this.moveProvisionSupported = moveProvisionSupported;
        this.externalAccountUrlTemplates = externalAccountUrlTemplates;
        this.multipleReservesAllowed = multipleReservesAllowed;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(AccountsSettingsModel value) {
        return new Builder(value);
    }

    public boolean isDisplayNameSupported() {
        return displayNameSupported;
    }

    public boolean isKeySupported() {
        return keySupported;
    }

    public boolean isDeleteSupported() {
        return deleteSupported;
    }

    public boolean isSoftDeleteSupported() {
        return softDeleteSupported;
    }

    public boolean isMoveSupported() {
        return moveSupported;
    }

    public boolean isRenameSupported() {
        return renameSupported;
    }

    public boolean isPerAccountVersionSupported() {
        return perAccountVersionSupported;
    }

    public boolean isPerProvisionVersionSupported() {
        return perProvisionVersionSupported;
    }

    public boolean isPerAccountLastUpdateSupported() {
        return perAccountLastUpdateSupported;
    }

    public boolean isPerProvisionLastUpdateSupported() {
        return perProvisionLastUpdateSupported;
    }

    public boolean isOperationIdDeduplicationSupported() {
        return operationIdDeduplicationSupported;
    }

    public boolean isSyncCoolDownDisabled() {
        return syncCoolDownDisabled;
    }

    public boolean isRetryCoolDownDisabled() {
        return retryCoolDownDisabled;
    }

    public long getAccountsSyncPageSize() {
        return accountsSyncPageSize != 0 ? accountsSyncPageSize : DEFAULT_ACCOUNTS_SYNC_PAGE_SIZE;
    }

    public boolean isMoveProvisionSupported() {
        return moveProvisionSupported;
    }

    public Set<ExternalAccountUrlTemplate> getExternalAccountUrlTemplates() {
        return externalAccountUrlTemplates != null ? externalAccountUrlTemplates : Collections.emptySet();
    }

    public Optional<Boolean> getMultipleReservesAllowed() {
        return Optional.ofNullable(multipleReservesAllowed);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountsSettingsModel that = (AccountsSettingsModel) o;
        return displayNameSupported == that.displayNameSupported
                && keySupported == that.keySupported
                && deleteSupported == that.deleteSupported
                && softDeleteSupported == that.softDeleteSupported
                && moveSupported == that.moveSupported
                && renameSupported == that.renameSupported
                && perAccountVersionSupported == that.perAccountVersionSupported
                && perProvisionVersionSupported == that.perProvisionVersionSupported
                && perAccountLastUpdateSupported == that.perAccountLastUpdateSupported
                && perProvisionLastUpdateSupported == that.perProvisionLastUpdateSupported
                && operationIdDeduplicationSupported == that.operationIdDeduplicationSupported
                && syncCoolDownDisabled == that.syncCoolDownDisabled
                && retryCoolDownDisabled == that.retryCoolDownDisabled
                && accountsSyncPageSize == that.accountsSyncPageSize
                && moveProvisionSupported == that.moveProvisionSupported &&
                Objects.equals(externalAccountUrlTemplates, that.externalAccountUrlTemplates) &&
                Objects.equals(multipleReservesAllowed, that.multipleReservesAllowed);
    }

    @Override
    public int hashCode() {
        return Objects.hash(displayNameSupported, keySupported, deleteSupported, softDeleteSupported, moveSupported,
                renameSupported, perAccountVersionSupported, perProvisionVersionSupported,
                perAccountLastUpdateSupported, perProvisionLastUpdateSupported, operationIdDeduplicationSupported,
                syncCoolDownDisabled, retryCoolDownDisabled, accountsSyncPageSize, moveProvisionSupported,
                externalAccountUrlTemplates, multipleReservesAllowed);
    }

    @Override
    public String toString() {
        return "AccountsSettingsModel{" +
                "displayNameSupported=" + displayNameSupported +
                ", keySupported=" + keySupported +
                ", deleteSupported=" + deleteSupported +
                ", softDeleteSupported=" + softDeleteSupported +
                ", moveSupported=" + moveSupported +
                ", renameSupported=" + renameSupported +
                ", perAccountVersionSupported=" + perAccountVersionSupported +
                ", perProvisionVersionSupported=" + perProvisionVersionSupported +
                ", perAccountLastUpdateSupported=" + perAccountLastUpdateSupported +
                ", perProvisionLastUpdateSupported=" + perProvisionLastUpdateSupported +
                ", operationIdDeduplicationSupported=" + operationIdDeduplicationSupported +
                ", syncCoolDownDisabled=" + syncCoolDownDisabled +
                ", retryCoolDownDisabled=" + retryCoolDownDisabled +
                ", accountsSyncPageSize=" + accountsSyncPageSize +
                ", moveProvisionSupported=" + moveProvisionSupported +
                ", externalAccountUrlTemplate='" + externalAccountUrlTemplates + '\'' +
                ", multipleReservesAllowed=" + multipleReservesAllowed +
                '}';
    }

    public static class Builder {

        private Boolean displayNameSupported;
        private Boolean keySupported;
        private Boolean deleteSupported;
        private Boolean softDeleteSupported;
        private Boolean moveSupported;
        private Boolean renameSupported;
        private Boolean perAccountVersionSupported;
        private Boolean perProvisionVersionSupported;
        private Boolean perAccountLastUpdateSupported;
        private Boolean perProvisionLastUpdateSupported;
        private Boolean operationIdDeduplicationSupported;
        private Boolean syncCoolDownDisabled;
        private Boolean retryCoolDownDisabled;
        private Long accountsSyncPageSize;
        private Boolean moveProvisionSupported = Boolean.FALSE;
        private Set<ExternalAccountUrlTemplate> externalAccountUrlTemplates;
        private Boolean multipleReservesAllowed;

        private Builder() {
        }

        private Builder(AccountsSettingsModel value) {
            this.displayNameSupported = value.isDisplayNameSupported();
            this.keySupported = value.isKeySupported();
            this.deleteSupported = value.isDeleteSupported();
            this.softDeleteSupported = value.isSoftDeleteSupported();
            this.moveSupported = value.isMoveSupported();
            this.renameSupported = value.isRenameSupported();
            this.perAccountVersionSupported = value.isPerAccountVersionSupported();
            this.perProvisionVersionSupported = value.isPerProvisionVersionSupported();
            this.perAccountLastUpdateSupported = value.isPerAccountLastUpdateSupported();
            this.perProvisionLastUpdateSupported = value.isPerProvisionLastUpdateSupported();
            this.operationIdDeduplicationSupported = value.isOperationIdDeduplicationSupported();
            this.syncCoolDownDisabled = value.isSyncCoolDownDisabled();
            this.retryCoolDownDisabled = value.isRetryCoolDownDisabled();
            this.accountsSyncPageSize = value.getAccountsSyncPageSize();
            this.moveProvisionSupported = value.isMoveProvisionSupported();
            this.externalAccountUrlTemplates = value.getExternalAccountUrlTemplates();
            this.multipleReservesAllowed = value.getMultipleReservesAllowed().orElse(null);
        }

        public Builder displayNameSupported(boolean displayNameSupported) {
            this.displayNameSupported = displayNameSupported;
            return this;
        }

        public Builder keySupported(boolean keySupported) {
            this.keySupported = keySupported;
            return this;
        }

        public Builder deleteSupported(boolean deleteSupported) {
            this.deleteSupported = deleteSupported;
            return this;
        }

        public Builder softDeleteSupported(boolean softDeleteSupported) {
            this.softDeleteSupported = softDeleteSupported;
            return this;
        }

        public Builder moveSupported(boolean moveSupported) {
            this.moveSupported = moveSupported;
            return this;
        }

        public Builder renameSupported(boolean renameSupported) {
            this.renameSupported = renameSupported;
            return this;
        }

        public Builder perAccountVersionSupported(boolean perAccountVersionSupported) {
            this.perAccountVersionSupported = perAccountVersionSupported;
            return this;
        }

        public Builder perProvisionVersionSupported(boolean perProvisionVersionSupported) {
            this.perProvisionVersionSupported = perProvisionVersionSupported;
            return this;
        }

        public Builder perAccountLastUpdateSupported(boolean perAccountLastUpdateSupported) {
            this.perAccountLastUpdateSupported = perAccountLastUpdateSupported;
            return this;
        }

        public Builder perProvisionLastUpdateSupported(boolean perProvisionLastUpdateSupported) {
            this.perProvisionLastUpdateSupported = perProvisionLastUpdateSupported;
            return this;
        }

        public Builder operationIdDeduplicationSupported(boolean operationIdDeduplicationSupported) {
            this.operationIdDeduplicationSupported = operationIdDeduplicationSupported;
            return this;
        }

        public Builder syncCoolDownDisabled(boolean syncCoolDownDisabled) {
            this.syncCoolDownDisabled = syncCoolDownDisabled;
            return this;
        }

        public Builder retryCoolDownDisabled(boolean retryCoolDownDisabled) {
            this.retryCoolDownDisabled = retryCoolDownDisabled;
            return this;
        }

        public Builder accountsSyncPageSize(long accountsSyncPageSize) {
            this.accountsSyncPageSize = accountsSyncPageSize;
            return this;
        }

        public Builder moveProvisionSupported(boolean moveProvisionSupported) {
            this.moveProvisionSupported = moveProvisionSupported;
            return this;
        }

        public Builder externalAccountUrlTemplates(Set<ExternalAccountUrlTemplate> externalAccountUrlTemplates) {
            this.externalAccountUrlTemplates = externalAccountUrlTemplates;
            return this;
        }

        public Builder multipleReservesAllowed(Boolean multipleReservesAllowed) {
            this.multipleReservesAllowed = multipleReservesAllowed;
            return this;
        }

        public Optional<Long> getAccountsSyncPageSize() {
            return Optional.ofNullable(accountsSyncPageSize);
        }

        public Optional<Boolean> getDisplayNameSupported() {
            return Optional.ofNullable(displayNameSupported);
        }

        public Optional<Boolean> getKeySupported() {
            return Optional.ofNullable(keySupported);
        }

        public Optional<Boolean> getDeleteSupported() {
            return Optional.ofNullable(deleteSupported);
        }

        public Optional<Boolean> getSoftDeleteSupported() {
            return Optional.ofNullable(softDeleteSupported);
        }

        public Optional<Boolean> getMoveSupported() {
            return Optional.ofNullable(moveSupported);
        }

        public Optional<Boolean> getRenameSupported() {
            return Optional.ofNullable(renameSupported);
        }

        public Optional<Boolean> getPerAccountVersionSupported() {
            return Optional.ofNullable(perAccountVersionSupported);
        }

        public Optional<Boolean> getPerProvisionVersionSupported() {
            return Optional.ofNullable(perProvisionVersionSupported);
        }

        public Optional<Boolean> getPerAccountLastUpdateSupported() {
            return Optional.ofNullable(perAccountLastUpdateSupported);
        }

        public Optional<Boolean> getPerProvisionLastUpdateSupported() {
            return Optional.ofNullable(perProvisionLastUpdateSupported);
        }

        public Optional<Boolean> getOperationIdDeduplicationSupported() {
            return Optional.ofNullable(operationIdDeduplicationSupported);
        }

        public Optional<Boolean> getSyncCoolDownDisabled() {
            return Optional.ofNullable(syncCoolDownDisabled);
        }

        public Optional<Boolean> getRetryCoolDownDisabled() {
            return Optional.ofNullable(retryCoolDownDisabled);
        }

        public Optional<Boolean> getMoveProvisionSupported() {
            return Optional.ofNullable(moveProvisionSupported);
        }

        public Set<ExternalAccountUrlTemplate> getExternalAccountUrlTemplates() {
            return externalAccountUrlTemplates;
        }

        public Optional<Boolean> getMultipleReservesAllowed() {
            return Optional.ofNullable(multipleReservesAllowed);
        }

        public AccountsSettingsModel build() {
            Preconditions.checkNotNull(displayNameSupported, "DisplayNameSupported is required");
            Preconditions.checkNotNull(keySupported, "KeySupported is required");
            Preconditions.checkNotNull(deleteSupported, "DeleteSupported is required");
            Preconditions.checkNotNull(softDeleteSupported, "SoftDeleteSupported is required");
            Preconditions.checkNotNull(moveSupported, "MoveSupported is required");
            Preconditions.checkNotNull(renameSupported, "RenameSupported is required");
            Preconditions.checkNotNull(perAccountVersionSupported, "PerAccountVersionSupported is required");
            Preconditions.checkNotNull(perProvisionVersionSupported, "PerProvisionVersionSupported is required");
            Preconditions.checkNotNull(perAccountLastUpdateSupported, "PerAccountLastUpdateSupported is required");
            Preconditions.checkNotNull(perProvisionLastUpdateSupported, "PerProvisionLastUpdateSupported is required");
            Preconditions.checkNotNull(operationIdDeduplicationSupported,
                    "OperationIdDeduplicationSupported is required");
            Preconditions.checkNotNull(syncCoolDownDisabled, "SyncCoolDownDisabled is required");
            Preconditions.checkNotNull(retryCoolDownDisabled, "RetryCoolDownDisabled is required");
            Preconditions.checkNotNull(accountsSyncPageSize, "AccountsSyncPageSize is required");
            return new AccountsSettingsModel(displayNameSupported, keySupported, deleteSupported, softDeleteSupported,
                    moveSupported, renameSupported, perAccountVersionSupported, perProvisionVersionSupported,
                    perAccountLastUpdateSupported, perProvisionLastUpdateSupported, operationIdDeduplicationSupported,
                    syncCoolDownDisabled, retryCoolDownDisabled, accountsSyncPageSize, moveProvisionSupported,
                    externalAccountUrlTemplates, multipleReservesAllowed);
        }

    }

}
