package ru.yandex.intranet.d.model.providers;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Provider.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ProviderModel {

    private final String id;
    private final TenantId tenantId;
    private final long version;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final String restApiUri;
    private final String grpcApiUri;
    private final long sourceTvmId;
    private final long destinationTvmId;
    private final long serviceId;
    private final boolean deleted;
    private final boolean readOnly;
    private final boolean multipleAccountsPerFolder;
    private final boolean accountTransferWithQuota;
    private final boolean managed;
    private final String key;
    private final AccountsSettingsModel accountsSettings;
    private final boolean importAllowed;
    private final boolean accountsSpacesSupported;
    private final boolean syncEnabled;
    private final boolean grpcTlsOn;
    private final BillingMeta billingMeta;
    private final Map<String, RelatedResourceMapping> relatedResourcesByResourceId;
    private final long trackerComponentId;
    private final String reserveFolderId;
    private final boolean hasDefaultQuotas;
    // Null means unspecified, use appropriate default value
    private final Boolean allocatedSupported;
    // Null means unspecified, use appropriate default value
    private final @Nullable AggregationSettings aggregationSettings;
    private final @Nullable AggregationAlgorithm aggregationAlgorithm;
    private final @Nullable ProviderUISettings uiSettings;

    @SuppressWarnings("ParameterNumber")
    public ProviderModel(
            String id,
            TenantId tenantId,
            long version,
            String nameEn,
            String nameRu,
            String descriptionEn,
            String descriptionRu,
            String restApiUri,
            String grpcApiUri,
            long sourceTvmId,
            long destinationTvmId,
            long serviceId,
            boolean deleted,
            boolean readOnly,
            boolean multipleAccountsPerFolder,
            boolean accountTransferWithQuota,
            boolean managed,
            String key,
            AccountsSettingsModel accountsSettings,
            boolean importAllowed,
            boolean accountsSpacesSupported,
            boolean syncEnabled,
            boolean grpcTlsOn,
            BillingMeta billingMeta,
            Map<String, RelatedResourceMapping> relatedResourcesByResourceId,
            long trackerComponentId,
            String reserveFolderId,
            boolean hasDefaultQuotas,
            Boolean allocatedSupported,
            @Nullable AggregationSettings aggregationSettings,
            @Nullable AggregationAlgorithm aggregationAlgorithm,
            @Nullable ProviderUISettings uiSettings
    ) {
        this.id = id;
        this.tenantId = tenantId;
        this.version = version;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.restApiUri = restApiUri;
        this.grpcApiUri = grpcApiUri;
        this.sourceTvmId = sourceTvmId;
        this.destinationTvmId = destinationTvmId;
        this.serviceId = serviceId;
        this.deleted = deleted;
        this.readOnly = readOnly;
        this.multipleAccountsPerFolder = multipleAccountsPerFolder;
        this.accountTransferWithQuota = accountTransferWithQuota;
        this.managed = managed;
        this.key = key;
        this.accountsSettings = accountsSettings;
        this.importAllowed = importAllowed;
        this.accountsSpacesSupported = accountsSpacesSupported;
        this.syncEnabled = syncEnabled;
        this.grpcTlsOn = grpcTlsOn;
        this.billingMeta = billingMeta;
        this.relatedResourcesByResourceId = relatedResourcesByResourceId != null
                ? ImmutableMap.copyOf(relatedResourcesByResourceId)
                : null;
        this.trackerComponentId = trackerComponentId;
        this.reserveFolderId = reserveFolderId;
        this.hasDefaultQuotas = hasDefaultQuotas;
        this.allocatedSupported = allocatedSupported;
        this.aggregationSettings = aggregationSettings;
        this.aggregationAlgorithm = aggregationAlgorithm;
        this.uiSettings = uiSettings;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ProviderModel provider) {
        return new Builder(provider);
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public long getVersion() {
        return version;
    }

    public String getNameEn() {
        return nameEn;
    }

    public String getNameRu() {
        return nameRu;
    }

    public String getDescriptionEn() {
        return descriptionEn;
    }

    public String getDescriptionRu() {
        return descriptionRu;
    }

    public Optional<String> getRestApiUri() {
        return Optional.ofNullable(restApiUri);
    }

    public Optional<String> getGrpcApiUri() {
        return Optional.ofNullable(grpcApiUri);
    }

    public long getSourceTvmId() {
        return sourceTvmId;
    }

    public long getDestinationTvmId() {
        return destinationTvmId;
    }

    public long getServiceId() {
        return serviceId;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public boolean isMultipleAccountsPerFolder() {
        return multipleAccountsPerFolder;
    }

    public boolean isAccountTransferWithQuota() {
        return accountTransferWithQuota;
    }

    public boolean isManaged() {
        return managed;
    }

    public String getKey() {
        return key;
    }

    public AccountsSettingsModel getAccountsSettings() {
        return accountsSettings;
    }

    public boolean isImportAllowed() {
        return importAllowed;
    }

    public boolean isAccountsSpacesSupported() {
        return accountsSpacesSupported;
    }

    public boolean isSyncEnabled() {
        return syncEnabled;
    }

    public boolean isGrpcTlsOn() {
        return grpcTlsOn;
    }

    public Optional<BillingMeta> getBillingMeta() {
        return Optional.ofNullable(billingMeta);
    }

    public Optional<Map<String, RelatedResourceMapping>> getRelatedResourcesByResourceId() {
        return Optional.ofNullable(relatedResourcesByResourceId);
    }

    public long getTrackerComponentId() {
        return trackerComponentId;
    }

    public Optional<String> getReserveFolderId() {
        return Optional.ofNullable(reserveFolderId);
    }

    public boolean hasDefaultQuotas() {
        return hasDefaultQuotas;
    }

    public Optional<Boolean> isAllocatedSupported() {
        return Optional.ofNullable(allocatedSupported);
    }

    public Optional<AggregationSettings> getAggregationSettings() {
        return Optional.ofNullable(aggregationSettings);
    }

    public Optional<AggregationAlgorithm> getAggregationAlgorithm() {
        return Optional.ofNullable(aggregationAlgorithm);
    }

    public Optional<ProviderUISettings> getUiSettings() {
        return Optional.ofNullable(uiSettings);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProviderModel that = (ProviderModel) o;
        return version == that.version &&
                sourceTvmId == that.sourceTvmId &&
                destinationTvmId == that.destinationTvmId &&
                serviceId == that.serviceId &&
                deleted == that.deleted &&
                readOnly == that.readOnly &&
                multipleAccountsPerFolder == that.multipleAccountsPerFolder &&
                accountTransferWithQuota == that.accountTransferWithQuota &&
                managed == that.managed &&
                importAllowed == that.importAllowed &&
                accountsSpacesSupported == that.accountsSpacesSupported &&
                syncEnabled == that.syncEnabled &&
                grpcTlsOn == that.grpcTlsOn &&
                trackerComponentId == that.trackerComponentId &&
                hasDefaultQuotas == that.hasDefaultQuotas &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(restApiUri, that.restApiUri) &&
                Objects.equals(grpcApiUri, that.grpcApiUri) &&
                Objects.equals(key, that.key) &&
                Objects.equals(accountsSettings, that.accountsSettings) &&
                Objects.equals(billingMeta, that.billingMeta) &&
                Objects.equals(relatedResourcesByResourceId, that.relatedResourcesByResourceId) &&
                Objects.equals(reserveFolderId, that.reserveFolderId) &&
                Objects.equals(allocatedSupported, that.allocatedSupported) &&
                Objects.equals(aggregationSettings, that.aggregationSettings) &&
                Objects.equals(aggregationAlgorithm, that.aggregationAlgorithm) &&
                Objects.equals(uiSettings, that.uiSettings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu, restApiUri,
                grpcApiUri, sourceTvmId, destinationTvmId, serviceId, deleted, readOnly, multipleAccountsPerFolder,
                accountTransferWithQuota, managed, key, accountsSettings, importAllowed, accountsSpacesSupported,
                syncEnabled, grpcTlsOn, billingMeta, relatedResourcesByResourceId, trackerComponentId, reserveFolderId,
                hasDefaultQuotas, allocatedSupported, aggregationSettings, aggregationAlgorithm, uiSettings);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public static final class Builder {

        private String id;
        private TenantId tenantId;
        private Long version;
        private String nameEn;
        private String nameRu;
        private String descriptionEn;
        private String descriptionRu;
        private String restApiUri;
        private String grpcApiUri;
        private Long sourceTvmId;
        private Long destinationTvmId;
        private Long serviceId;
        private Boolean deleted;
        private Boolean readOnly;
        private Boolean multipleAccountsPerFolder;
        private Boolean accountTransferWithQuota;
        private Boolean managed;
        private String key;
        private AccountsSettingsModel accountsSettings;
        private Boolean importAllowed;
        private Boolean accountsSpacesSupported;
        private Boolean syncEnabled;
        private Boolean grpcTlsOn;
        private BillingMeta billingMeta;
        private Map<String, RelatedResourceMapping> relatedResourcesByResourceId;
        private Long trackerComponentId;
        private String reserveFolderId;
        private boolean hasDefaultQuotas = false;
        private Boolean allocatedSupported;
        private @Nullable AggregationSettings aggregationSettings;
        private @Nullable AggregationAlgorithm aggregationAlgorithm;
        private @Nullable ProviderUISettings uiSettings;

        private Builder() {
        }

        private Builder(ProviderModel provider) {
            this.id = provider.getId();
            this.tenantId = provider.getTenantId();
            this.version = provider.getVersion();
            this.nameEn = provider.getNameEn();
            this.nameRu = provider.getNameRu();
            this.descriptionEn = provider.getDescriptionEn();
            this.descriptionRu = provider.getDescriptionRu();
            this.restApiUri = provider.getRestApiUri().orElse(null);
            this.grpcApiUri = provider.getGrpcApiUri().orElse(null);
            this.sourceTvmId = provider.getSourceTvmId();
            this.destinationTvmId = provider.getDestinationTvmId();
            this.serviceId = provider.getServiceId();
            this.deleted = provider.isDeleted();
            this.readOnly = provider.isReadOnly();
            this.multipleAccountsPerFolder = provider.isMultipleAccountsPerFolder();
            this.accountTransferWithQuota = provider.isAccountTransferWithQuota();
            this.managed = provider.isManaged();
            this.key = provider.getKey();
            this.accountsSettings = provider.getAccountsSettings();
            this.importAllowed = provider.isImportAllowed();
            this.accountsSpacesSupported = provider.isAccountsSpacesSupported();
            this.syncEnabled = provider.isSyncEnabled();
            this.grpcTlsOn = provider.isGrpcTlsOn();
            this.billingMeta = provider.getBillingMeta().orElse(null);
            this.relatedResourcesByResourceId = provider.getRelatedResourcesByResourceId().orElse(null);
            this.trackerComponentId = provider.getTrackerComponentId();
            this.reserveFolderId = provider.reserveFolderId;
            this.hasDefaultQuotas = provider.hasDefaultQuotas;
            this.allocatedSupported = provider.allocatedSupported;
            this.aggregationSettings = provider.aggregationSettings;
            this.aggregationAlgorithm = provider.aggregationAlgorithm;
            this.uiSettings = provider.uiSettings;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder version(long version) {
            this.version = version;
            return this;
        }

        public Builder nameEn(String nameEn) {
            this.nameEn = nameEn;
            return this;
        }

        public Builder nameRu(String nameRu) {
            this.nameRu = nameRu;
            return this;
        }

        public Builder descriptionEn(String descriptionEn) {
            this.descriptionEn = descriptionEn;
            return this;
        }

        public Builder descriptionRu(String descriptionRu) {
            this.descriptionRu = descriptionRu;
            return this;
        }

        public Builder restApiUri(String restApiUri) {
            this.restApiUri = restApiUri;
            return this;
        }

        public Builder grpcApiUri(String grpcApiUri) {
            this.grpcApiUri = grpcApiUri;
            return this;
        }

        public Builder sourceTvmId(long sourceTvmId) {
            this.sourceTvmId = sourceTvmId;
            return this;
        }

        public Builder destinationTvmId(long destinationTvmId) {
            this.destinationTvmId = destinationTvmId;
            return this;
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder deleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder readOnly(boolean readOnly) {
            this.readOnly = readOnly;
            return this;
        }

        public Builder multipleAccountsPerFolder(boolean multipleAccountsPerFolder) {
            this.multipleAccountsPerFolder = multipleAccountsPerFolder;
            return this;
        }

        public Builder accountTransferWithQuota(boolean accountTransferWithQuota) {
            this.accountTransferWithQuota = accountTransferWithQuota;
            return this;
        }

        public Builder managed(boolean managed) {
            this.managed = managed;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder accountsSettings(AccountsSettingsModel accountsSettings) {
            this.accountsSettings = accountsSettings;
            return this;
        }

        public Builder importAllowed(boolean importAllowed) {
            this.importAllowed = importAllowed;
            return this;
        }

        public Builder accountsSpacesSupported(boolean accountsSpacesSupported) {
            this.accountsSpacesSupported = accountsSpacesSupported;
            return this;
        }

        public Builder syncEnabled(boolean syncEnabled) {
            this.syncEnabled = syncEnabled;
            return this;
        }

        public Builder grpcTlsOn(boolean grpcTlsOn) {
            this.grpcTlsOn = grpcTlsOn;
            return this;
        }

        public Builder billingMeta(BillingMeta billingMeta) {
            this.billingMeta = billingMeta;
            return this;
        }

        public Builder trackerComponentId(long trackerComponentId) {
            this.trackerComponentId = trackerComponentId;
            return this;
        }

        public Builder relatedResourcesByResourceId(Map<String, RelatedResourceMapping> relatedResourcesByResourceId) {
            this.relatedResourcesByResourceId = relatedResourcesByResourceId;
            return this;
        }

        public Builder addRelatedResource(String id, RelatedResourceMapping relatedResource) {
            if (relatedResourcesByResourceId == null) {
                relatedResourcesByResourceId = new HashMap<>();
            }
            relatedResourcesByResourceId.put(id, relatedResource);
            return this;
        }

        public Builder reserveFolderId(String reserveFolderId) {
            this.reserveFolderId = reserveFolderId;
            return this;
        }

        public Builder hasDefaultQuotas(boolean hasDefaultQuotas) {
            this.hasDefaultQuotas = hasDefaultQuotas;
            return this;
        }

        public Builder allocatedSupported(Boolean allocatedSupported) {
            this.allocatedSupported = allocatedSupported;
            return this;
        }

        public Builder aggregationSettings(@Nullable AggregationSettings aggregationSettings) {
            this.aggregationSettings = aggregationSettings;
            return this;
        }

        public Builder aggregationAlgorithm(@Nullable AggregationAlgorithm aggregationAlgorithm) {
            this.aggregationAlgorithm = aggregationAlgorithm;
            return this;
        }

        public Builder uiSettings(@Nullable ProviderUISettings uiSettings) {
            this.uiSettings = uiSettings;
            return this;
        }

        public Optional<String> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<TenantId> getTenantId() {
            return Optional.ofNullable(tenantId);
        }

        public Optional<Long> getVersion() {
            return Optional.ofNullable(version);
        }

        public Optional<String> getNameEn() {
            return Optional.ofNullable(nameEn);
        }

        public Optional<String> getNameRu() {
            return Optional.ofNullable(nameRu);
        }

        public Optional<String> getDescriptionEn() {
            return Optional.ofNullable(descriptionEn);
        }

        public Optional<String> getDescriptionRu() {
            return Optional.ofNullable(descriptionRu);
        }

        public Optional<String> getRestApiUri() {
            return Optional.ofNullable(restApiUri);
        }

        public Optional<String> getGrpcApiUri() {
            return Optional.ofNullable(grpcApiUri);
        }

        public Optional<Long> getSourceTvmId() {
            return Optional.ofNullable(sourceTvmId);
        }

        public Optional<Long> getDestinationTvmId() {
            return Optional.ofNullable(destinationTvmId);
        }

        public Optional<Long> getServiceId() {
            return Optional.ofNullable(serviceId);
        }

        public Optional<Boolean> getDeleted() {
            return Optional.ofNullable(deleted);
        }

        public Optional<Boolean> getReadOnly() {
            return Optional.ofNullable(readOnly);
        }

        public Optional<Boolean> getMultipleAccountsPerFolder() {
            return Optional.ofNullable(multipleAccountsPerFolder);
        }

        public Optional<Boolean> getAccountTransferWithQuota() {
            return Optional.ofNullable(accountTransferWithQuota);
        }

        public Optional<Boolean> getManaged() {
            return Optional.ofNullable(managed);
        }

        public Optional<String> getKey() {
            return Optional.ofNullable(key);
        }

        public Optional<AccountsSettingsModel> getAccountsSettings() {
            return Optional.ofNullable(accountsSettings);
        }

        public Optional<Boolean> getImportAllowed() {
            return Optional.ofNullable(importAllowed);
        }

        public Optional<Boolean> getAccountsSpacesSupported() {
            return Optional.ofNullable(accountsSpacesSupported);
        }

        public Optional<Boolean> getSyncEnabled() {
            return Optional.ofNullable(syncEnabled);
        }

        public Optional<Boolean> getGrpcTlsOn() {
            return Optional.ofNullable(grpcTlsOn);
        }

        public Optional<BillingMeta> getBillingMeta() {
            return Optional.ofNullable(billingMeta);
        }

        public Optional<String> getReserveFolderId() {
            return Optional.ofNullable(reserveFolderId);
        }

        public Optional<Map<String, RelatedResourceMapping>> getRelatedResourcesByResourceId() {
            return Optional.ofNullable(relatedResourcesByResourceId);
        }

        public Optional<Long> getTrackerComponentId() {
            return Optional.ofNullable(trackerComponentId);
        }

        public Optional<Boolean> isAllocatedSupported() {
            return Optional.ofNullable(allocatedSupported);
        }

        public Optional<AggregationSettings> getAggregationSettings() {
            return Optional.ofNullable(aggregationSettings);
        }

        public Optional<AggregationAlgorithm> getAggregationAlgorithm() {
            return Optional.ofNullable(aggregationAlgorithm);
        }

        public boolean hasChanges(ProviderModel provider) {
            if (!Objects.equals(id, provider.getId())) {
                return true;
            }
            if (!Objects.equals(tenantId, provider.getTenantId())) {
                return true;
            }
            if (!Objects.equals(nameEn, provider.getNameEn())) {
                return true;
            }
            if (!Objects.equals(nameRu, provider.getNameRu())) {
                return true;
            }
            if (!Objects.equals(descriptionEn, provider.getDescriptionEn())) {
                return true;
            }
            if (!Objects.equals(descriptionRu, provider.getDescriptionRu())) {
                return true;
            }
            if (!Objects.equals(restApiUri, provider.getRestApiUri().orElse(null))) {
                return true;
            }
            if (!Objects.equals(grpcApiUri, provider.getGrpcApiUri().orElse(null))) {
                return true;
            }
            if (!Objects.equals(sourceTvmId, provider.getSourceTvmId())) {
                return true;
            }
            if (!Objects.equals(destinationTvmId, provider.getDestinationTvmId())) {
                return true;
            }
            if (!Objects.equals(serviceId, provider.getServiceId())) {
                return true;
            }
            if (!Objects.equals(deleted, provider.isDeleted())) {
                return true;
            }
            if (!Objects.equals(readOnly, provider.isReadOnly())) {
                return true;
            }
            if (!Objects.equals(multipleAccountsPerFolder, provider.isMultipleAccountsPerFolder())) {
                return true;
            }
            if (!Objects.equals(accountTransferWithQuota, provider.isAccountTransferWithQuota())) {
                return true;
            }
            if (!Objects.equals(managed, provider.isManaged())) {
                return true;
            }
            if (!Objects.equals(key, provider.getKey())) {
                return true;
            }
            if (!Objects.equals(accountsSettings, provider.getAccountsSettings())) {
                return true;
            }
            if (!Objects.equals(importAllowed, provider.isImportAllowed())) {
                return true;
            }
            if (!Objects.equals(accountsSpacesSupported, provider.isAccountsSpacesSupported())) {
                return true;
            }
            if (!Objects.equals(syncEnabled, provider.isSyncEnabled())) {
                return true;
            }
            if (!Objects.equals(grpcTlsOn, provider.isGrpcTlsOn())) {
                return true;
            }
            if (!Objects.equals(billingMeta, provider.getBillingMeta().orElse(null))) {
                return true;
            }
            if (!Objects.equals(relatedResourcesByResourceId,
                    provider.getRelatedResourcesByResourceId().orElse(null))) {
                return true;
            }
            if (!Objects.equals(trackerComponentId, provider.getTrackerComponentId())) {
                return true;
            }
            if (!Objects.equals(reserveFolderId, provider.getReserveFolderId().orElse(null))) {
                return true;
            }
            if (hasDefaultQuotas != provider.hasDefaultQuotas) {
                return true;
            }
            if (!Objects.equals(allocatedSupported, provider.allocatedSupported)) {
                return true;
            }
            if (!Objects.equals(aggregationSettings, provider.aggregationSettings)) {
                return true;
            }
            if (!Objects.equals(aggregationAlgorithm, provider.aggregationAlgorithm)) {
                return true;
            }
            if (!Objects.equals(uiSettings, provider.uiSettings)) {
                return true;
            }
            return false;
        }

        public ProviderModel build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(tenantId, "Tenant id is required");
            Preconditions.checkNotNull(version, "Version is required");
            Preconditions.checkNotNull(nameEn, "NameEn is required");
            Preconditions.checkNotNull(nameRu, "NameRu is required");
            Preconditions.checkNotNull(descriptionEn, "DescriptionEn is required");
            Preconditions.checkNotNull(descriptionRu, "DescriptionRu is required");
            Preconditions.checkNotNull(sourceTvmId, "SourceTvmId is required");
            Preconditions.checkNotNull(destinationTvmId, "DestinationTvmId is required");
            Preconditions.checkNotNull(serviceId, "ServiceId is required");
            Preconditions.checkNotNull(deleted, "Deleted is required");
            Preconditions.checkNotNull(readOnly, "ReadOnly is required");
            Preconditions.checkNotNull(multipleAccountsPerFolder, "MultipleAccountsPerFolder is required");
            Preconditions.checkNotNull(accountTransferWithQuota, "AccountTransferWithQuota is required");
            Preconditions.checkNotNull(managed, "Managed is required");
            Preconditions.checkNotNull(key, "Key is required");
            Preconditions.checkNotNull(accountsSettings, "AccountsSettings is required");
            Preconditions.checkNotNull(importAllowed, "ImportAllowed is required");
            Preconditions.checkNotNull(accountsSpacesSupported, "AccountsSpacesSupported is required");
            Preconditions.checkNotNull(syncEnabled, "SyncEnabled is required");
            Preconditions.checkNotNull(grpcTlsOn, "GrpcTlsOn is required");
            Preconditions.checkNotNull(trackerComponentId, "TrackerComponentId is required");
            return new ProviderModel(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu,
                    restApiUri, grpcApiUri, sourceTvmId, destinationTvmId, serviceId, deleted, readOnly,
                    multipleAccountsPerFolder, accountTransferWithQuota, managed, key, accountsSettings,
                    importAllowed, accountsSpacesSupported, syncEnabled, grpcTlsOn, billingMeta,
                    relatedResourcesByResourceId, trackerComponentId, reserveFolderId, hasDefaultQuotas,
                    allocatedSupported, aggregationSettings, aggregationAlgorithm, uiSettings);
        }

    }

}
