package ru.yandex.intranet.d.model.quotas;

import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Quota in default folder
 *
 * @author Nikita Minin <spasitel@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class QuotaModel {
    private final TenantId tenantId;
    private final String folderId;
    private final String providerId;
    private final String resourceId;
    private final long quota;
    private final long balance;
    private final long frozenQuota;

    @JsonCreator
    public QuotaModel(TenantId tenantId,
                      String folderId,
                      String providerId,
                      String resourceId,
                      long quota,
                      long balance,
                      long frozenQuota) {
        this.tenantId = tenantId;
        this.folderId = folderId;
        this.providerId = providerId;
        this.resourceId = resourceId;
        this.quota = quota;
        this.balance = balance;
        this.frozenQuota = frozenQuota;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(QuotaModel value) {
        return new Builder(value);
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getFolderId() {
        return folderId;
    }

    public String getResourceId() {
        return resourceId;
    }

    public Long getQuota() {
        return quota;
    }

    public Long getBalance() {
        return balance;
    }

    public long getFrozenQuota() {
        return frozenQuota;
    }

    public String getProviderId() {
        return providerId;
    }

    public Key toKey() {
        return new Key(folderId, providerId, resourceId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        QuotaModel that = (QuotaModel) o;
        return quota == that.quota &&
                balance == that.balance &&
                frozenQuota == that.frozenQuota &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(resourceId, that.resourceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, folderId, providerId, resourceId, quota, balance, frozenQuota);
    }

    @Override
    public String toString() {
        return "QuotaModel{" +
                "tenantId=" + tenantId +
                ", folderId='" + folderId + '\'' +
                ", providerId='" + providerId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", quota=" + quota +
                ", balance=" + balance +
                ", frozenQuota=" + frozenQuota +
                '}';
    }

    public static final class Key {

        private final String folderId;
        private final String providerId;
        private final String resourceId;

        public Key(String folderId, String providerId, String resourceId) {
            this.folderId = folderId;
            this.providerId = providerId;
            this.resourceId = resourceId;
        }

        public String getFolderId() {
            return folderId;
        }

        public String getProviderId() {
            return providerId;
        }

        public String getResourceId() {
            return resourceId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return Objects.equals(folderId, key.folderId) &&
                    Objects.equals(providerId, key.providerId) &&
                    Objects.equals(resourceId, key.resourceId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(folderId, providerId, resourceId);
        }

        @Override
        public String toString() {
            return "Key{" +
                    "folderId='" + folderId + '\'' +
                    ", providerId='" + providerId + '\'' +
                    ", resourceId='" + resourceId + '\'' +
                    '}';
        }

    }

    public static final class Builder {

        private TenantId tenantId;
        private String folderId;
        private String providerId;
        private String resourceId;
        private Long quota;
        private Long balance;
        private Long frozenQuota;

        private Builder() {
        }

        private Builder(QuotaModel value) {
            this.tenantId = value.tenantId;
            this.folderId = value.folderId;
            this.providerId = value.providerId;
            this.resourceId = value.resourceId;
            this.quota = value.quota;
            this.balance = value.balance;
            this.frozenQuota = value.frozenQuota;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder quota(long quota) {
            this.quota = quota;
            return this;
        }

        public Builder balance(long balance) {
            this.balance = balance;
            return this;
        }

        public Builder frozenQuota(Long frozenQuota) {
            this.frozenQuota = frozenQuota;
            return this;
        }

        public QuotaModel build() {
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(folderId, "FolderId is required");
            Preconditions.checkNotNull(providerId, "ProviderId is required");
            Preconditions.checkNotNull(resourceId, "ResourceId is required");
            Preconditions.checkNotNull(quota, "Quota is required");
            Preconditions.checkNotNull(balance, "Balance is required");
            Preconditions.checkNotNull(frozenQuota, "FrozenQuota is required");
            return new QuotaModel(tenantId, folderId, providerId, resourceId, quota, balance, frozenQuota);
        }

    }

}
