package ru.yandex.intranet.d.model.resources;

import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.AggregationSettings;

/**
 * Resource.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceModel {

    private final String id;
    private final TenantId tenantId;
    private final long version;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final boolean deleted;
    private final String unitsEnsembleId;
    private final String providerId;
    private final String resourceTypeId;
    private final Set<ResourceSegmentSettingsModel> segments;
    private final ResourceUnitsModel resourceUnits;
    private final boolean managed;
    private final boolean orderable;
    private final String key;
    private final boolean readOnly;
    private final String baseUnitId;
    private final String accountsSpacesId;
    private final Long defaultQuota;
    private final boolean virtual;
    // Null means "same as provider flag"
    private final Boolean allocatedSupported;
    // Null means unspecified, use appropriate default value
    private final @Nullable AggregationSettings aggregationSettings;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceModel(
            String id,
            TenantId tenantId,
            long version,
            String nameEn,
            String nameRu,
            String descriptionEn,
            String descriptionRu,
            boolean deleted,
            String unitsEnsembleId,
            String providerId,
            String resourceTypeId,
            Set<ResourceSegmentSettingsModel> segments,
            ResourceUnitsModel resourceUnits,
            boolean managed,
            boolean orderable,
            String key,
            boolean readOnly,
            String baseUnitId,
            String accountsSpacesId,
            Long defaultQuota,
            boolean virtual,
            Boolean allocatedSupported,
            @Nullable AggregationSettings aggregationSettings
    ) {
        this.id = id;
        this.tenantId = tenantId;
        this.version = version;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.deleted = deleted;
        this.unitsEnsembleId = unitsEnsembleId;
        this.providerId = providerId;
        this.resourceTypeId = resourceTypeId;
        this.segments = segments;
        this.resourceUnits = resourceUnits;
        this.managed = managed;
        this.orderable = orderable;
        this.key = key;
        this.readOnly = readOnly;
        this.baseUnitId = baseUnitId;
        this.accountsSpacesId = accountsSpacesId;
        this.defaultQuota = defaultQuota;
        this.virtual = virtual;
        this.allocatedSupported = allocatedSupported;
        this.aggregationSettings = aggregationSettings;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceModel source) {
        return new Builder(source);
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public long getVersion() {
        return version;
    }

    public String getNameEn() {
        return nameEn;
    }

    public String getNameRu() {
        return nameRu;
    }

    public String getDescriptionEn() {
        return descriptionEn;
    }

    public String getDescriptionRu() {
        return descriptionRu;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public String getUnitsEnsembleId() {
        return unitsEnsembleId;
    }

    public String getProviderId() {
        return providerId;
    }

    public String getResourceTypeId() {
        return resourceTypeId;
    }

    public Set<ResourceSegmentSettingsModel> getSegments() {
        return segments != null ? segments : Set.of();
    }

    public ResourceUnitsModel getResourceUnits() {
        return resourceUnits;
    }

    public boolean isManaged() {
        return managed;
    }

    public boolean isOrderable() {
        return orderable;
    }

    public String getKey() {
        return key;
    }

    public boolean isReadOnly() {
        return readOnly;
    }

    public String getBaseUnitId() {
        return baseUnitId;
    }

    @Nullable
    public String getAccountsSpacesId() {
        return accountsSpacesId;
    }

    public Optional<Long> getDefaultQuota() {
        return Optional.ofNullable(defaultQuota);
    }

    public boolean isVirtual() {
        return virtual;
    }

    public Optional<Boolean> isAllocatedSupported() {
        return Optional.ofNullable(allocatedSupported);
    }

    public Optional<AggregationSettings> getAggregationSettings() {
        return Optional.ofNullable(aggregationSettings);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceModel that = (ResourceModel) o;
        return version == that.version &&
                deleted == that.deleted &&
                managed == that.managed &&
                orderable == that.orderable &&
                readOnly == that.readOnly &&
                virtual == that.virtual &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(unitsEnsembleId, that.unitsEnsembleId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(resourceTypeId, that.resourceTypeId) &&
                Objects.equals(segments, that.segments) &&
                Objects.equals(resourceUnits, that.resourceUnits) &&
                Objects.equals(key, that.key) &&
                Objects.equals(baseUnitId, that.baseUnitId) &&
                Objects.equals(accountsSpacesId, that.accountsSpacesId) &&
                Objects.equals(defaultQuota, that.defaultQuota) &&
                Objects.equals(allocatedSupported, that.allocatedSupported) &&
                Objects.equals(aggregationSettings, that.aggregationSettings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu, deleted,
                unitsEnsembleId, providerId, resourceTypeId, segments, resourceUnits, managed, orderable, key,
                readOnly, baseUnitId, accountsSpacesId, defaultQuota, virtual, allocatedSupported, aggregationSettings);
    }

    @Override
    public String toString() {
        return "ResourceModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", version=" + version +
                ", nameEn='" + nameEn + '\'' +
                ", nameRu='" + nameRu + '\'' +
                ", descriptionEn='" + descriptionEn + '\'' +
                ", descriptionRu='" + descriptionRu + '\'' +
                ", deleted=" + deleted +
                ", unitsEnsembleId='" + unitsEnsembleId + '\'' +
                ", providerId='" + providerId + '\'' +
                ", resourceTypeId='" + resourceTypeId + '\'' +
                ", segments=" + segments +
                ", resourceUnits=" + resourceUnits +
                ", managed=" + managed +
                ", orderable=" + orderable +
                ", key='" + key + '\'' +
                ", readOnly=" + readOnly +
                ", baseUnitId='" + baseUnitId + '\'' +
                ", accountsSpacesId='" + accountsSpacesId + '\'' +
                ", defaultQuota=" + defaultQuota +
                ", virtual=" + virtual +
                ", allocatedSupported=" + allocatedSupported +
                ", aggregationSettings=" + aggregationSettings +
                '}';
    }

    public static final class Builder {

        private String id;
        private TenantId tenantId;
        private Long version;
        private String nameEn;
        private String nameRu;
        private String descriptionEn;
        private String descriptionRu;
        private Boolean deleted;
        private String unitsEnsembleId;
        private String providerId;
        private String resourceTypeId;
        private Set<ResourceSegmentSettingsModel> segments;
        private ResourceUnitsModel resourceUnits;
        private Boolean managed;
        private Boolean orderable;
        private String key;
        private Boolean readOnly;
        private String baseUnitId;
        private String accountsSpacesId;
        private Long defaultQuota;
        private Boolean virtual = Boolean.FALSE;
        private Boolean allocatedSupported;
        private @Nullable AggregationSettings aggregationSettings;

        private Builder() {
        }

        private Builder(ResourceModel source) {
            this.id = source.getId();
            this.tenantId = source.getTenantId();
            this.version = source.getVersion();
            this.nameEn = source.getNameEn();
            this.nameRu = source.getNameRu();
            this.descriptionEn = source.getDescriptionEn();
            this.descriptionRu = source.getDescriptionRu();
            this.deleted = source.isDeleted();
            this.unitsEnsembleId = source.getUnitsEnsembleId();
            this.providerId = source.getProviderId();
            this.resourceTypeId = source.resourceTypeId;
            this.segments = source.segments;
            this.resourceUnits = source.getResourceUnits();
            this.managed = source.isManaged();
            this.orderable = source.isOrderable();
            this.key = source.getKey();
            this.readOnly = source.isReadOnly();
            this.baseUnitId = source.getBaseUnitId();
            this.accountsSpacesId = source.accountsSpacesId;
            this.defaultQuota = source.defaultQuota;
            this.virtual = source.virtual;
            this.allocatedSupported = source.allocatedSupported;
            this.aggregationSettings = source.aggregationSettings;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder version(long version) {
            this.version = version;
            return this;
        }

        public Builder nameEn(String nameEn) {
            this.nameEn = nameEn;
            return this;
        }

        public Builder nameRu(String nameRu) {
            this.nameRu = nameRu;
            return this;
        }

        public Builder descriptionEn(String descriptionEn) {
            this.descriptionEn = descriptionEn;
            return this;
        }

        public Builder descriptionRu(String descriptionRu) {
            this.descriptionRu = descriptionRu;
            return this;
        }

        public Builder deleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder unitsEnsembleId(String unitsEnsembleId) {
            this.unitsEnsembleId = unitsEnsembleId;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder resourceTypeId(String resourceTypeId) {
            this.resourceTypeId = resourceTypeId;
            return this;
        }

        public Builder segments(Set<ResourceSegmentSettingsModel> segments) {
            this.segments = segments;
            return this;
        }

        public Builder resourceUnits(ResourceUnitsModel resourceUnits) {
            this.resourceUnits = resourceUnits;
            return this;
        }

        public Builder managed(boolean managed) {
            this.managed = managed;
            return this;
        }

        public Builder orderable(boolean orderable) {
            this.orderable = orderable;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder readOnly(boolean readOnly) {
            this.readOnly = readOnly;
            return this;
        }

        public Builder baseUnitId(String baseUnitId) {
            this.baseUnitId = baseUnitId;
            return this;
        }

        public Builder accountsSpacesId(String accountsSpacesId) {
            this.accountsSpacesId = accountsSpacesId;
            return this;
        }

        public Builder defaultQuota(Long defaultQuota) {
            this.defaultQuota = defaultQuota;
            return this;
        }

        public Builder virtual(Boolean virtual) {
            this.virtual = virtual;
            return this;
        }

        public Builder allocatedSupported(Boolean allocatedSupported) {
            this.allocatedSupported = allocatedSupported;
            return this;
        }

        public Builder aggregationSettings(@Nullable AggregationSettings aggregationSettings) {
            this.aggregationSettings = aggregationSettings;
            return this;
        }

        public Optional<String> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<TenantId> getTenantId() {
            return Optional.ofNullable(tenantId);
        }

        public Optional<Long> getVersion() {
            return Optional.ofNullable(version);
        }

        public Optional<String> getNameEn() {
            return Optional.ofNullable(nameEn);
        }

        public Optional<String> getNameRu() {
            return Optional.ofNullable(nameRu);
        }

        public Optional<String> getDescriptionEn() {
            return Optional.ofNullable(descriptionEn);
        }

        public Optional<String> getDescriptionRu() {
            return Optional.ofNullable(descriptionRu);
        }

        public Optional<Boolean> getDeleted() {
            return Optional.ofNullable(deleted);
        }

        public Optional<String> getUnitsEnsembleId() {
            return Optional.ofNullable(unitsEnsembleId);
        }

        public Optional<String> getProviderId() {
            return Optional.ofNullable(providerId);
        }

        public Optional<ResourceUnitsModel> getResourceUnits() {
            return Optional.ofNullable(resourceUnits);
        }

        public Optional<Boolean> getManaged() {
            return Optional.ofNullable(managed);
        }

        public Optional<Boolean> getOrderable() {
            return Optional.ofNullable(orderable);
        }

        public Optional<String> getKey() {
            return Optional.ofNullable(key);
        }

        public Optional<Boolean> getReadOnly() {
            return Optional.ofNullable(readOnly);
        }

        public Optional<String> getBaseUnitId() {
            return Optional.ofNullable(baseUnitId);
        }

        public Optional<Long> getDefaultQuota() {
            return Optional.ofNullable(defaultQuota);
        }

        public Optional<Boolean> getVirtual() {
            return Optional.ofNullable(virtual);
        }

        public Optional<Boolean> isAllocatedSupported() {
            return Optional.ofNullable(allocatedSupported);
        }

        public Optional<AggregationSettings> getAggregationSettings() {
            return Optional.ofNullable(aggregationSettings);
        }

        public boolean hasChanges(ResourceModel value) {
            if (!Objects.equals(id, value.getId())) {
                return true;
            }
            if (!Objects.equals(tenantId, value.getTenantId())) {
                return true;
            }
            if (!Objects.equals(nameEn, value.getNameEn())) {
                return true;
            }
            if (!Objects.equals(nameRu, value.getNameRu())) {
                return true;
            }
            if (!Objects.equals(descriptionEn, value.getDescriptionEn())) {
                return true;
            }
            if (!Objects.equals(descriptionRu, value.getDescriptionRu())) {
                return true;
            }
            if (!Objects.equals(deleted, value.isDeleted())) {
                return true;
            }
            if (!Objects.equals(unitsEnsembleId, value.getUnitsEnsembleId())) {
                return true;
            }
            if (!Objects.equals(providerId, value.getProviderId())) {
                return true;
            }
            if (!Objects.equals(resourceTypeId, value.getResourceTypeId())) {
                return true;
            }
            if (!Objects.equals(segments, value.getSegments())) {
                return true;
            }
            if (!Objects.equals(resourceUnits, value.getResourceUnits())) {
                return true;
            }
            if (!Objects.equals(managed, value.isManaged())) {
                return true;
            }
            if (!Objects.equals(orderable, value.isOrderable())) {
                return true;
            }
            if (!Objects.equals(key, value.getKey())) {
                return true;
            }
            if (!Objects.equals(readOnly, value.isReadOnly())) {
                return true;
            }
            if (!Objects.equals(baseUnitId, value.getBaseUnitId())) {
                return true;
            }
            if (!Objects.equals(accountsSpacesId, value.getAccountsSpacesId())) {
                return true;
            }
            if (!Objects.equals(defaultQuota, value.defaultQuota)) {
                return true;
            }
            if (!Objects.equals(virtual, value.virtual)) {
                return true;
            }
            if (!Objects.equals(allocatedSupported, value.allocatedSupported)) {
                return true;
            }
            if (!Objects.equals(aggregationSettings, value.aggregationSettings)) {
                return true;
            }
            return false;
        }

        public ResourceModel build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(version, "Version is required");
            Preconditions.checkNotNull(nameEn, "NameEn is required");
            Preconditions.checkNotNull(nameRu, "NameRu is required");
            Preconditions.checkNotNull(descriptionEn, "DescriptionEn is required");
            Preconditions.checkNotNull(descriptionRu, "DescriptionRu is required");
            Preconditions.checkNotNull(deleted, "Deleted is required");
            Preconditions.checkNotNull(unitsEnsembleId, "UnitsEnsembleId is required");
            Preconditions.checkNotNull(providerId, "ProviderId is required");
            Preconditions.checkNotNull(resourceUnits, "ResourceUnits is required");
            Preconditions.checkNotNull(managed, "Managed is required");
            Preconditions.checkNotNull(orderable, "Orderable is required");
            Preconditions.checkNotNull(key, "Key is required");
            Preconditions.checkNotNull(readOnly, "ReadOnly is required");
            Preconditions.checkNotNull(baseUnitId, "BaseUnitId is required");
            return new ResourceModel(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu, deleted,
                    unitsEnsembleId, providerId, resourceTypeId, segments, resourceUnits, managed, orderable, key,
                    readOnly, baseUnitId, accountsSpacesId, defaultQuota, virtual, allocatedSupported,
                    aggregationSettings);
        }

    }

}
