package ru.yandex.intranet.d.model.resources;

import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;

/**
 * Resource segment settings.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceSegmentSettingsModel {

    private final String segmentationId;
    private final String segmentId;

    @JsonCreator
    public ResourceSegmentSettingsModel(
            String segmentationId,
            String segmentId
    ) {
        this.segmentationId = segmentationId;
        this.segmentId = segmentId;
    }

    public static String toString(Set<ResourceSegmentSettingsModel> segments) {
        return segments.stream()
                .map(s -> "\"" + s.segmentationId + "\":\"" + s.segmentId + "\"")
                .collect(Collectors.joining(",", "{", "}"));
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceSegmentSettingsModel source) {
        return new Builder(source);
    }

    public String getSegmentationId() {
        return segmentationId;
    }

    public String getSegmentId() {
        return segmentId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceSegmentSettingsModel that = (ResourceSegmentSettingsModel) o;
        return Objects.equals(segmentationId, that.segmentationId) &&
                Objects.equals(segmentId, that.segmentId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(segmentationId, segmentId);
    }

    @Override
    public String toString() {
        return "ResourceSegmentSettingsModel{" +
                "segmentationId='" + segmentationId + '\'' +
                ", segmentId='" + segmentId + '\'' +
                '}';
    }

    public static final class Builder {

        private String segmentationId;
        private String segmentId;

        private Builder() {
        }

        private Builder(ResourceSegmentSettingsModel source) {
            this.segmentationId = source.getSegmentationId();
            this.segmentId = source.getSegmentId();
        }

        public Builder segmentationId(String segmentationId) {
            this.segmentationId = segmentationId;
            return this;
        }

        public Builder segmentId(String segmentId) {
            this.segmentId = segmentId;
            return this;
        }

        public Optional<String> getSegmentationId() {
            return Optional.ofNullable(segmentationId);
        }

        public Optional<String> getSegmentId() {
            return Optional.ofNullable(segmentId);
        }

        public boolean hasChanges(ResourceSegmentSettingsModel value) {
            if (!Objects.equals(segmentationId, value.getSegmentationId())) {
                return true;
            }
            if (!Objects.equals(segmentId, value.getSegmentId())) {
                return true;
            }
            return false;
        }

        public ResourceSegmentSettingsModel build() {
            Preconditions.checkNotNull(segmentationId, "SegmentationId is required");
            Preconditions.checkNotNull(segmentId, "SegmentId is required");
            return new ResourceSegmentSettingsModel(segmentationId, segmentId);
        }

    }

}
