package ru.yandex.intranet.d.model.resources;

import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.google.common.base.Preconditions;

/**
 * Resource units.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceUnitsModel {

    private final Set<String> allowedUnitIds;
    private final String defaultUnitId;
    private final String providerApiUnitId;

    @JsonCreator
    public ResourceUnitsModel(
            Set<String> allowedUnitIds,
            String defaultUnitId,
            String providerApiUnitId) {
        this.allowedUnitIds = allowedUnitIds;
        this.defaultUnitId = defaultUnitId;
        this.providerApiUnitId = providerApiUnitId;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceUnitsModel source) {
        return new Builder(source);
    }

    public String getProviderApiUnitId() {
        return providerApiUnitId;
    }

    public Set<String> getAllowedUnitIds() {
        return allowedUnitIds;
    }

    public String getDefaultUnitId() {
        return defaultUnitId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceUnitsModel that = (ResourceUnitsModel) o;
        return Objects.equals(allowedUnitIds, that.allowedUnitIds) &&
                Objects.equals(defaultUnitId, that.defaultUnitId) &&
                Objects.equals(providerApiUnitId, that.providerApiUnitId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(allowedUnitIds, defaultUnitId, providerApiUnitId);
    }

    @Override
    public String toString() {
        return "ResourceUnitsModel{" +
                "allowedUnitIds=" + allowedUnitIds +
                ", defaultUnitId='" + defaultUnitId + '\'' +
                ", providerApiUnitId='" + providerApiUnitId + '\'' +
                '}';
    }

    public static final class Builder {

        private final Set<String> allowedUnitIds = new HashSet<>();

        private String defaultUnitId;

        private String providerApiUnitId;

        private Builder() {
        }

        private Builder(ResourceUnitsModel source) {
            allowedUnitIds.addAll(source.getAllowedUnitIds());
            defaultUnitId = source.getDefaultUnitId();
        }

        public Builder addAllowedUnitId(String allowedUnitId) {
            this.allowedUnitIds.add(allowedUnitId);
            return this;
        }

        public Builder addAllowedUnitIds(Set<String> allowedUnitIds) {
            this.allowedUnitIds.addAll(allowedUnitIds);
            return this;
        }

        public Builder defaultUnitId(String defaultUnitId) {
            this.defaultUnitId = defaultUnitId;
            return this;
        }

        public Builder providerApiUnitId(String providerApiUnitId) {
            this.providerApiUnitId = providerApiUnitId;
            return this;
        }

        public Set<String> getAllowedUnitIds() {
            return allowedUnitIds;
        }

        public Optional<String> getDefaultUnitId() {
            return Optional.ofNullable(defaultUnitId);
        }

        public Optional<String> getProviderApiUnitId() {
            return Optional.ofNullable(providerApiUnitId);
        }

        public boolean hasChanges(ResourceUnitsModel value) {
            if (!Objects.equals(allowedUnitIds, value.getAllowedUnitIds())) {
                return true;
            }
            if (!Objects.equals(defaultUnitId, value.getDefaultUnitId())) {
                return true;
            }
            if (!Objects.equals(providerApiUnitId, value.getProviderApiUnitId())) {
                return true;
            }
            return false;
        }

        public ResourceUnitsModel build() {
            Preconditions.checkNotNull(defaultUnitId, "DefaultUnitId is required");
            return new ResourceUnitsModel(allowedUnitIds, defaultUnitId, providerApiUnitId);
        }

    }

}
