package ru.yandex.intranet.d.model.resources.segmentations;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import io.swagger.v3.oas.annotations.media.Schema;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Resource segmentation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceSegmentationModel {

    private final String id;
    private final TenantId tenantId;
    private final String providerId;
    private final long version;
    private final String key;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final boolean deleted;
    private final int groupingOrder;
    private @Nullable final SegmentationUISettings uiSettings;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceSegmentationModel(
            String id,
            TenantId tenantId,
            String providerId,
            long version,
            String key,
            String nameEn,
            String nameRu,
            String descriptionEn,
            String descriptionRu,
            boolean deleted,
            int groupingOrder,
            @Nullable SegmentationUISettings uiSettings
    ) {
        this.id = id;
        this.tenantId = tenantId;
        this.providerId = providerId;
        this.version = version;
        this.key = key;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.deleted = deleted;
        this.groupingOrder = groupingOrder;
        this.uiSettings = uiSettings;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceSegmentationModel resourceSegmentation) {
        return new Builder(resourceSegmentation);
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getProviderId() {
        return providerId;
    }

    public long getVersion() {
        return version;
    }

    public String getKey() {
        return key;
    }

    public String getNameEn() {
        return nameEn;
    }

    public String getNameRu() {
        return nameRu;
    }

    public String getDescriptionEn() {
        return descriptionEn;
    }

    public String getDescriptionRu() {
        return descriptionRu;
    }

    public boolean isDeleted() {
        return deleted;
    }

    @Schema(description = "Order position for grouping and collecting quotas sums.")
    public int getGroupingOrder() {
        return groupingOrder;
    }

    public Optional<SegmentationUISettings> getUiSettings() {
        return Optional.ofNullable(uiSettings);
    }

    @JsonIgnore
    public ProviderKey getProviderKey() {
        return new ProviderKey(providerId, key);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceSegmentationModel that = (ResourceSegmentationModel) o;
        return version == that.version &&
                deleted == that.deleted &&
                groupingOrder == that.groupingOrder &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(key, that.key) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(uiSettings, that.uiSettings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, providerId, version, key, nameEn, nameRu, descriptionEn, descriptionRu,
                deleted, groupingOrder, uiSettings);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public static final class Builder {

        private String id;
        private TenantId tenantId;
        private String providerId;
        private Long version;
        private String key;
        private String nameEn;
        private String nameRu;
        private String descriptionEn;
        private String descriptionRu;
        private Boolean deleted;
        private Integer groupingOrder = 0;
        private SegmentationUISettings uiSettings = null;

        private Builder() {
        }

        private Builder(ResourceSegmentationModel resourceSegmentation) {
            this.id = resourceSegmentation.getId();
            this.tenantId = resourceSegmentation.getTenantId();
            this.providerId = resourceSegmentation.getProviderId();
            this.version = resourceSegmentation.getVersion();
            this.key = resourceSegmentation.getKey();
            this.nameEn = resourceSegmentation.getNameEn();
            this.nameRu = resourceSegmentation.getNameRu();
            this.descriptionEn = resourceSegmentation.getDescriptionEn();
            this.descriptionRu = resourceSegmentation.getDescriptionRu();
            this.deleted = resourceSegmentation.isDeleted();
            this.groupingOrder = resourceSegmentation.groupingOrder;
            this.uiSettings = resourceSegmentation.uiSettings;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder version(Long version) {
            this.version = version;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder nameEn(String nameEn) {
            this.nameEn = nameEn;
            return this;
        }

        public Builder nameRu(String nameRu) {
            this.nameRu = nameRu;
            return this;
        }

        public Builder descriptionEn(String descriptionEn) {
            this.descriptionEn = descriptionEn;
            return this;
        }

        public Builder descriptionRu(String descriptionRu) {
            this.descriptionRu = descriptionRu;
            return this;
        }

        public Builder deleted(Boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder groupingOrder(Integer groupingOrder) {
            this.groupingOrder = groupingOrder;
            return this;
        }

        public Builder uiSettings(SegmentationUISettings uiSettings) {
            this.uiSettings = uiSettings;
            return this;
        }

        public boolean hasChanges(ResourceSegmentationModel resourceSegmentation) {
            if (!Objects.equals(id, resourceSegmentation.getId())) {
                return true;
            }
            if (!Objects.equals(tenantId, resourceSegmentation.getTenantId())) {
                return true;
            }
            if (!Objects.equals(providerId, resourceSegmentation.getProviderId())) {
                return true;
            }
            if (!Objects.equals(key, resourceSegmentation.getKey())) {
                return true;
            }
            if (!Objects.equals(nameEn, resourceSegmentation.getNameEn())) {
                return true;
            }
            if (!Objects.equals(nameRu, resourceSegmentation.getNameRu())) {
                return true;
            }
            if (!Objects.equals(descriptionEn, resourceSegmentation.getDescriptionEn())) {
                return true;
            }
            if (!Objects.equals(descriptionRu, resourceSegmentation.getDescriptionRu())) {
                return true;
            }
            if (!Objects.equals(deleted, resourceSegmentation.isDeleted())) {
                return true;
            }
            if (!Objects.equals(groupingOrder, resourceSegmentation.getGroupingOrder())) {
                return true;
            }
            if (!Objects.equals(uiSettings, resourceSegmentation.uiSettings)) {
                return true;
            }
            return false;
        }

        public ResourceSegmentationModel build() {
            return new ResourceSegmentationModel(id, tenantId, providerId, version, key, nameEn, nameRu,
                    descriptionEn, descriptionRu, deleted, groupingOrder, uiSettings);
        }

    }

    public static final class ProviderKey {
        private final String providerId;
        private final String key;

        public ProviderKey(String providerId, String key) {
            this.providerId = providerId;
            this.key = key;
        }

        public String getProviderId() {
            return providerId;
        }

        public String getKey() {
            return key;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ProviderKey that = (ProviderKey) o;
            return Objects.equals(providerId, that.providerId) &&
                    Objects.equals(key, that.key);
        }

        @Override
        public int hashCode() {
            return Objects.hash(providerId, key);
        }

        @Override
        public String toString() {
            return "ProviderKey{" +
                    "providerId='" + providerId + '\'' +
                    ", key='" + key + '\'' +
                    '}';
        }
    }
}
