package ru.yandex.intranet.d.model.resources.segments;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import ru.yandex.intranet.d.model.TenantId;

/**
 * Resource segment.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceSegmentModel {

    private final String id;
    private final TenantId tenantId;
    private final String segmentationId;
    private final long version;
    private final String key;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final boolean deleted;
    @Nullable
    private final Boolean uncommon;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceSegmentModel(@JsonProperty("id") String id,
                                @JsonProperty("tenantId") TenantId tenantId,
                                @JsonProperty("segmentationId") String segmentationId,
                                @JsonProperty("version") long version,
                                @JsonProperty("key") String key,
                                @JsonProperty("nameEn") String nameEn,
                                @JsonProperty("nameRu") String nameRu,
                                @JsonProperty("descriptionEn") String descriptionEn,
                                @JsonProperty("descriptionRu") String descriptionRu,
                                @JsonProperty("deleted") boolean deleted,
                                @JsonProperty("uncommon") @Nullable Boolean uncommon) {
        this.id = id;
        this.tenantId = tenantId;
        this.segmentationId = segmentationId;
        this.version = version;
        this.key = key;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.deleted = deleted;
        this.uncommon = uncommon;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceSegmentModel resourceSegment) {
        return new Builder(resourceSegment);
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getSegmentationId() {
        return segmentationId;
    }

    public long getVersion() {
        return version;
    }

    public String getKey() {
        return key;
    }

    public String getNameEn() {
        return nameEn;
    }

    public String getNameRu() {
        return nameRu;
    }

    public String getDescriptionEn() {
        return descriptionEn;
    }

    public String getDescriptionRu() {
        return descriptionRu;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public Optional<Boolean> getUncommon() {
        return Optional.ofNullable(uncommon);
    }

    @JsonIgnore
    public SegmentationAndKey getSegmentationAndKey() {
        return new SegmentationAndKey(segmentationId, key);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceSegmentModel that = (ResourceSegmentModel) o;
        return version == that.version &&
                deleted == that.deleted &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(segmentationId, that.segmentationId) &&
                Objects.equals(key, that.key) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(uncommon, that.uncommon);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, segmentationId, version, key, nameEn, nameRu, descriptionEn, descriptionRu,
                deleted, uncommon);
    }

    @Override
    public String toString() {
        return "ResourceSegmentModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", segmentationId='" + segmentationId + '\'' +
                ", version=" + version +
                ", key='" + key + '\'' +
                ", nameEn='" + nameEn + '\'' +
                ", nameRu='" + nameRu + '\'' +
                ", descriptionEn='" + descriptionEn + '\'' +
                ", descriptionRu='" + descriptionRu + '\'' +
                ", deleted=" + deleted +
                ", uncommon=" + uncommon +
                '}';
    }

    public static final class Builder {

        private String id;
        private TenantId tenantId;
        private String segmentationId;
        private Long version;
        private String key;
        private String nameEn;
        private String nameRu;
        private String descriptionEn;
        private String descriptionRu;
        private Boolean deleted;
        private Boolean uncommon;

        private Builder() {
        }

        private Builder(ResourceSegmentModel resourceSegment) {
            this.id = resourceSegment.getId();
            this.tenantId = resourceSegment.getTenantId();
            this.segmentationId = resourceSegment.getSegmentationId();
            this.version = resourceSegment.getVersion();
            this.key = resourceSegment.getKey();
            this.nameEn = resourceSegment.getNameEn();
            this.nameRu = resourceSegment.getNameRu();
            this.descriptionEn = resourceSegment.getDescriptionEn();
            this.descriptionRu = resourceSegment.getDescriptionRu();
            this.deleted = resourceSegment.isDeleted();
            this.uncommon = resourceSegment.getUncommon().orElse(null);
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder segmentationId(String segmentationId) {
            this.segmentationId = segmentationId;
            return this;
        }

        public Builder version(Long version) {
            this.version = version;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder nameEn(String nameEn) {
            this.nameEn = nameEn;
            return this;
        }

        public Builder nameRu(String nameRu) {
            this.nameRu = nameRu;
            return this;
        }

        public Builder descriptionEn(String descriptionEn) {
            this.descriptionEn = descriptionEn;
            return this;
        }

        public Builder descriptionRu(String descriptionRu) {
            this.descriptionRu = descriptionRu;
            return this;
        }

        public Builder deleted(Boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder uncommon(Boolean uncommon) {
            this.uncommon = uncommon;
            return this;
        }

        public boolean hasChanges(ResourceSegmentModel resourceSegment) {
            if (!Objects.equals(id, resourceSegment.getId())) {
                return true;
            }
            if (!Objects.equals(tenantId, resourceSegment.getTenantId())) {
                return true;
            }
            if (!Objects.equals(segmentationId, resourceSegment.getSegmentationId())) {
                return true;
            }
            if (!Objects.equals(key, resourceSegment.getKey())) {
                return true;
            }
            if (!Objects.equals(nameEn, resourceSegment.getNameEn())) {
                return true;
            }
            if (!Objects.equals(nameRu, resourceSegment.getNameRu())) {
                return true;
            }
            if (!Objects.equals(descriptionEn, resourceSegment.getDescriptionEn())) {
                return true;
            }
            if (!Objects.equals(descriptionRu, resourceSegment.getDescriptionRu())) {
                return true;
            }
            if (!Objects.equals(deleted, resourceSegment.isDeleted())) {
                return true;
            }
            if (!Objects.equals(uncommon, resourceSegment.getUncommon().orElse(null))) {
                return true;
            }
            return false;
        }

        public ResourceSegmentModel build() {
            return new ResourceSegmentModel(id, tenantId, segmentationId, version, key, nameEn, nameRu, descriptionEn,
                    descriptionRu, deleted, uncommon);
        }

    }

    public static final class SegmentationAndKey {
        private final String segmentationId;
        private final String key;

        public SegmentationAndKey(String segmentationId, String key) {
            this.segmentationId = segmentationId;
            this.key = key;
        }

        public String getSegmentationId() {
            return segmentationId;
        }

        public String getKey() {
            return key;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            SegmentationAndKey that = (SegmentationAndKey) o;
            return Objects.equals(segmentationId, that.segmentationId) && Objects.equals(key, that.key);
        }

        @Override
        public int hashCode() {
            return Objects.hash(segmentationId, key);
        }

        @Override
        public String toString() {
            return "SegmentationAndKey{" +
                    "segmentationId='" + segmentationId + '\'' +
                    ", key='" + key + '\'' +
                    '}';
        }
    }
}
