package ru.yandex.intranet.d.model.resources.types;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;

import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.providers.AggregationSettings;

/**
 * Resource type.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceTypeModel {

    private final String id;
    private final TenantId tenantId;
    private final String providerId;
    private final long version;
    private final String key;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final boolean deleted;
    private final String unitsEnsembleId;
    private final String baseUnitId;
    private final @Nullable Long sortingOrder;
    // Null means unspecified, use appropriate default value
    private final @Nullable AggregationSettings aggregationSettings;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public ResourceTypeModel(
            String id,
            TenantId tenantId,
            String providerId,
            long version,
            String key,
            String nameEn,
            String nameRu,
            String descriptionEn,
            String descriptionRu,
            boolean deleted,
            String unitsEnsembleId,
            String baseUnitId,
            @Nullable Long sortingOrder,
            @Nullable AggregationSettings aggregationSettings
    ) {
        this.id = id;
        this.tenantId = tenantId;
        this.providerId = providerId;
        this.version = version;
        this.key = key;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.deleted = deleted;
        this.unitsEnsembleId = unitsEnsembleId;
        this.baseUnitId = baseUnitId;
        this.sortingOrder = sortingOrder;
        this.aggregationSettings = aggregationSettings;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceTypeModel resourceType) {
        return new Builder(resourceType);
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getProviderId() {
        return providerId;
    }

    public long getVersion() {
        return version;
    }

    public String getKey() {
        return key;
    }

    public String getNameEn() {
        return nameEn;
    }

    public String getNameRu() {
        return nameRu;
    }

    public String getDescriptionEn() {
        return descriptionEn;
    }

    public String getDescriptionRu() {
        return descriptionRu;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public String getUnitsEnsembleId() {
        return unitsEnsembleId;
    }

    @Nullable
    public String getBaseUnitId() {
        return baseUnitId;
    }

    @Nullable
    public Long getSortingOrder() {
        return sortingOrder;
    }

    public Optional<AggregationSettings> getAggregationSettings() {
        return Optional.ofNullable(aggregationSettings);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceTypeModel that = (ResourceTypeModel) o;
        return version == that.version &&
                deleted == that.deleted &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(key, that.key) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(unitsEnsembleId, that.unitsEnsembleId) &&
                Objects.equals(baseUnitId, that.baseUnitId) &&
                Objects.equals(sortingOrder, that.sortingOrder) &&
                Objects.equals(aggregationSettings, that.aggregationSettings);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, providerId, version, key, nameEn, nameRu, descriptionEn, descriptionRu,
                deleted, unitsEnsembleId, baseUnitId, sortingOrder, aggregationSettings);
    }

    @Override
    public String toString() {
        return "ResourceTypeModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", providerId='" + providerId + '\'' +
                ", version=" + version +
                ", key='" + key + '\'' +
                ", nameEn='" + nameEn + '\'' +
                ", nameRu='" + nameRu + '\'' +
                ", descriptionEn='" + descriptionEn + '\'' +
                ", descriptionRu='" + descriptionRu + '\'' +
                ", deleted=" + deleted +
                ", unitsEnsembleId='" + unitsEnsembleId + '\'' +
                ", baseUnitId='" + baseUnitId + '\'' +
                ", sortingOrder=" + sortingOrder +
                ", aggregationSettings=" + aggregationSettings +
                '}';
    }

    public static final class Builder {

        private String id;
        private TenantId tenantId;
        private String providerId;
        private Long version;
        private String key;
        private String nameEn;
        private String nameRu;
        private String descriptionEn;
        private String descriptionRu;
        private Boolean deleted;
        private String unitsEnsembleId;
        private String baseUnitId;
        private @Nullable Long sortingOrder;
        private @Nullable AggregationSettings aggregationSettings;

        private Builder() {
        }

        private Builder(ResourceTypeModel resourceType) {
            this.id = resourceType.getId();
            this.tenantId = resourceType.getTenantId();
            this.providerId = resourceType.getProviderId();
            this.version = resourceType.getVersion();
            this.key = resourceType.getKey();
            this.nameEn = resourceType.getNameEn();
            this.nameRu = resourceType.getNameRu();
            this.descriptionEn = resourceType.getDescriptionEn();
            this.descriptionRu = resourceType.getDescriptionRu();
            this.deleted = resourceType.isDeleted();
            this.unitsEnsembleId = resourceType.getUnitsEnsembleId();
            this.baseUnitId = resourceType.getBaseUnitId();
            this.sortingOrder = resourceType.getSortingOrder();
            this.aggregationSettings = resourceType.getAggregationSettings().orElse(null);
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder providerId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder version(Long version) {
            this.version = version;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder nameEn(String nameEn) {
            this.nameEn = nameEn;
            return this;
        }

        public Builder nameRu(String nameRu) {
            this.nameRu = nameRu;
            return this;
        }

        public Builder descriptionEn(String descriptionEn) {
            this.descriptionEn = descriptionEn;
            return this;
        }

        public Builder descriptionRu(String descriptionRu) {
            this.descriptionRu = descriptionRu;
            return this;
        }

        public Builder deleted(Boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder unitsEnsembleId(String unitsEnsembleId) {
            this.unitsEnsembleId = unitsEnsembleId;
            return this;
        }

        public Builder baseUnitId(String baseUnitId) {
            this.baseUnitId = baseUnitId;
            return this;
        }

        public Builder sortingOrder(@Nullable Long sortingOrder) {
            this.sortingOrder = sortingOrder;
            return this;
        }

        public Builder aggregationSettings(@Nullable AggregationSettings aggregationSettings) {
            this.aggregationSettings = aggregationSettings;
            return this;
        }

        public boolean hasChanges(ResourceTypeModel resourceType) {
            if (!Objects.equals(id, resourceType.getId())) {
                return true;
            }
            if (!Objects.equals(tenantId, resourceType.getTenantId())) {
                return true;
            }
            if (!Objects.equals(providerId, resourceType.getProviderId())) {
                return true;
            }
            if (!Objects.equals(key, resourceType.getKey())) {
                return true;
            }
            if (!Objects.equals(nameEn, resourceType.getNameEn())) {
                return true;
            }
            if (!Objects.equals(nameRu, resourceType.getNameRu())) {
                return true;
            }
            if (!Objects.equals(descriptionEn, resourceType.getDescriptionEn())) {
                return true;
            }
            if (!Objects.equals(descriptionRu, resourceType.getDescriptionRu())) {
                return true;
            }
            if (!Objects.equals(deleted, resourceType.isDeleted())) {
                return true;
            }
            if (!Objects.equals(unitsEnsembleId, resourceType.getUnitsEnsembleId())) {
                return true;
            }
            if (!Objects.equals(baseUnitId, resourceType.getBaseUnitId())) {
                return true;
            }
            if (!Objects.equals(sortingOrder, resourceType.getSortingOrder())) {
                return true;
            }
            if (!Objects.equals(aggregationSettings, resourceType.getAggregationSettings().orElse(null))) {
                return true;
            }
            return false;
        }

        public ResourceTypeModel build() {
            return new ResourceTypeModel(id, tenantId, providerId, version, key, nameEn, nameRu, descriptionEn,
                    descriptionRu, deleted, unitsEnsembleId, baseUnitId, sortingOrder, aggregationSettings);
        }

    }

}
