package ru.yandex.intranet.d.model.services;

import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

/**
 * Service for recipe.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ServiceRecipeModel {

    private final long id;
    private final String name;
    private final String nameEn;
    private final String slug;
    private final ServiceState state;
    private final ServiceReadOnlyState readOnlyState;
    private final boolean exportable;
    private final Long parentId;

    @SuppressWarnings("checkstyle:ParameterNumber")
    public ServiceRecipeModel(long id,
                              String name,
                              String nameEn,
                              String slug,
                              ServiceState state,
                              ServiceReadOnlyState readOnlyState,
                              boolean exportable,
                              Long parentId) {
        this.id = id;
        this.name = name;
        this.nameEn = nameEn;
        this.slug = slug;
        this.state = state;
        this.readOnlyState = readOnlyState;
        this.exportable = exportable;
        this.parentId = parentId;
    }

    public static Builder builder() {
        return new Builder();
    }

    public long getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public String getNameEn() {
        return nameEn;
    }

    public String getSlug() {
        return slug;
    }

    public ServiceState getState() {
        return state;
    }

    public ServiceReadOnlyState getReadOnlyState() {
        return readOnlyState;
    }

    public boolean isExportable() {
        return exportable;
    }

    public Optional<Long> getParentId() {
        return Optional.ofNullable(parentId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ServiceRecipeModel that = (ServiceRecipeModel) o;
        return id == that.id && exportable == that.exportable && Objects.equals(name, that.name) &&
                Objects.equals(nameEn, that.nameEn) && Objects.equals(slug, that.slug) && state == that.state
                && readOnlyState == that.readOnlyState && Objects.equals(parentId, that.parentId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, name, nameEn, slug, state, readOnlyState, exportable, parentId);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public static final class Builder {

        private Long id;
        private String name;
        private String nameEn;
        private String slug;
        private ServiceState state;
        private ServiceReadOnlyState readOnlyState;
        private Boolean exportable;
        private Long parentId;

        private Builder() {
        }

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder nameEn(String nameEn) {
            this.nameEn = nameEn;
            return this;
        }

        public Builder slug(String slug) {
            this.slug = slug;
            return this;
        }

        public Builder state(ServiceState state) {
            this.state = state;
            return this;
        }

        public Builder readOnlyState(ServiceReadOnlyState readOnlyState) {
            this.readOnlyState = readOnlyState;
            return this;
        }

        public Builder exportable(boolean exportable) {
            this.exportable = exportable;
            return this;
        }

        public Builder parentId(Long parentId) {
            this.parentId = parentId;
            return this;
        }

        public ServiceRecipeModel build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(name, "Name is required");
            Preconditions.checkNotNull(nameEn, "NameEn is required");
            Preconditions.checkNotNull(slug, "Slug is required");
            Preconditions.checkNotNull(state, "State is required");
            Preconditions.checkNotNull(exportable, "Exportable is required");
            return new ServiceRecipeModel(id, name, nameEn, slug, state, readOnlyState, exportable, parentId);
        }

    }

}
