package ru.yandex.intranet.d.model.sync;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import ru.yandex.intranet.d.model.TenantId;

/**
 * ProvidersSyncErrorsModel.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 02-04-2021
 */
public final class ProvidersSyncErrorsModel {
    private final TenantId tenantId;
    private final Identity identity;
    private final Instant syncStart;
    private final Instant requestTimestamp;
    private final String request; // Json
    private final Errors errors;

    @SuppressWarnings("ParameterNumber")
    public ProvidersSyncErrorsModel(
            TenantId tenantId,
            String providerId,
            String accountSpaceId,
            String syncId,
            String errorId,
            Instant syncStart,
            Instant requestTimestamp,
            String request,
            Errors errors
    ) {
        this.tenantId = tenantId;
        this.identity = new Identity(
                providerId,
                accountSpaceId,
                syncId,
                errorId
        );
        this.syncStart = syncStart;
        this.requestTimestamp = requestTimestamp;
        this.request = request;
        this.errors = errors;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getProviderId() {
        return identity.providerId;
    }

    public Optional<String> getAccountSpaceId() {
        return Optional.ofNullable(identity.accountSpaceId);
    }

    public String getSyncId() {
        return identity.syncId;
    }

    public String getErrorId() {
        return identity.errorId;
    }

    public Instant getSyncStart() {
        return syncStart;
    }

    public Instant getRequestTimestamp() {
        return requestTimestamp;
    }

    public String getRequest() {
        return request;
    }

    public Errors getErrors() {
        return errors;
    }

    public Identity getIdentity() {
        return identity;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProvidersSyncErrorsModel that = (ProvidersSyncErrorsModel) o;
        return Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(identity.providerId, that.identity.providerId) &&
                Objects.equals(identity.accountSpaceId, that.identity.accountSpaceId) &&
                Objects.equals(identity.syncId, that.identity.syncId) &&
                Objects.equals(identity.errorId, that.identity.errorId) &&
                Objects.equals(syncStart, that.syncStart) &&
                Objects.equals(requestTimestamp, that.requestTimestamp) &&
                Objects.equals(request, that.request) &&
                Objects.equals(errors, that.errors);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                tenantId,
                identity.providerId,
                identity.accountSpaceId,
                identity.syncId,
                identity.errorId,
                syncStart,
                requestTimestamp,
                request,
                errors
        );
    }

    @Override
    public String toString() {
        return "ProvidersSyncErrorsModel{" +
                "tenantId=" + tenantId +
                ", providerId='" + identity.providerId + '\'' +
                ", accountSpaceId='" + identity.accountSpaceId + '\'' +
                ", syncId='" + identity.syncId + '\'' +
                ", errorId='" + identity.errorId + '\'' +
                ", syncStart=" + syncStart +
                ", requestTimestamp=" + requestTimestamp +
                ", request='" + request + '\'' +
                ", errors=" + errors +
                '}';
    }

    @SuppressWarnings("UnusedReturnValue")
    public static class Builder {
        private TenantId tenantId;
        private String providerId;
        private String accountSpaceId;
        private String syncId;
        private String errorId;
        private Instant syncStart;
        private Instant requestTimestamp;
        private String request;
        private Errors errors;

        public Builder() {
        }

        public Builder(ProvidersSyncStatusModel syncStatus, String accountSpaceId, String errorId) {
            this.tenantId = syncStatus.getTenantId();
            this.providerId = syncStatus.getProviderId();
            this.accountSpaceId = accountSpaceId;
            this.syncId = syncStatus.getNewSyncId();
            this.errorId = errorId;
            this.syncStart = syncStatus.getNewSyncStart();
        }

        public Builder setTenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder setProviderId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder setAccountSpaceId(String accountSpaceId) {
            this.accountSpaceId = accountSpaceId;
            return this;
        }

        public Builder setSyncId(String syncId) {
            this.syncId = syncId;
            return this;
        }

        public Builder setErrorId(String errorId) {
            this.errorId = errorId;
            return this;
        }

        public Builder setSyncStart(Instant syncStart) {
            this.syncStart = syncStart;
            return this;
        }

        public Builder setRequestTimestamp(Instant requestTimestamp) {
            this.requestTimestamp = requestTimestamp;
            return this;
        }

        public Builder setRequest(String request) {
            this.request = request;
            return this;
        }

        public Builder setErrors(Errors errors) {
            this.errors = errors;
            return this;
        }

        public ProvidersSyncErrorsModel build() {
            return new ProvidersSyncErrorsModel(
                    tenantId,
                    providerId,
                    accountSpaceId,
                    syncId,
                    errorId,
                    syncStart,
                    requestTimestamp,
                    request,
                    errors
            );
        }
    }

    public static final class Identity {
        private final String providerId;
        private final String accountSpaceId;
        private final String syncId;
        private final String errorId;

        public Identity(String providerId, String accountSpaceId, String syncId, String errorId) {
            this.providerId = providerId;
            this.accountSpaceId = accountSpaceId;
            this.syncId = syncId;
            this.errorId = errorId;
        }

        public String getProviderId() {
            return providerId;
        }

        public Optional<String> getAccountSpaceId() {
            return Optional.ofNullable(accountSpaceId);
        }

        public String getSyncId() {
            return syncId;
        }

        public String getErrorId() {
            return errorId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Identity identity = (Identity) o;
            return Objects.equals(providerId, identity.providerId) &&
                    Objects.equals(accountSpaceId, identity.accountSpaceId) &&
                    Objects.equals(syncId, identity.syncId) &&
                    Objects.equals(errorId, identity.errorId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(providerId, accountSpaceId, syncId, errorId);
        }

        @Override
        public String toString() {
            return "Identity{" +
                    "providerId='" + providerId + '\'' +
                    ", accountSpaceId='" + accountSpaceId + '\'' +
                    ", syncId='" + syncId + '\'' +
                    ", errorId='" + errorId + '\'' +
                    '}';
        }
    }
}
