package ru.yandex.intranet.d.model.sync;

import java.time.Instant;
import java.util.Objects;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import org.apache.commons.lang3.builder.ReflectionToStringBuilder;

import ru.yandex.intranet.d.model.TenantId;

/**
 * ProvidersSyncStatusModel.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 01-04-2021
 */
public final class ProvidersSyncStatusModel {
    private final TenantId tenantId;
    private final String providerId;
    private final Instant lastSyncStart;
    private final String lastSyncId;
    private final Instant lastSyncFinish;
    private final SyncStatuses lastSyncStatus;
    private final SyncStats lastSyncStats;
    private final Instant newSyncStart;
    private final String newSyncId;
    private final SyncStatuses newSyncStatus;
    private final Instant lastSuccessfulSyncFinish;

    @SuppressWarnings("ParameterNumber")
    public ProvidersSyncStatusModel(
            TenantId tenantId,
            String providerId,
            Instant lastSyncStart,
            String lastSyncId,
            Instant lastSyncFinish,
            SyncStatuses lastSyncStatus,
            SyncStats lastSyncStats, Instant newSyncStart,
            String newSyncId,
            SyncStatuses newSyncStatus,
            Instant lastSuccessfulSyncFinish
    ) {
        this.tenantId = tenantId;
        this.providerId = providerId;
        this.lastSyncStart = lastSyncStart;
        this.lastSyncId = lastSyncId;
        this.lastSyncFinish = lastSyncFinish;
        this.lastSyncStatus = lastSyncStatus;
        this.lastSyncStats = lastSyncStats;
        this.newSyncStart = newSyncStart;
        this.newSyncId = newSyncId;
        this.newSyncStatus = newSyncStatus;
        this.lastSuccessfulSyncFinish = lastSuccessfulSyncFinish;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getProviderId() {
        return providerId;
    }

    @Nullable
    public Instant getLastSyncStart() {
        return lastSyncStart;
    }

    @Nullable
    public String getLastSyncId() {
        return lastSyncId;
    }

    @Nullable
    public Instant getLastSyncFinish() {
        return lastSyncFinish;
    }

    @Nullable
    public SyncStatuses getLastSyncStatus() {
        return lastSyncStatus;
    }

    @Nullable
    public SyncStats getLastSyncStats() {
        return lastSyncStats;
    }

    public Instant getNewSyncStart() {
        return newSyncStart;
    }

    public String getNewSyncId() {
        return newSyncId;
    }

    public SyncStatuses getNewSyncStatus() {
        return newSyncStatus;
    }

    @Nullable
    public Instant getLastSuccessfulSyncFinish() {
        return lastSuccessfulSyncFinish;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProvidersSyncStatusModel that = (ProvidersSyncStatusModel) o;
        return Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(providerId, that.providerId) &&
                Objects.equals(lastSyncStart, that.lastSyncStart) &&
                Objects.equals(lastSyncId, that.lastSyncId) &&
                Objects.equals(lastSyncFinish, that.lastSyncFinish) &&
                lastSyncStatus == that.lastSyncStatus &&
                Objects.equals(lastSyncStats, that.lastSyncStats) &&
                Objects.equals(newSyncStart, that.newSyncStart) &&
                Objects.equals(newSyncId, that.newSyncId) &&
                newSyncStatus == that.newSyncStatus &&
                Objects.equals(lastSuccessfulSyncFinish, that.lastSuccessfulSyncFinish);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, providerId, lastSyncStart, lastSyncId, lastSyncFinish, lastSyncStatus,
                lastSyncStats, newSyncStart, newSyncId, newSyncStatus, lastSuccessfulSyncFinish);
    }

    @Override
    public String toString() {
        return ReflectionToStringBuilder.toString(this);
    }

    public enum SyncStatuses {
        RUNNING,
        DONE_OK,
        DONE_ERROR
    }

    @SuppressWarnings("UnusedReturnValue")
    public static class Builder {
        private TenantId tenantId;
        private String providerId;
        private Instant lastSyncStart;
        private String lastSyncId;
        private Instant lastSyncFinish;
        private SyncStatuses lastSyncStatus;
        private SyncStats lastSyncStats;
        private Instant newSyncStart;
        private String newSyncId;
        private SyncStatuses newSyncStatus;
        private Instant lastSuccessfulSyncFinish;

        public Builder() {
        }

        public Builder(ProvidersSyncStatusModel model) {
            this.tenantId = model.tenantId;
            this.providerId = model.providerId;
            this.lastSyncStart = model.lastSyncStart;
            this.lastSyncId = model.lastSyncId;
            this.lastSyncFinish = model.lastSyncFinish;
            this.lastSyncStatus = model.lastSyncStatus;
            this.lastSyncStats = model.lastSyncStats;
            this.newSyncStart = model.newSyncStart;
            this.newSyncId = model.newSyncId;
            this.newSyncStatus = model.newSyncStatus;
            this.lastSuccessfulSyncFinish = model.lastSuccessfulSyncFinish;
        }

        public Builder setTenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder setProviderId(String providerId) {
            this.providerId = providerId;
            return this;
        }

        public Builder setLastSyncStart(Instant lastSyncStart) {
            this.lastSyncStart = lastSyncStart;
            return this;
        }

        public Builder setLastSyncId(String lastSyncId) {
            this.lastSyncId = lastSyncId;
            return this;
        }

        public Builder setLastSyncFinish(Instant lastSyncFinish) {
            this.lastSyncFinish = lastSyncFinish;
            return this;
        }

        public Builder setLastSyncStatus(SyncStatuses lastSyncStatus) {
            this.lastSyncStatus = lastSyncStatus;
            return this;
        }

        public Builder setLastSyncStatus(String lastSyncStatus) {
            if (lastSyncStatus != null) {
                this.lastSyncStatus = SyncStatuses.valueOf(lastSyncStatus);
            }
            return this;
        }

        public Builder setLastSyncStats(SyncStats lastSyncStats) {
            this.lastSyncStats = lastSyncStats;
            return this;
        }

        public Builder setNewSyncStart(Instant newSyncStart) {
            this.newSyncStart = newSyncStart;
            return this;
        }

        public Builder setNewSyncId(String newSyncId) {
            this.newSyncId = newSyncId;
            return this;
        }

        public Builder setNewSyncStatus(SyncStatuses newSyncStatus) {
            this.newSyncStatus = newSyncStatus;
            return this;
        }

        public Builder setNewSyncStatus(String newSyncStatus) {
            this.newSyncStatus = SyncStatuses.valueOf(newSyncStatus);
            return this;
        }

        public Builder setLastSuccessfulSyncFinish(Instant lastSuccessfulSyncFinish) {
            this.lastSuccessfulSyncFinish = lastSuccessfulSyncFinish;
            return this;
        }

        public ProvidersSyncStatusModel build() {
            Preconditions.checkNotNull(tenantId);
            Preconditions.checkNotNull(providerId);
            Preconditions.checkNotNull(newSyncStart);
            Preconditions.checkNotNull(newSyncId);
            Preconditions.checkNotNull(newSyncStatus);
            return new ProvidersSyncStatusModel(
                    tenantId,
                    providerId,
                    lastSyncStart,
                    lastSyncId,
                    lastSyncFinish,
                    lastSyncStatus,
                    lastSyncStats,
                    newSyncStart,
                    newSyncId,
                    newSyncStatus,
                    lastSuccessfulSyncFinish
            );
        }
    }
}
