package ru.yandex.intranet.d.model.transfers;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Account transfer. There is exactly one AccountTransfer for each transferred account.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class AccountTransfer {

    private final String accountId;
    private final String sourceFolderId;
    private final String destinationFolderId;
    private final long sourceServiceId;
    private final long destinationServiceId;
    private final boolean transferQuotas;
    private final Set<ResourceQuotaTransfer> quotaTransfers;

    @JsonCreator
    public AccountTransfer(@JsonProperty("accountId") String accountId,
                           @JsonProperty("sourceFolderId") String sourceFolderId,
                           @JsonProperty("destinationFolderId") String destinationFolderId,
                           @JsonProperty("sourceServiceId") long sourceServiceId,
                           @JsonProperty("destinationServiceId") long destinationServiceId,
                           @JsonProperty("transferQuotas") boolean transferQuotas,
                           @JsonProperty("quotaTransfers") Set<ResourceQuotaTransfer> quotaTransfers) {
        this.accountId = accountId;
        this.sourceFolderId = sourceFolderId;
        this.destinationFolderId = destinationFolderId;
        this.sourceServiceId = sourceServiceId;
        this.destinationServiceId = destinationServiceId;
        this.transferQuotas = transferQuotas;
        this.quotaTransfers = quotaTransfers;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(AccountTransfer value) {
        return new Builder(value);
    }

    public String getAccountId() {
        return accountId;
    }

    public String getSourceFolderId() {
        return sourceFolderId;
    }

    public String getDestinationFolderId() {
        return destinationFolderId;
    }

    public long getSourceServiceId() {
        return sourceServiceId;
    }

    public long getDestinationServiceId() {
        return destinationServiceId;
    }

    public boolean isTransferQuotas() {
        return transferQuotas;
    }

    public Set<ResourceQuotaTransfer> getQuotaTransfers() {
        return quotaTransfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AccountTransfer that = (AccountTransfer) o;
        return sourceServiceId == that.sourceServiceId &&
                destinationServiceId == that.destinationServiceId &&
                transferQuotas == that.transferQuotas &&
                Objects.equals(accountId, that.accountId) &&
                Objects.equals(sourceFolderId, that.sourceFolderId) &&
                Objects.equals(destinationFolderId, that.destinationFolderId) &&
                Objects.equals(quotaTransfers, that.quotaTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(accountId, sourceFolderId, destinationFolderId, sourceServiceId, destinationServiceId,
                transferQuotas, quotaTransfers);
    }

    @Override
    public String toString() {
        return "AccountTransfer{" +
                "accountId='" + accountId + '\'' +
                ", sourceFolderId='" + sourceFolderId + '\'' +
                ", destinationFolderId='" + destinationFolderId + '\'' +
                ", sourceServiceId=" + sourceServiceId +
                ", destinationServiceId=" + destinationServiceId +
                ", transferQuotas=" + transferQuotas +
                ", quotaTransfers=" + quotaTransfers +
                '}';
    }

    public static final class Builder {

        private final Set<ResourceQuotaTransfer> quotaTransfers = new HashSet<>();

        private String accountId;
        private String sourceFolderId;
        private String destinationFolderId;
        private Long sourceServiceId;
        private Long destinationServiceId;
        private Boolean transferQuotas;

        private Builder() {
        }

        private Builder(AccountTransfer value) {
            this.accountId = value.accountId;
            this.sourceFolderId = value.sourceFolderId;
            this.destinationFolderId = value.destinationFolderId;
            this.sourceServiceId = value.sourceServiceId;
            this.destinationServiceId = value.destinationServiceId;
            this.transferQuotas = value.transferQuotas;
            this.quotaTransfers.addAll(value.quotaTransfers);
        }

        public Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public Builder sourceFolderId(String sourceFolderId) {
            this.sourceFolderId = sourceFolderId;
            return this;
        }

        public Builder destinationFolderId(String destinationFolderId) {
            this.destinationFolderId = destinationFolderId;
            return this;
        }

        public Builder sourceServiceId(long sourceServiceId) {
            this.sourceServiceId = sourceServiceId;
            return this;
        }

        public Builder destinationServiceId(long destinationServiceId) {
            this.destinationServiceId = destinationServiceId;
            return this;
        }

        public Builder transferQuotas(boolean transferQuotas) {
            this.transferQuotas = transferQuotas;
            return this;
        }

        public Builder addQuotaTransfer(ResourceQuotaTransfer transfer) {
            this.quotaTransfers.add(transfer);
            return this;
        }

        public Builder addQuotaTransfers(ResourceQuotaTransfer... transfers) {
            this.quotaTransfers.addAll(Arrays.asList(transfers));
            return this;
        }

        public Builder addQuotaTransfers(Collection<? extends ResourceQuotaTransfer> transfers) {
            this.quotaTransfers.addAll(transfers);
            return this;
        }

        public AccountTransfer build() {
            Preconditions.checkNotNull(accountId, "AccountId is required");
            Preconditions.checkNotNull(sourceFolderId, "SourceFolderId is required");
            Preconditions.checkNotNull(destinationFolderId, "DestinationFolderId is required");
            Preconditions.checkNotNull(sourceServiceId, "SourceServiceId is required");
            Preconditions.checkNotNull(destinationServiceId, "DestinationServiceId is required");
            Preconditions.checkNotNull(transferQuotas, "TransferQuotas is required");
            return new AccountTransfer(accountId, sourceFolderId, destinationFolderId, sourceServiceId,
                    destinationServiceId, transferQuotas, quotaTransfers);
        }

    }

}
