package ru.yandex.intranet.d.model.transfers;

import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Folder transfer. There is exactly one FolderTransfer for each transferred folder.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class FolderTransfer {

    private final String folderId;
    private final long sourceServiceId;
    private final long destinationServiceId;

    @JsonCreator
    public FolderTransfer(@JsonProperty("folderId") String folderId,
                          @JsonProperty("sourceServiceId") long sourceServiceId,
                          @JsonProperty("destinationServiceId") long destinationServiceId) {
        this.folderId = folderId;
        this.sourceServiceId = sourceServiceId;
        this.destinationServiceId = destinationServiceId;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(FolderTransfer value) {
        return new Builder(value);
    }

    public String getFolderId() {
        return folderId;
    }

    public long getSourceServiceId() {
        return sourceServiceId;
    }

    public long getDestinationServiceId() {
        return destinationServiceId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        FolderTransfer that = (FolderTransfer) o;
        return sourceServiceId == that.sourceServiceId &&
                destinationServiceId == that.destinationServiceId &&
                Objects.equals(folderId, that.folderId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(folderId, sourceServiceId, destinationServiceId);
    }

    @Override
    public String toString() {
        return "FolderTransfer{" +
                "folderId='" + folderId + '\'' +
                ", sourceServiceId=" + sourceServiceId +
                ", destinationServiceId=" + destinationServiceId +
                '}';
    }

    public static final class Builder {

        private String folderId;
        private Long sourceServiceId;
        private Long destinationServiceId;

        private Builder() {
        }

        private Builder(FolderTransfer value) {
            this.folderId = value.folderId;
            this.sourceServiceId = value.sourceServiceId;
            this.destinationServiceId = value.destinationServiceId;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder sourceServiceId(long sourceServiceId) {
            this.sourceServiceId = sourceServiceId;
            return this;
        }

        public Builder destinationServiceId(long destinationServiceId) {
            this.destinationServiceId = destinationServiceId;
            return this;
        }

        public FolderTransfer build() {
            Preconditions.checkNotNull(folderId, "FolderId is required");
            Preconditions.checkNotNull(sourceServiceId, "SourceServiceId is required");
            Preconditions.checkNotNull(destinationServiceId, "DestinationServiceId is required");
            return new FolderTransfer(folderId, sourceServiceId, destinationServiceId);
        }

    }

}
