package ru.yandex.intranet.d.model.transfers;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Provider responsible.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ProviderResponsible {

    private final String responsibleId;
    private final Set<String> providerIds;

    @JsonCreator
    public ProviderResponsible(@JsonProperty("responsibleId") String responsibleId,
                               @JsonProperty("providerIds") Set<String> providerIds) {
        this.responsibleId = responsibleId;
        this.providerIds = providerIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getResponsibleId() {
        return responsibleId;
    }

    public Set<String> getProviderIds() {
        return providerIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ProviderResponsible that = (ProviderResponsible) o;
        return Objects.equals(responsibleId, that.responsibleId) &&
                Objects.equals(providerIds, that.providerIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsibleId, providerIds);
    }

    @Override
    public String toString() {
        return "ProviderResponsible{" +
                "responsibleId='" + responsibleId + '\'' +
                ", providerIds=" + providerIds +
                '}';
    }

    public static final class Builder {

        private final Set<String> providerIds = new HashSet<>();

        private String responsibleId;

        private Builder() {
        }

        private Builder(ProviderResponsible value) {
            this.responsibleId = value.responsibleId;
            this.providerIds.addAll(value.providerIds);
        }

        public Builder responsibleId(String responsibleId) {
            this.responsibleId = responsibleId;
            return this;
        }

        public Builder addProviderId(String providerId) {
            this.providerIds.add(providerId);
            return this;
        }

        public Builder addProviderIds(String... providerIds) {
            this.providerIds.addAll(Arrays.asList(providerIds));
            return this;
        }

        public Builder addProviderIds(Collection<? extends String> providerIds) {
            this.providerIds.addAll(providerIds);
            return this;
        }

        public ProviderResponsible build() {
            Preconditions.checkNotNull(responsibleId, "ResponsibleId is required");
            return new ProviderResponsible(responsibleId, providerIds);
        }

    }

}
