package ru.yandex.intranet.d.model.transfers;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Quota transfer.
 * Each transfer is described by at least two QuotaTransfers,
 * one is for source folder, another is for destination folder.
 * Actually, the only invariant is this: for each resource sum of all deltas is zero.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class QuotaTransfer {

    private final String destinationFolderId;
    private final long destinationServiceId;
    private final Set<ResourceQuotaTransfer> transfers;

    @JsonCreator
    public QuotaTransfer(@JsonProperty("destinationFolderId") String destinationFolderId,
                         @JsonProperty("destinationServiceId") long destinationServiceId,
                         @JsonProperty("transfers") Set<ResourceQuotaTransfer> transfers) {
        this.destinationFolderId = destinationFolderId;
        this.destinationServiceId = destinationServiceId;
        this.transfers = transfers;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(QuotaTransfer value) {
        return new Builder(value);
    }

    public String getDestinationFolderId() {
        return destinationFolderId;
    }

    public long getDestinationServiceId() {
        return destinationServiceId;
    }

    public Set<ResourceQuotaTransfer> getTransfers() {
        return transfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        QuotaTransfer that = (QuotaTransfer) o;
        return destinationServiceId == that.destinationServiceId &&
                Objects.equals(destinationFolderId, that.destinationFolderId) &&
                Objects.equals(transfers, that.transfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(destinationFolderId, destinationServiceId, transfers);
    }

    @Override
    public String toString() {
        return "QuotaTransfer{" +
                "destinationFolderId='" + destinationFolderId + '\'' +
                ", destinationServiceId=" + destinationServiceId +
                ", transfers=" + transfers +
                '}';
    }

    public static final class Builder {

        private final Set<ResourceQuotaTransfer> transfers = new HashSet<>();

        private String destinationFolderId;
        private Long destinationServiceId;

        private Builder() {
        }

        private Builder(QuotaTransfer value) {
            this.destinationFolderId = value.destinationFolderId;
            this.destinationServiceId = value.destinationServiceId;
            this.transfers.addAll(value.transfers);
        }

        public Builder folderId(String folderId) {
            this.destinationFolderId = folderId;
            return this;
        }

        public Builder serviceId(long serviceId) {
            this.destinationServiceId = serviceId;
            return this;
        }

        public Builder addTransfer(ResourceQuotaTransfer transfer) {
            this.transfers.add(transfer);
            return this;
        }

        public Builder addFolderTransfers(ResourceQuotaTransfer... transfers) {
            this.transfers.addAll(Arrays.asList(transfers));
            return this;
        }

        public Builder addFolderTransfers(Collection<? extends ResourceQuotaTransfer> transfers) {
            this.transfers.addAll(transfers);
            return this;
        }

        public QuotaTransfer build() {
            Preconditions.checkNotNull(destinationFolderId, "DestinationFolderId is required");
            Preconditions.checkNotNull(destinationServiceId, "DestinationServiceId is required");
            return new QuotaTransfer(destinationFolderId, destinationServiceId, transfers);
        }

    }

}
