package ru.yandex.intranet.d.model.transfers;

import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Resource provision transfer.
 * Delta is subtracted from source and added to destination.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceProvisionTransfer {

    private final String resourceId;
    private final long delta;
    private final boolean transferQuota;

    @JsonCreator
    public ResourceProvisionTransfer(@JsonProperty("resourceId") String resourceId,
                                     @JsonProperty("delta") long delta,
                                     @JsonProperty("transferQuota") boolean transferQuota) {
        this.resourceId = resourceId;
        this.delta = delta;
        this.transferQuota = transferQuota;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceProvisionTransfer value) {
        return new Builder(value);
    }

    public String getResourceId() {
        return resourceId;
    }

    public long getDelta() {
        return delta;
    }

    public boolean isTransferQuota() {
        return transferQuota;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceProvisionTransfer that = (ResourceProvisionTransfer) o;
        return delta == that.delta &&
                transferQuota == that.transferQuota &&
                Objects.equals(resourceId, that.resourceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceId, delta, transferQuota);
    }

    @Override
    public String toString() {
        return "ResourceProvisionTransfer{" +
                "resourceId='" + resourceId + '\'' +
                ", delta=" + delta +
                ", transferQuota=" + transferQuota +
                '}';
    }

    public static final class Builder {

        private String resourceId;
        private Long delta;
        private Boolean transferQuota;

        private Builder() {
        }

        private Builder(ResourceProvisionTransfer value) {
            this.resourceId = value.resourceId;
            this.delta = value.delta;
            this.transferQuota = value.transferQuota;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder delta(long delta) {
            this.delta = delta;
            return this;
        }

        public Builder transferQuota(Boolean transferQuota) {
            this.transferQuota = transferQuota;
            return this;
        }

        public ResourceProvisionTransfer build() {
            Preconditions.checkNotNull(resourceId, "ResourceId is required");
            Preconditions.checkNotNull(delta, "Delta is required");
            Preconditions.checkNotNull(transferQuota, "TransferQuota is required");
            return new ResourceProvisionTransfer(resourceId, delta, transferQuota);
        }

    }

}
