package ru.yandex.intranet.d.model.transfers;

import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Resource quota transfer.
 * Delta is subtracted from source and added to destination.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class ResourceQuotaTransfer {

    private final String resourceId;
    private final long delta;

    @JsonCreator
    public ResourceQuotaTransfer(@JsonProperty("resourceId") String resourceId,
                                 @JsonProperty("delta") long delta) {
        this.resourceId = resourceId;
        this.delta = delta;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ResourceQuotaTransfer value) {
        return new Builder(value);
    }

    public String getResourceId() {
        return resourceId;
    }

    public long getDelta() {
        return delta;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ResourceQuotaTransfer that = (ResourceQuotaTransfer) o;
        return delta == that.delta &&
                Objects.equals(resourceId, that.resourceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceId, delta);
    }

    @Override
    public String toString() {
        return "ResourceQuotaTransfer{" +
                "resourceId='" + resourceId + '\'' +
                ", delta=" + delta +
                '}';
    }

    public static final class Builder {

        private String resourceId;
        private Long delta;

        private Builder() {
        }

        private Builder(ResourceQuotaTransfer value) {
            this.resourceId = value.resourceId;
            this.delta = value.delta;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder delta(long delta) {
            this.delta = delta;
            return this;
        }

        public ResourceQuotaTransfer build() {
            Preconditions.checkNotNull(resourceId, "ResourceId is required");
            Preconditions.checkNotNull(delta, "Delta is required");
            return new ResourceQuotaTransfer(resourceId, delta);
        }

    }

}
