package ru.yandex.intranet.d.model.transfers;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;

/**
 * Transfer application details.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public final class TransferApplicationDetails {

    private final TransferApplicationErrors errorsEn;
    private final TransferApplicationErrors errorsRu;
    private final Map<String, Set<String>> folderOpLogIdsByFolderId;
    private final Set<String> operationIds;
    private final Map<String, OperationStatus> operationStatusById;
    private final Map<String, LocalizedTransferApplicationErrors> errorsByOperationId;

    @JsonCreator
    public TransferApplicationDetails(TransferApplicationErrors errorsEn,
                                      TransferApplicationErrors errorsRu,
                                      Map<String, Set<String>> folderOpLogIds,
                                      Set<String> operationIds,
                                      Map<String, OperationStatus> operationStatusById,
                                      Map<String, LocalizedTransferApplicationErrors> errorsByOperationId) {
        this.errorsEn = errorsEn;
        this.errorsRu = errorsRu;
        this.folderOpLogIdsByFolderId = folderOpLogIds != null ? folderOpLogIds : new HashMap<>();
        this.operationIds = operationIds != null ? operationIds : new HashSet<>();
        this.operationStatusById = operationStatusById != null ? operationStatusById : new HashMap<>();
        this.errorsByOperationId = errorsByOperationId != null ? errorsByOperationId : new HashMap<>();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferApplicationDetails value) {
        return new Builder(value);
    }

    public Optional<TransferApplicationErrors> getErrorsEn() {
        return Optional.ofNullable(errorsEn);
    }

    public Optional<TransferApplicationErrors> getErrorsRu() {
        return Optional.ofNullable(errorsRu);
    }

    public Map<String, Set<String>> getFolderOpLogIdsByFolderId() {
        return folderOpLogIdsByFolderId;
    }

    public Set<String> getOperationIds() {
        return operationIds;
    }

    public Map<String, OperationStatus> getOperationStatusById() {
        return operationStatusById;
    }

    public Map<String, LocalizedTransferApplicationErrors> getErrorsByOperationId() {
        return errorsByOperationId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferApplicationDetails that = (TransferApplicationDetails) o;
        return Objects.equals(errorsEn, that.errorsEn) &&
                Objects.equals(errorsRu, that.errorsRu) &&
                Objects.equals(folderOpLogIdsByFolderId, that.folderOpLogIdsByFolderId) &&
                Objects.equals(operationIds, that.operationIds) &&
                Objects.equals(operationStatusById, that.operationStatusById) &&
                Objects.equals(errorsByOperationId, that.errorsByOperationId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(errorsEn, errorsRu, folderOpLogIdsByFolderId, operationIds, operationStatusById,
                errorsByOperationId);
    }

    @Override
    public String toString() {
        return "TransferApplicationDetails{" +
                "errorsEn=" + errorsEn +
                ", errorsRu=" + errorsRu +
                ", folderOpLogIdsByFolderId=" + folderOpLogIdsByFolderId +
                ", operationIds=" + operationIds +
                ", operationStatusById=" + operationStatusById +
                ", errorsByOperationId=" + errorsByOperationId +
                '}';
    }

    public static final class Builder {

        private final Map<String, Set<String>> folderOpLogIdsByFolderId = new HashMap<>();
        private final Set<String> operationIds = new HashSet<>();
        private final Map<String, OperationStatus> operationStatusById = new HashMap<>();
        private final Map<String, LocalizedTransferApplicationErrors> errorsByOperationId = new HashMap<>();

        private TransferApplicationErrors errorsEn;
        private TransferApplicationErrors errorsRu;

        private Builder() {
        }

        private Builder(TransferApplicationDetails value) {
            this.errorsEn = value.errorsEn;
            this.errorsRu = value.errorsRu;
            value.folderOpLogIdsByFolderId.forEach((k, v) -> folderOpLogIdsByFolderId.put(k, new HashSet<>(v)));
            this.operationIds.addAll(value.operationIds);
            this.operationStatusById.putAll(value.operationStatusById);
            this.errorsByOperationId.putAll(value.errorsByOperationId);
        }

        public Builder errorsEn(TransferApplicationErrors errorsEn) {
            this.errorsEn = errorsEn;
            return this;
        }

        public Builder errorsRu(TransferApplicationErrors errorsRu) {
            this.errorsRu = errorsRu;
            return this;
        }

        public Builder addFolderOpLogId(String folderId, String opLogId) {
            this.folderOpLogIdsByFolderId.computeIfAbsent(folderId, k -> new HashSet<>()).add(opLogId);
            return this;
        }

        public Builder addFolderOpLogIds(String folderId, String... opLogIds) {
            this.folderOpLogIdsByFolderId.computeIfAbsent(folderId,
                    k -> new HashSet<>()).addAll(Arrays.asList(opLogIds));
            return this;
        }

        public Builder addFolderOpLogIds(String folderId, Collection<? extends String> opLogIds) {
            this.folderOpLogIdsByFolderId.computeIfAbsent(folderId, k -> new HashSet<>()).addAll(opLogIds);
            return this;
        }

        public Builder addFolderOpLogIds(Map<String, Collection<? extends String>> folderOpLogIdsByFolderId) {
            folderOpLogIdsByFolderId.forEach((folderId, opLogIds) ->
                    this.folderOpLogIdsByFolderId.computeIfAbsent(folderId, k -> new HashSet<>()).addAll(opLogIds));
            return this;
        }

        public Builder addOperationId(String id) {
            this.operationIds.add(id);
            return this;
        }

        public Builder addOperationIds(String... ids) {
            this.operationIds.addAll(Arrays.asList(ids));
            return this;
        }

        public Builder addOperationIds(Collection<? extends String> ids) {
            this.operationIds.addAll(ids);
            return this;
        }

        public Builder addOperationStatus(String operationId, OperationStatus status) {
            this.operationStatusById.put(operationId, status);
            return this;
        }

        public Builder addOperationStatuses(Map<String, OperationStatus> operationStatusById) {
            this.operationStatusById.putAll(operationStatusById);
            return this;
        }

        public Builder addOperationErrors(String operationId, LocalizedTransferApplicationErrors errors) {
            this.errorsByOperationId.put(operationId, errors);
            return this;
        }

        public TransferApplicationDetails build() {
            return new TransferApplicationDetails(errorsEn, errorsRu, folderOpLogIdsByFolderId, operationIds,
                    operationStatusById, errorsByOperationId);
        }

    }

}
