package ru.yandex.intranet.d.model.transfers;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.TypedError;

/**
 * Transfer application errors.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public final class TransferApplicationErrors {

    private final Set<String> errors;
    private final Map<String, Set<String>> fieldErrors;
    private final Map<String, Set<Object>> details;

    @JsonCreator
    public TransferApplicationErrors(Set<String> errors,
                                     Map<String, Set<String>> fieldErrors,
                                     Map<String, Set<Object>> details) {
        this.errors = errors != null ? errors : new HashSet<>();
        this.fieldErrors = fieldErrors != null ? fieldErrors : new HashMap<>();
        this.details = details != null ? details : new HashMap<>();
    }

    @JsonIgnore
    public TransferApplicationErrors(ErrorCollection errorCollection) {
        this.errors = errorCollection.getErrors().stream().map(TypedError::getError).collect(Collectors.toSet());
        this.fieldErrors = errorCollection.getFieldErrors().entrySet()
                .stream()
                .collect(Collectors.toMap(
                                Map.Entry::getKey,
                                e -> e.getValue().stream().map(TypedError::getError).collect(Collectors.toSet())
                        )
                );
        this.details = errorCollection.getDetails();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferApplicationErrors value) {
        return new Builder(value);
    }

    public Set<String> getErrors() {
        return errors;
    }

    public Map<String, Set<String>> getFieldErrors() {
        return fieldErrors;
    }

    public Map<String, Set<Object>> getDetails() {
        return details;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferApplicationErrors that = (TransferApplicationErrors) o;
        return Objects.equals(errors, that.errors) &&
                Objects.equals(fieldErrors, that.fieldErrors) &&
                Objects.equals(details, that.details);
    }

    @Override
    public int hashCode() {
        return Objects.hash(errors, fieldErrors, details);
    }

    @Override
    public String toString() {
        return "TransferApplicationErrors{" +
                "errors=" + errors +
                ", fieldErrors=" + fieldErrors +
                ", details=" + details +
                '}';
    }

    public static final class Builder {

        private final Set<String> errors = new HashSet<>();
        private final Map<String, Set<String>> fieldErrors = new HashMap<>();
        private final Map<String, Set<Object>> details = new HashMap<>();

        private Builder() {
        }

        private Builder(TransferApplicationErrors value) {
            this.errors.addAll(value.errors);
            value.fieldErrors.forEach((k, v) -> fieldErrors.put(k, new HashSet<>(v)));
        }

        public Builder addError(String value) {
            this.errors.add(value);
            return this;
        }

        public Builder addErrors(String... values) {
            this.errors.addAll(Arrays.asList(values));
            return this;
        }

        public Builder addErrors(Collection<? extends String> values) {
            this.errors.addAll(values);
            return this;
        }

        public Builder addFieldError(String field, String value) {
            this.fieldErrors.computeIfAbsent(field, k -> new HashSet<>()).add(value);
            return this;
        }

        public Builder addFieldErrors(String field, String... values) {
            this.fieldErrors.computeIfAbsent(field, k -> new HashSet<>()).addAll(Arrays.asList(values));
            return this;
        }

        public Builder addFieldErrors(String field, Collection<? extends String> values) {
            this.fieldErrors.computeIfAbsent(field, k -> new HashSet<>()).addAll(values);
            return this;
        }

        public Builder addFieldErrors(Map<String, Collection<? extends String>> map) {
            map.forEach((folderId, opLogIds) -> this.fieldErrors.computeIfAbsent(folderId, k -> new HashSet<>())
                    .addAll(opLogIds));
            return this;
        }

        public Builder addDetails(Map<String, ? extends Collection<?>> map) {
            map.forEach((detail, values) -> this.details.computeIfAbsent(detail, k -> new HashSet<>())
                    .addAll(values));
            return this;
        }

        public TransferApplicationErrors build() {
            return new TransferApplicationErrors(errors, fieldErrors, details);
        }

    }

}
