package ru.yandex.intranet.d.model.transfers;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Transfer parameters.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
@JsonInclude(JsonInclude.Include.NON_EMPTY)
public final class TransferParameters {

    private final Set<FolderTransfer> folderTransfers;
    private final Set<QuotaTransfer> quotaTransfers;
    private final Set<AccountTransfer> accountTransfers;
    private final Set<ProvisionTransfer> provisionTransfers;

    @JsonCreator
    public TransferParameters(@JsonProperty("folderTransfers") Set<FolderTransfer> folderTransfers,
                              @JsonProperty("quotaTransfers") Set<QuotaTransfer> quotaTransfers,
                              @JsonProperty("accountTransfers") Set<AccountTransfer> accountTransfers,
                              @JsonProperty("provisionTransfers") Set<ProvisionTransfer> provisionTransfers) {
        this.folderTransfers = folderTransfers != null ? folderTransfers : new HashSet<>();
        this.quotaTransfers = quotaTransfers != null ? quotaTransfers : new HashSet<>();
        this.accountTransfers = accountTransfers != null ? accountTransfers : new HashSet<>();
        this.provisionTransfers = provisionTransfers != null ? provisionTransfers : new HashSet<>();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferParameters value) {
        return new Builder(value);
    }

    public Set<FolderTransfer> getFolderTransfers() {
        return folderTransfers;
    }

    public Set<QuotaTransfer> getQuotaTransfers() {
        return quotaTransfers;
    }

    public Set<AccountTransfer> getAccountTransfers() {
        return accountTransfers;
    }

    public Set<ProvisionTransfer> getProvisionTransfers() {
        return provisionTransfers;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferParameters that = (TransferParameters) o;
        return Objects.equals(folderTransfers, that.folderTransfers) &&
                Objects.equals(quotaTransfers, that.quotaTransfers) &&
                Objects.equals(accountTransfers, that.accountTransfers) &&
                Objects.equals(provisionTransfers, that.provisionTransfers);
    }

    @Override
    public int hashCode() {
        return Objects.hash(folderTransfers, quotaTransfers, accountTransfers, provisionTransfers);
    }

    @Override
    public String toString() {
        return "TransferParameters{" +
                "folderTransfers=" + folderTransfers +
                ", quotaTransfers=" + quotaTransfers +
                ", accountTransfers=" + accountTransfers +
                ", provisionTransfers=" + provisionTransfers +
                '}';
    }

    public static final class Builder {

        private final Set<FolderTransfer> folderTransfers = new HashSet<>();
        private final Set<QuotaTransfer> quotaTransfers = new HashSet<>();
        private final Set<AccountTransfer> accountTransfers = new HashSet<>();
        private final Set<ProvisionTransfer> provisionTransfers = new HashSet<>();

        private Builder() {
        }

        private Builder(TransferParameters value) {
            this.folderTransfers.addAll(value.folderTransfers);
            this.quotaTransfers.addAll(value.quotaTransfers);
            this.accountTransfers.addAll(value.accountTransfers);
            this.provisionTransfers.addAll(value.provisionTransfers);
        }

        public Builder addFolderTransfer(FolderTransfer transfer) {
            this.folderTransfers.add(transfer);
            return this;
        }

        public Builder addFolderTransfers(FolderTransfer... transfers) {
            this.folderTransfers.addAll(Arrays.asList(transfers));
            return this;
        }

        public Builder addFolderTransfers(Collection<? extends FolderTransfer> transfers) {
            this.folderTransfers.addAll(transfers);
            return this;
        }

        public Builder addQuotaTransfer(QuotaTransfer transfer) {
            this.quotaTransfers.add(transfer);
            return this;
        }

        public Builder addQuotaTransfers(QuotaTransfer... transfers) {
            this.quotaTransfers.addAll(Arrays.asList(transfers));
            return this;
        }

        public Builder addQuotaTransfers(Collection<? extends QuotaTransfer> transfers) {
            this.quotaTransfers.addAll(transfers);
            return this;
        }

        public Builder addAccountTransfer(AccountTransfer transfer) {
            this.accountTransfers.add(transfer);
            return this;
        }

        public Builder addAccountTransfers(AccountTransfer... transfers) {
            this.accountTransfers.addAll(Arrays.asList(transfers));
            return this;
        }

        public Builder addAccountTransfers(Collection<? extends AccountTransfer> transfers) {
            this.accountTransfers.addAll(transfers);
            return this;
        }

        public Builder addProvisionTransfer(ProvisionTransfer transfer) {
            this.provisionTransfers.add(transfer);
            return this;
        }

        public Builder addProvisionTransfers(ProvisionTransfer... transfers) {
            this.provisionTransfers.addAll(Arrays.asList(transfers));
            return this;
        }

        public Builder addProvisionTransfers(Collection<? extends ProvisionTransfer> transfers) {
            this.provisionTransfers.addAll(transfers);
            return this;
        }

        public TransferParameters build() {
            return new TransferParameters(folderTransfers, quotaTransfers, accountTransfers, provisionTransfers);
        }

    }

}
