package ru.yandex.intranet.d.model.transfers;

import java.time.Instant;
import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Transfer request by folder.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferRequestByFolderModel {

    private final TenantId tenantId;
    private final String folderId;
    private final TransferRequestStatus status;
    private final Instant createdAt;
    private final String transferRequestId;

    public TransferRequestByFolderModel(TenantId tenantId,
                                        String folderId,
                                        TransferRequestStatus status,
                                        Instant createdAt,
                                        String transferRequestId) {
        this.tenantId = tenantId;
        this.folderId = folderId;
        this.status = status;
        this.createdAt = createdAt;
        this.transferRequestId = transferRequestId;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferRequestByFolderModel value) {
        return new Builder(value);
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getFolderId() {
        return folderId;
    }

    public TransferRequestStatus getStatus() {
        return status;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public String getTransferRequestId() {
        return transferRequestId;
    }

    public Identity getIdentity() {
        return new Identity(folderId, status, createdAt, transferRequestId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestByFolderModel that = (TransferRequestByFolderModel) o;
        return Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(folderId, that.folderId) &&
                status == that.status &&
                Objects.equals(createdAt, that.createdAt) &&
                Objects.equals(transferRequestId, that.transferRequestId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, folderId, status, createdAt, transferRequestId);
    }

    @Override
    public String toString() {
        return "TransferRequestByFolderModel{" +
                "tenantId=" + tenantId +
                ", folderId='" + folderId + '\'' +
                ", status=" + status +
                ", createdAt=" + createdAt +
                ", transferRequestId='" + transferRequestId + '\'' +
                '}';
    }

    public static final class Identity {

        private final String folderId;
        private final TransferRequestStatus status;
        private final Instant createdAt;
        private final String transferRequestId;

        public Identity(String folderId,
                        TransferRequestStatus status,
                        Instant createdAt,
                        String transferRequestId) {
            this.folderId = folderId;
            this.status = status;
            this.createdAt = createdAt;
            this.transferRequestId = transferRequestId;
        }

        public String getFolderId() {
            return folderId;
        }

        public TransferRequestStatus getStatus() {
            return status;
        }

        public Instant getCreatedAt() {
            return createdAt;
        }

        public String getTransferRequestId() {
            return transferRequestId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Identity identity = (Identity) o;
            return Objects.equals(folderId, identity.folderId) &&
                    status == identity.status &&
                    Objects.equals(createdAt, identity.createdAt) &&
                    Objects.equals(transferRequestId, identity.transferRequestId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(folderId, status, createdAt, transferRequestId);
        }

        @Override
        public String toString() {
            return "Identity{" +
                    "folderId='" + folderId + '\'' +
                    ", status=" + status +
                    ", createdAt=" + createdAt +
                    ", transferRequestId='" + transferRequestId + '\'' +
                    '}';
        }

    }

    public static final class Builder {

        private TenantId tenantId;
        private String folderId;
        private TransferRequestStatus status;
        private Instant createdAt;
        private String transferRequestId;

        private Builder() {
        }

        private Builder(TransferRequestByFolderModel value) {
            this.tenantId = value.tenantId;
            this.folderId = value.folderId;
            this.status = value.status;
            this.createdAt = value.createdAt;
            this.transferRequestId = value.transferRequestId;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder status(TransferRequestStatus status) {
            this.status = status;
            return this;
        }

        public Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder transferRequestId(String transferRequestId) {
            this.transferRequestId = transferRequestId;
            return this;
        }

        public TransferRequestByFolderModel build() {
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(folderId, "FolderId is required");
            Preconditions.checkNotNull(status, "Status is required");
            Preconditions.checkNotNull(createdAt, "CreatedAt is required");
            Preconditions.checkNotNull(transferRequestId, "TransferRequestId is required");
            return new TransferRequestByFolderModel(tenantId, folderId, status, createdAt, transferRequestId);
        }

    }

}
