package ru.yandex.intranet.d.model.transfers;

import java.time.Instant;
import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Transfer request by responsible.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferRequestByResponsibleModel {

    private final TenantId tenantId;
    private final String responsibleId;
    private final TransferRequestStatus status;
    private final Instant createdAt;
    private final String transferRequestId;

    public TransferRequestByResponsibleModel(TenantId tenantId,
                                             String responsibleId,
                                             TransferRequestStatus status,
                                             Instant createdAt,
                                             String transferRequestId) {
        this.tenantId = tenantId;
        this.responsibleId = responsibleId;
        this.status = status;
        this.createdAt = createdAt;
        this.transferRequestId = transferRequestId;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferRequestByResponsibleModel value) {
        return new Builder(value);
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getResponsibleId() {
        return responsibleId;
    }

    public TransferRequestStatus getStatus() {
        return status;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public String getTransferRequestId() {
        return transferRequestId;
    }

    public Identity getIdentity() {
        return new Identity(responsibleId, status, createdAt, transferRequestId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestByResponsibleModel that = (TransferRequestByResponsibleModel) o;
        return Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(responsibleId, that.responsibleId) &&
                status == that.status &&
                Objects.equals(createdAt, that.createdAt) &&
                Objects.equals(transferRequestId, that.transferRequestId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, responsibleId, status, createdAt, transferRequestId);
    }

    @Override
    public String toString() {
        return "TransferRequestByResponsibleModel{" +
                "tenantId=" + tenantId +
                ", responsibleId='" + responsibleId + '\'' +
                ", status=" + status +
                ", createdAt=" + createdAt +
                ", transferRequestId='" + transferRequestId + '\'' +
                '}';
    }

    public static final class Identity {

        private final String responsibleId;
        private final TransferRequestStatus status;
        private final Instant createdAt;
        private final String transferRequestId;

        public Identity(String responsibleId,
                        TransferRequestStatus status,
                        Instant createdAt,
                        String transferRequestId) {
            this.responsibleId = responsibleId;
            this.status = status;
            this.createdAt = createdAt;
            this.transferRequestId = transferRequestId;
        }

        public String getResponsibleId() {
            return responsibleId;
        }

        public TransferRequestStatus getStatus() {
            return status;
        }

        public Instant getCreatedAt() {
            return createdAt;
        }

        public String getTransferRequestId() {
            return transferRequestId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Identity identity = (Identity) o;
            return Objects.equals(responsibleId, identity.responsibleId) &&
                    status == identity.status &&
                    Objects.equals(createdAt, identity.createdAt) &&
                    Objects.equals(transferRequestId, identity.transferRequestId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(responsibleId, status, createdAt, transferRequestId);
        }

        @Override
        public String toString() {
            return "Identity{" +
                    "responsibleId='" + responsibleId + '\'' +
                    ", status=" + status +
                    ", createdAt=" + createdAt +
                    ", transferRequestId='" + transferRequestId + '\'' +
                    '}';
        }

    }

    public static final class Builder {

        private TenantId tenantId;
        private String responsibleId;
        private TransferRequestStatus status;
        private Instant createdAt;
        private String transferRequestId;

        private Builder() {
        }

        private Builder(TransferRequestByResponsibleModel value) {
            this.tenantId = value.tenantId;
            this.responsibleId = value.responsibleId;
            this.status = value.status;
            this.createdAt = value.createdAt;
            this.transferRequestId = value.transferRequestId;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder responsibleId(String responsibleId) {
            this.responsibleId = responsibleId;
            return this;
        }

        public Builder status(TransferRequestStatus status) {
            this.status = status;
            return this;
        }

        public Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder transferRequestId(String transferRequestId) {
            this.transferRequestId = transferRequestId;
            return this;
        }

        public TransferRequestByResponsibleModel build() {
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(responsibleId, "ResponsibleId is required");
            Preconditions.checkNotNull(status, "Status is required");
            Preconditions.checkNotNull(createdAt, "CreatedAt is required");
            Preconditions.checkNotNull(transferRequestId, "TransferRequestId is required");
            return new TransferRequestByResponsibleModel(tenantId, responsibleId, status, createdAt,
                    transferRequestId);
        }

    }

}
