package ru.yandex.intranet.d.model.transfers;

import java.time.Instant;
import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Transfer request by service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferRequestByServiceModel {

    private final TenantId tenantId;
    private final long serviceId;
    private final TransferRequestStatus status;
    private final Instant createdAt;
    private final String transferRequestId;

    public TransferRequestByServiceModel(TenantId tenantId,
                                         long serviceId,
                                         TransferRequestStatus status,
                                         Instant createdAt,
                                         String transferRequestId) {
        this.tenantId = tenantId;
        this.serviceId = serviceId;
        this.status = status;
        this.createdAt = createdAt;
        this.transferRequestId = transferRequestId;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferRequestByServiceModel value) {
        return new Builder(value);
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public long getServiceId() {
        return serviceId;
    }

    public TransferRequestStatus getStatus() {
        return status;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public String getTransferRequestId() {
        return transferRequestId;
    }

    public Identity getIdentity() {
        return new Identity(serviceId, status, createdAt, transferRequestId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestByServiceModel that = (TransferRequestByServiceModel) o;
        return serviceId == that.serviceId &&
                Objects.equals(tenantId, that.tenantId) &&
                status == that.status &&
                Objects.equals(createdAt, that.createdAt) &&
                Objects.equals(transferRequestId, that.transferRequestId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(tenantId, serviceId, status, createdAt, transferRequestId);
    }

    @Override
    public String toString() {
        return "TransferRequestByServiceModel{" +
                "tenantId=" + tenantId +
                ", serviceId=" + serviceId +
                ", status=" + status +
                ", createdAt=" + createdAt +
                ", transferRequestId='" + transferRequestId + '\'' +
                '}';
    }

    public static final class Identity {

        private final long serviceId;
        private final TransferRequestStatus status;
        private final Instant createdAt;
        private final String transferRequestId;

        public Identity(long serviceId, TransferRequestStatus status, Instant createdAt, String transferRequestId) {
            this.serviceId = serviceId;
            this.status = status;
            this.createdAt = createdAt;
            this.transferRequestId = transferRequestId;
        }

        public long getServiceId() {
            return serviceId;
        }

        public TransferRequestStatus getStatus() {
            return status;
        }

        public Instant getCreatedAt() {
            return createdAt;
        }

        public String getTransferRequestId() {
            return transferRequestId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Identity identity = (Identity) o;
            return serviceId == identity.serviceId &&
                    status == identity.status &&
                    Objects.equals(createdAt, identity.createdAt) &&
                    Objects.equals(transferRequestId, identity.transferRequestId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(serviceId, status, createdAt, transferRequestId);
        }

        @Override
        public String toString() {
            return "Identity{" +
                    "serviceId=" + serviceId +
                    ", status=" + status +
                    ", createdAt=" + createdAt +
                    ", transferRequestId='" + transferRequestId + '\'' +
                    '}';
        }

    }

    public static final class Builder {

        private TenantId tenantId;
        private Long serviceId;
        private TransferRequestStatus status;
        private Instant createdAt;
        private String transferRequestId;

        private Builder() {
        }

        private Builder(TransferRequestByServiceModel value) {
            this.tenantId = value.tenantId;
            this.serviceId = value.serviceId;
            this.status = value.status;
            this.createdAt = value.createdAt;
            this.transferRequestId = value.transferRequestId;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder status(TransferRequestStatus status) {
            this.status = status;
            return this;
        }

        public Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder transferRequestId(String transferRequestId) {
            this.transferRequestId = transferRequestId;
            return this;
        }

        public TransferRequestByServiceModel build() {
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(serviceId, "ServiceId is required");
            Preconditions.checkNotNull(status, "Status is required");
            Preconditions.checkNotNull(createdAt, "CreatedAt is required");
            Preconditions.checkNotNull(transferRequestId, "TransferRequestId is required");
            return new TransferRequestByServiceModel(tenantId, serviceId, status, createdAt, transferRequestId);
        }

    }

}
