package ru.yandex.intranet.d.model.transfers;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Transfer request history.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferRequestHistoryModel {

    private final String id;
    private final TenantId tenantId;
    private final String transferRequestId;
    private final TransferRequestEventType type;
    private final Instant timestamp;
    private final String authorId;
    private final TransferRequestHistoryFields oldFields;
    private final TransferRequestHistoryFields newFields;
    private final TransferParameters oldParameters;
    private final TransferResponsible oldResponsible;
    private final TransferVotes oldVotes;
    private final TransferApplicationDetails oldApplicationDetails;
    private final TransferParameters newParameters;
    private final TransferResponsible newResponsible;
    private final TransferVotes newVotes;
    private final TransferApplicationDetails newApplicationDetails;
    private final long order;
    @Nullable
    private final LoanMeta oldLoanMeta;
    @Nullable
    private final LoanMeta newLoanMeta;

    @SuppressWarnings("ParameterNumber")
    public TransferRequestHistoryModel(String id,
                                       TenantId tenantId,
                                       String transferRequestId,
                                       TransferRequestEventType type,
                                       Instant timestamp,
                                       String authorId,
                                       TransferRequestHistoryFields oldFields,
                                       TransferRequestHistoryFields newFields,
                                       TransferParameters oldParameters,
                                       TransferResponsible oldResponsible,
                                       TransferVotes oldVotes,
                                       TransferApplicationDetails oldApplicationDetails,
                                       TransferParameters newParameters,
                                       TransferResponsible newResponsible,
                                       TransferVotes newVotes,
                                       TransferApplicationDetails newApplicationDetails,
                                       long order,
                                       @Nullable LoanMeta oldLoanMeta,
                                       @Nullable LoanMeta newLoanMeta) {
        this.id = id;
        this.tenantId = tenantId;
        this.transferRequestId = transferRequestId;
        this.type = type;
        this.timestamp = timestamp;
        this.authorId = authorId;
        this.oldFields = oldFields;
        this.newFields = newFields;
        this.oldParameters = oldParameters;
        this.oldResponsible = oldResponsible;
        this.oldVotes = oldVotes;
        this.oldApplicationDetails = oldApplicationDetails;
        this.newParameters = newParameters;
        this.newResponsible = newResponsible;
        this.newVotes = newVotes;
        this.newApplicationDetails = newApplicationDetails;
        this.order = order;
        this.oldLoanMeta = oldLoanMeta;
        this.newLoanMeta = newLoanMeta;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferRequestHistoryModel value) {
        return new Builder(value);
    }

    public Builder copyBuilder() {
        return new Builder(this);
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public String getTransferRequestId() {
        return transferRequestId;
    }

    public TransferRequestEventType getType() {
        return type;
    }

    public Instant getTimestamp() {
        return timestamp;
    }

    public Optional<String> getAuthorId() {
        return Optional.ofNullable(authorId);
    }

    public Optional<TransferRequestHistoryFields> getOldFields() {
        return Optional.ofNullable(oldFields);
    }

    public Optional<TransferRequestHistoryFields> getNewFields() {
        return Optional.ofNullable(newFields);
    }

    public Optional<TransferParameters> getOldParameters() {
        return Optional.ofNullable(oldParameters);
    }

    public Optional<TransferResponsible> getOldResponsible() {
        return Optional.ofNullable(oldResponsible);
    }

    public Optional<TransferVotes> getOldVotes() {
        return Optional.ofNullable(oldVotes);
    }

    public Optional<TransferApplicationDetails> getOldApplicationDetails() {
        return Optional.ofNullable(oldApplicationDetails);
    }

    public Optional<TransferParameters> getNewParameters() {
        return Optional.ofNullable(newParameters);
    }

    public Optional<TransferResponsible> getNewResponsible() {
        return Optional.ofNullable(newResponsible);
    }

    public Optional<TransferVotes> getNewVotes() {
        return Optional.ofNullable(newVotes);
    }

    public Optional<TransferApplicationDetails> getNewApplicationDetails() {
        return Optional.ofNullable(newApplicationDetails);
    }

    public long getOrder() {
        return order;
    }

    public Optional<LoanMeta> getOldLoanMeta() {
        return Optional.ofNullable(oldLoanMeta);
    }

    public Optional<LoanMeta> getNewLoanMeta() {
        return Optional.ofNullable(newLoanMeta);
    }

    public Identity getIdentity() {
        return new Identity(id, transferRequestId, timestamp);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestHistoryModel that = (TransferRequestHistoryModel) o;
        return order == that.order &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(transferRequestId, that.transferRequestId) &&
                type == that.type &&
                Objects.equals(timestamp, that.timestamp) &&
                Objects.equals(authorId, that.authorId) &&
                Objects.equals(oldFields, that.oldFields) &&
                Objects.equals(newFields, that.newFields) &&
                Objects.equals(oldParameters, that.oldParameters) &&
                Objects.equals(oldResponsible, that.oldResponsible) &&
                Objects.equals(oldVotes, that.oldVotes) &&
                Objects.equals(oldApplicationDetails, that.oldApplicationDetails) &&
                Objects.equals(newParameters, that.newParameters) &&
                Objects.equals(newResponsible, that.newResponsible) &&
                Objects.equals(newVotes, that.newVotes) &&
                Objects.equals(newApplicationDetails, that.newApplicationDetails) &&
                Objects.equals(oldLoanMeta, that.oldLoanMeta) &&
                Objects.equals(newLoanMeta, that.newLoanMeta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, transferRequestId, type, timestamp, authorId, oldFields, newFields,
                oldParameters, oldResponsible, oldVotes, oldApplicationDetails, newParameters, newResponsible,
                newVotes, newApplicationDetails, order, oldLoanMeta, newLoanMeta);
    }

    @Override
    public String toString() {
        return "TransferRequestHistoryModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", transferRequestId='" + transferRequestId + '\'' +
                ", type=" + type +
                ", timestamp=" + timestamp +
                ", authorId='" + authorId + '\'' +
                ", oldFields=" + oldFields +
                ", newFields=" + newFields +
                ", oldParameters=" + oldParameters +
                ", oldResponsible=" + oldResponsible +
                ", oldVotes=" + oldVotes +
                ", oldApplicationDetails=" + oldApplicationDetails +
                ", newParameters=" + newParameters +
                ", newResponsible=" + newResponsible +
                ", newVotes=" + newVotes +
                ", newApplicationDetails=" + newApplicationDetails +
                ", order=" + order +
                ", oldLoanMeta=" + oldLoanMeta +
                ", newLoanMeta=" + newLoanMeta +
                '}';
    }

    public static final class Identity {

        private final String id;
        private final String transferRequestId;
        private final Instant timestamp;

        public Identity(String id, String transferRequestId, Instant timestamp) {
            this.id = id;
            this.transferRequestId = transferRequestId;
            this.timestamp = timestamp;
        }

        public String getId() {
            return id;
        }

        public String getTransferRequestId() {
            return transferRequestId;
        }

        public Instant getTimestamp() {
            return timestamp;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Identity identity = (Identity) o;
            return Objects.equals(id, identity.id) &&
                    Objects.equals(transferRequestId, identity.transferRequestId) &&
                    Objects.equals(timestamp, identity.timestamp);
        }

        @Override
        public int hashCode() {
            return Objects.hash(id, transferRequestId, timestamp);
        }

        @Override
        public String toString() {
            return "Identity{" +
                    "id='" + id + '\'' +
                    ", transferRequestId='" + transferRequestId + '\'' +
                    ", timestamp=" + timestamp +
                    '}';
        }

    }

    public static final class Builder {

        private String id;
        private TenantId tenantId;
        private String transferRequestId;
        private TransferRequestEventType type;
        private Instant timestamp;
        private String authorId;
        private TransferRequestHistoryFields oldFields;
        private TransferRequestHistoryFields newFields;
        private TransferParameters oldParameters;
        private TransferResponsible oldResponsible;
        private TransferVotes oldVotes;
        private TransferApplicationDetails oldApplicationDetails;
        private TransferParameters newParameters;
        private TransferResponsible newResponsible;
        private TransferVotes newVotes;
        private TransferApplicationDetails newApplicationDetails;
        private Long order;
        @Nullable
        private LoanMeta oldLoanMeta;
        @Nullable
        private LoanMeta newLoanMeta;

        private Builder() {
        }

        private Builder(TransferRequestHistoryModel value) {
            this.id = value.id;
            this.tenantId = value.tenantId;
            this.transferRequestId = value.transferRequestId;
            this.type = value.type;
            this.timestamp = value.timestamp;
            this.authorId = value.authorId;
            this.oldFields = value.oldFields;
            this.newFields = value.newFields;
            this.oldParameters = value.oldParameters;
            this.oldResponsible = value.oldResponsible;
            this.oldVotes = value.oldVotes;
            this.oldApplicationDetails = value.oldApplicationDetails;
            this.newParameters = value.newParameters;
            this.newResponsible = value.newResponsible;
            this.newVotes = value.newVotes;
            this.newApplicationDetails = value.newApplicationDetails;
            this.order = value.order;
            this.oldLoanMeta = value.oldLoanMeta;
            this.newLoanMeta = value.newLoanMeta;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder transferRequestId(String transferRequestId) {
            this.transferRequestId = transferRequestId;
            return this;
        }

        public Builder type(TransferRequestEventType type) {
            this.type = type;
            return this;
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Builder authorId(String authorId) {
            this.authorId = authorId;
            return this;
        }

        public Builder oldFields(TransferRequestHistoryFields oldFields) {
            this.oldFields = oldFields;
            return this;
        }

        public Builder newFields(TransferRequestHistoryFields newFields) {
            this.newFields = newFields;
            return this;
        }

        public Builder oldParameters(TransferParameters oldParameters) {
            this.oldParameters = oldParameters;
            return this;
        }

        public Builder oldResponsible(TransferResponsible oldResponsible) {
            this.oldResponsible = oldResponsible;
            return this;
        }

        public Builder oldVotes(TransferVotes oldVotes) {
            this.oldVotes = oldVotes;
            return this;
        }

        public Builder oldApplicationDetails(TransferApplicationDetails oldApplicationDetails) {
            this.oldApplicationDetails = oldApplicationDetails;
            return this;
        }

        public Builder newParameters(TransferParameters newParameters) {
            this.newParameters = newParameters;
            return this;
        }

        public Builder newResponsible(TransferResponsible newResponsible) {
            this.newResponsible = newResponsible;
            return this;
        }

        public Builder newVotes(TransferVotes newVotes) {
            this.newVotes = newVotes;
            return this;
        }

        public Builder newApplicationDetails(TransferApplicationDetails newApplicationDetails) {
            this.newApplicationDetails = newApplicationDetails;
            return this;
        }

        public Builder order(long order) {
            this.order = order;
            return this;
        }

        public Builder oldLoanMeta(@Nullable LoanMeta oldLoanMeta) {
            this.oldLoanMeta = oldLoanMeta;
            return this;
        }

        public Builder newLoanMeta(@Nullable LoanMeta newLoanMeta) {
            this.newLoanMeta = newLoanMeta;
            return this;
        }

        public TransferRequestHistoryModel build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(transferRequestId, "TransferRequestId is required");
            Preconditions.checkNotNull(type, "Type is required");
            Preconditions.checkNotNull(timestamp, "Timestamp is required");
            Preconditions.checkNotNull(order, "Order is required");
            return new TransferRequestHistoryModel(id, tenantId, transferRequestId, type, timestamp, authorId,
                    oldFields, newFields, oldParameters, oldResponsible, oldVotes, oldApplicationDetails,
                    newParameters, newResponsible, newVotes, newApplicationDetails, order, oldLoanMeta, newLoanMeta);
        }

    }

}
