package ru.yandex.intranet.d.model.transfers;

import java.time.Instant;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Transfer request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class TransferRequestModel {

    private final String id;
    private final TenantId tenantId;
    private final long version;
    private final String summary;
    private final String description;
    private final String trackerIssueKey;
    private final TransferRequestType type;
    private final TransferRequestStatus status;
    private final String createdBy;
    private final String updatedBy;
    private final Instant createdAt;
    private final Instant updatedAt;
    private final Instant appliedAt;
    private final TransferParameters parameters;
    private final TransferResponsible responsible;
    private final TransferVotes votes;
    private final TransferApplicationDetails applicationDetails;
    private final long nextHistoryOrder;
    private final TransferNotified transferNotified;
    @Nullable
    private final LoanMeta loanMeta;
    @Nullable
    private final TransferRequestSubtype subtype;

    @SuppressWarnings("ParameterNumber")
    public TransferRequestModel(String id,
                                TenantId tenantId,
                                long version,
                                String summary,
                                String description,
                                String trackerIssueKey,
                                TransferRequestType type,
                                TransferRequestStatus status,
                                String createdBy,
                                String updatedBy,
                                Instant createdAt,
                                Instant updatedAt,
                                Instant appliedAt,
                                TransferParameters parameters,
                                TransferResponsible responsible,
                                TransferVotes votes,
                                TransferApplicationDetails applicationDetails,
                                long nextHistoryOrder,
                                TransferNotified transferNotified,
                                @Nullable LoanMeta loanMeta,
                                @Nullable TransferRequestSubtype subtype) {
        this.id = id;
        this.tenantId = tenantId;
        this.version = version;
        this.summary = summary;
        this.description = description;
        this.trackerIssueKey = trackerIssueKey;
        this.type = type;
        this.status = status;
        this.createdBy = createdBy;
        this.updatedBy = updatedBy;
        this.createdAt = createdAt;
        this.updatedAt = updatedAt;
        this.appliedAt = appliedAt;
        this.parameters = parameters;
        this.responsible = responsible;
        this.votes = votes;
        this.applicationDetails = applicationDetails;
        this.nextHistoryOrder = nextHistoryOrder;
        this.transferNotified = transferNotified;
        this.loanMeta = loanMeta;
        this.subtype = subtype;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferRequestModel value) {
        return new Builder(value);
    }

    public Builder copyBuilder() {
        return new Builder(this);
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public long getVersion() {
        return version;
    }

    public Optional<String> getSummary() {
        return Optional.ofNullable(summary);
    }

    public Optional<String> getDescription() {
        return Optional.ofNullable(description);
    }

    public Optional<String> getTrackerIssueKey() {
        return Optional.ofNullable(trackerIssueKey);
    }

    public TransferRequestType getType() {
        return type;
    }

    public TransferRequestStatus getStatus() {
        return status;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public Optional<String> getUpdatedBy() {
        return Optional.ofNullable(updatedBy);
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public Optional<Instant> getUpdatedAt() {
        return Optional.ofNullable(updatedAt);
    }

    public Optional<Instant> getAppliedAt() {
        return Optional.ofNullable(appliedAt);
    }

    public TransferParameters getParameters() {
        return parameters;
    }

    public TransferResponsible getResponsible() {
        return responsible;
    }

    public TransferVotes getVotes() {
        return votes;
    }

    public Optional<TransferApplicationDetails> getApplicationDetails() {
        return Optional.ofNullable(applicationDetails);
    }

    public long getNextHistoryOrder() {
        return nextHistoryOrder;
    }

    public Optional<TransferNotified> getTransferNotified() {
        return Optional.ofNullable(transferNotified);
    }

    public Optional<LoanMeta> getLoanMeta() {
        return Optional.ofNullable(loanMeta);
    }

    public Optional<TransferRequestSubtype> getSubtype() {
        return Optional.ofNullable(subtype);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferRequestModel that = (TransferRequestModel) o;
        return version == that.version &&
                nextHistoryOrder == that.nextHistoryOrder &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(summary, that.summary) &&
                Objects.equals(description, that.description) &&
                Objects.equals(trackerIssueKey, that.trackerIssueKey) &&
                type == that.type &&
                subtype == that.subtype &&
                status == that.status &&
                Objects.equals(createdBy, that.createdBy) &&
                Objects.equals(updatedBy, that.updatedBy) &&
                Objects.equals(createdAt, that.createdAt) &&
                Objects.equals(updatedAt, that.updatedAt) &&
                Objects.equals(appliedAt, that.appliedAt) &&
                Objects.equals(parameters, that.parameters) &&
                Objects.equals(responsible, that.responsible) &&
                Objects.equals(votes, that.votes) &&
                Objects.equals(applicationDetails, that.applicationDetails) &&
                Objects.equals(transferNotified, that.transferNotified) &&
                Objects.equals(loanMeta, that.loanMeta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, version, summary, description, trackerIssueKey, type, status, createdBy,
                updatedBy, createdAt, updatedAt, appliedAt, parameters, responsible, votes, applicationDetails,
                nextHistoryOrder, transferNotified, loanMeta, subtype);
    }

    @Override
    public String toString() {
        return "TransferRequestModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", version=" + version +
                ", summary='" + summary + '\'' +
                ", description='" + description + '\'' +
                ", trackerIssueKey='" + trackerIssueKey + '\'' +
                ", type=" + type +
                ", subtype=" + subtype +
                ", status=" + status +
                ", createdBy='" + createdBy + '\'' +
                ", updatedBy='" + updatedBy + '\'' +
                ", createdAt=" + createdAt +
                ", updatedAt=" + updatedAt +
                ", appliedAt=" + appliedAt +
                ", parameters=" + parameters +
                ", responsible=" + responsible +
                ", votes=" + votes +
                ", applicationDetails=" + applicationDetails +
                ", nextHistoryOrder=" + nextHistoryOrder +
                ", transferNotified=" + transferNotified +
                ", loanMeta=" + loanMeta +
                '}';
    }

    public static final class Builder {

        private String id;
        private TenantId tenantId;
        private Long version;
        private String summary;
        private String description;
        private String trackerIssueKey;
        private TransferRequestType type;
        private TransferRequestSubtype subtype;
        private TransferRequestStatus status;
        private String createdBy;
        private String updatedBy;
        private Instant createdAt;
        private Instant updatedAt;
        private Instant appliedAt;
        private TransferParameters parameters;
        private TransferResponsible responsible;
        private TransferVotes votes;
        private TransferApplicationDetails applicationDetails;
        private Long nextHistoryOrder;
        private TransferNotified transferNotified;
        @Nullable
        private LoanMeta loanMeta;

        private Builder() {
        }

        private Builder(TransferRequestModel value) {
            this.id = value.id;
            this.tenantId = value.tenantId;
            this.version = value.version;
            this.summary = value.summary;
            this.description = value.description;
            this.trackerIssueKey = value.trackerIssueKey;
            this.type = value.type;
            this.subtype = value.subtype;
            this.status = value.status;
            this.createdBy = value.createdBy;
            this.updatedBy = value.updatedBy;
            this.createdAt = value.createdAt;
            this.updatedAt = value.updatedAt;
            this.appliedAt = value.appliedAt;
            this.parameters = value.parameters;
            this.responsible = value.responsible;
            this.votes = value.votes;
            this.applicationDetails = value.applicationDetails;
            this.nextHistoryOrder = value.nextHistoryOrder;
            this.transferNotified = value.transferNotified;
            this.loanMeta = value.loanMeta;
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder version(long version) {
            this.version = version;
            return this;
        }

        public Builder summary(String summary) {
            this.summary = summary;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public Builder trackerIssueKey(String trackerIssueKey) {
            this.trackerIssueKey = trackerIssueKey;
            return this;
        }

        public Builder type(TransferRequestType type) {
            this.type = type;
            return this;
        }

        public Builder subtype(TransferRequestSubtype subtype) {
            this.subtype = subtype;
            return this;
        }

        public Builder status(TransferRequestStatus status) {
            this.status = status;
            return this;
        }

        public Builder createdBy(String createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public Builder updatedBy(String updatedBy) {
            this.updatedBy = updatedBy;
            return this;
        }

        public Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public Builder appliedAt(Instant appliedAt) {
            this.appliedAt = appliedAt;
            return this;
        }

        public Builder parameters(TransferParameters parameters) {
            this.parameters = parameters;
            return this;
        }

        public Builder responsible(TransferResponsible responsible) {
            this.responsible = responsible;
            return this;
        }

        public Builder votes(TransferVotes votes) {
            this.votes = votes;
            return this;
        }

        public Builder applicationDetails(TransferApplicationDetails applicationDetails) {
            this.applicationDetails = applicationDetails;
            return this;
        }

        public Builder nextHistoryOrder(long nextHistoryOrder) {
            this.nextHistoryOrder = nextHistoryOrder;
            return this;
        }

        public Builder transferNotified(TransferNotified transferNotified) {
            this.transferNotified = transferNotified;
            return this;
        }

        public Builder loanMeta(@Nullable LoanMeta loanMeta) {
            this.loanMeta = loanMeta;
            return this;
        }

        public boolean hasChanges(TransferRequestModel value) {
            if (!Objects.equals(summary, value.summary)) {
                return true;
            }
            if (!Objects.equals(description, value.description)) {
                return true;
            }
            if (!Objects.equals(trackerIssueKey, value.trackerIssueKey)) {
                return true;
            }
            if (!Objects.equals(status, value.status)) {
                return true;
            }
            if (!Objects.equals(updatedBy, value.updatedBy)) {
                return true;
            }
            if (!Objects.equals(updatedAt, value.updatedAt)) {
                return true;
            }
            if (!Objects.equals(appliedAt, value.appliedAt)) {
                return true;
            }
            if (!Objects.equals(parameters, value.parameters)) {
                return true;
            }
            if (!Objects.equals(responsible, value.responsible)) {
                return true;
            }
            if (!Objects.equals(votes, value.votes)) {
                return true;
            }
            if (!Objects.equals(applicationDetails, value.applicationDetails)) {
                return true;
            }
            if (!Objects.equals(loanMeta, value.loanMeta)) {
                return true;
            }
            if (!Objects.equals(subtype, value.subtype)) {
                return true;
            }
            return false;
        }

        public TransferRequestModel build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(tenantId, "TenantId is required");
            Preconditions.checkNotNull(version, "Version is required");
            Preconditions.checkNotNull(type, "Type is required");
            Preconditions.checkNotNull(status, "Status is required");
            Preconditions.checkNotNull(createdBy, "CreatedBy is required");
            Preconditions.checkNotNull(createdAt, "CreatedAt is required");
            Preconditions.checkNotNull(parameters, "Parameters is required");
            Preconditions.checkNotNull(responsible, "Responsible is required");
            Preconditions.checkNotNull(votes, "Votes is required");
            Preconditions.checkNotNull(nextHistoryOrder, "NextHistoryOrder is required");
            return new TransferRequestModel(id, tenantId, version, summary, description, trackerIssueKey, type,
                    status, createdBy, updatedBy, createdAt, updatedAt, appliedAt, parameters, responsible, votes,
                    applicationDetails, nextHistoryOrder, transferNotified, loanMeta, subtype);
        }

    }

}
