package ru.yandex.intranet.d.model.transfers;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;

/**
 * Transfer responsible.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferResponsible {

    private final Set<FoldersResponsible> responsible;
    private final Set<ProviderResponsible> providerResponsible;
    private final ReserveResponsibleModel reserveResponsibleModel;

    @JsonCreator
    public TransferResponsible(Set<FoldersResponsible> responsible,
                               Set<ProviderResponsible> providerResponsible,
                               ReserveResponsibleModel reserveResponsibleModel) {
        this.responsible = responsible;
        this.providerResponsible = providerResponsible != null ? providerResponsible : Set.of();
        this.reserveResponsibleModel = reserveResponsibleModel;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferResponsible value) {
        return new Builder(value);
    }

    public Set<FoldersResponsible> getResponsible() {
        return responsible;
    }

    public Set<ProviderResponsible> getProviderResponsible() {
        return providerResponsible;
    }

    public Optional<ReserveResponsibleModel> getReserveResponsibleModel() {
        return Optional.ofNullable(reserveResponsibleModel);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferResponsible that = (TransferResponsible) o;
        return Objects.equals(responsible, that.responsible) && Objects.equals(providerResponsible,
                that.providerResponsible) && Objects.equals(reserveResponsibleModel, that.reserveResponsibleModel);
    }

    @Override
    public int hashCode() {
        return Objects.hash(responsible, providerResponsible, reserveResponsibleModel);
    }

    @Override
    public String toString() {
        return "TransferResponsible{" +
                "responsible=" + responsible +
                ", providerResponsible=" + providerResponsible +
                ", reserveResponsibleModel=" + reserveResponsibleModel +
                '}';
    }

    public static final class Builder {

        private final Set<FoldersResponsible> responsible = new HashSet<>();
        private final Set<ProviderResponsible> providerResponsible = new HashSet<>();
        private ReserveResponsibleModel reserveResponsibleModel;

        private Builder() {
        }

        private Builder(TransferResponsible value) {
            this.responsible.addAll(value.responsible);
            this.providerResponsible.addAll(value.providerResponsible);
            this.reserveResponsibleModel = value.reserveResponsibleModel;
        }

        public Builder addResponsible(FoldersResponsible foldersResponsible) {
            this.responsible.add(foldersResponsible);
            return this;
        }

        public Builder addResponsibles(FoldersResponsible... foldersResponsibles) {
            this.responsible.addAll(Arrays.asList(foldersResponsibles));
            return this;
        }

        public Builder addResponsibles(Collection<? extends FoldersResponsible> foldersResponsibles) {
            this.responsible.addAll(foldersResponsibles);
            return this;
        }

        public Builder addProviderResponsible(ProviderResponsible providerResponsible) {
            this.providerResponsible.add(providerResponsible);
            return this;
        }

        public Builder addProviderResponsibles(ProviderResponsible... providerResponsibles) {
            this.providerResponsible.addAll(Arrays.asList(providerResponsibles));
            return this;
        }

        public Builder addProviderResponsibles(Collection<? extends ProviderResponsible> providerResponsibles) {
            this.providerResponsible.addAll(providerResponsibles);
            return this;
        }

        public Builder reserveResponsibleModel(ReserveResponsibleModel reserveResponsibleModel) {
            this.reserveResponsibleModel = reserveResponsibleModel;
            return this;
        }

        public TransferResponsible build() {
            return new TransferResponsible(responsible, providerResponsible, reserveResponsibleModel);
        }

    }

}
