package ru.yandex.intranet.d.model.transfers;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

/**
 * Transfer vote.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class TransferVote {

    private final String userId;
    private final Set<Long> serviceIds;
    private final Set<String> folderIds;
    private final Instant timestamp;
    private final VoteType type;
    private final Set<String> providerIds;

    @JsonCreator
    public TransferVote(@JsonProperty("userId") String userId,
                        @JsonProperty("serviceIds") Set<Long> serviceIds,
                        @JsonProperty("folderIds") Set<String> folderIds,
                        @JsonProperty("timestamp") Instant timestamp,
                        @JsonProperty("type") VoteType type,
                        @JsonProperty("providerIds") Set<String> providerIds) {
        this.userId = userId;
        this.serviceIds = serviceIds;
        this.folderIds = folderIds;
        this.timestamp = timestamp;
        this.type = type;
        this.providerIds = providerIds != null ? providerIds : Set.of();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(TransferVote value) {
        return new Builder(value);
    }

    public String getUserId() {
        return userId;
    }

    public Set<Long> getServiceIds() {
        return serviceIds;
    }

    public Set<String> getFolderIds() {
        return folderIds;
    }

    public Instant getTimestamp() {
        return timestamp;
    }

    public VoteType getType() {
        return type;
    }

    public Set<String> getProviderIds() {
        return providerIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TransferVote that = (TransferVote) o;
        return Objects.equals(userId, that.userId) &&
                Objects.equals(serviceIds, that.serviceIds) &&
                Objects.equals(folderIds, that.folderIds) &&
                Objects.equals(timestamp, that.timestamp) &&
                type == that.type &&
                Objects.equals(providerIds, that.providerIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(userId, serviceIds, folderIds, timestamp, type, providerIds);
    }

    @Override
    public String toString() {
        return "TransferVote{" +
                "userId='" + userId + '\'' +
                ", serviceIds=" + serviceIds +
                ", folderIds=" + folderIds +
                ", timestamp=" + timestamp +
                ", type=" + type +
                ", providerIds=" + providerIds +
                '}';
    }

    public static final class Builder {

        private final Set<Long> serviceIds = new HashSet<>();
        private final Set<String> folderIds = new HashSet<>();
        private final Set<String> providerIds = new HashSet<>();

        private String userId;
        private Instant timestamp;
        private VoteType type;

        private Builder() {
        }

        private Builder(TransferVote value) {
            this.userId = value.userId;
            this.serviceIds.addAll(value.serviceIds);
            this.folderIds.addAll(value.folderIds);
            this.timestamp = value.timestamp;
            this.type = value.type;
            this.providerIds.addAll(value.providerIds);
        }

        public Builder userId(String userId) {
            this.userId = userId;
            return this;
        }

        public Builder addServiceId(long id) {
            this.serviceIds.add(id);
            return this;
        }

        public Builder addServiceIds(Long... ids) {
            this.serviceIds.addAll(Arrays.asList(ids));
            return this;
        }

        public Builder addServiceIds(Collection<? extends Long> ids) {
            this.serviceIds.addAll(ids);
            return this;
        }

        public Builder addFolderId(String id) {
            this.folderIds.add(id);
            return this;
        }

        public Builder addFolderIds(String... ids) {
            this.folderIds.addAll(Arrays.asList(ids));
            return this;
        }

        public Builder addFolderIds(Collection<? extends String> ids) {
            this.folderIds.addAll(ids);
            return this;
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return this;
        }

        public Builder type(VoteType type) {
            this.type = type;
            return this;
        }

        public Builder addProviderId(String id) {
            this.providerIds.add(id);
            return this;
        }

        public Builder addProviderIds(String... ids) {
            this.providerIds.addAll(Arrays.asList(ids));
            return this;
        }

        public Builder addProviderIds(Collection<? extends String> ids) {
            this.providerIds.addAll(ids);
            return this;
        }

        public TransferVote build() {
            Preconditions.checkNotNull(userId, "UserId is required");
            Preconditions.checkNotNull(timestamp, "Timestamp is required");
            Preconditions.checkNotNull(type, "Type is required");
            return new TransferVote(userId, serviceIds, folderIds, timestamp, type, providerIds);
        }

    }

}
