package ru.yandex.intranet.d.model.units;

import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.TenantId;

/**
 * Units ensemble.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@JsonIgnoreProperties(ignoreUnknown = true)
public final class UnitsEnsembleModel {

    private final String id;
    private final TenantId tenantId;
    private final long version;
    private final String nameEn;
    private final String nameRu;
    private final String descriptionEn;
    private final String descriptionRu;
    private final boolean fractionsAllowed;
    private final boolean deleted;
    private final Set<UnitModel> units;
    private final String key;
    private final Map<String, UnitModel> unitsById;
    private final Map<String, UnitModel> unitsByKey;

    @JsonCreator
    @SuppressWarnings("ParameterNumber")
    public UnitsEnsembleModel(@JsonProperty("id") String id,
                              @JsonProperty("tenantId") TenantId tenantId,
                              @JsonProperty("version") long version,
                              @JsonProperty("nameEn") String nameEn,
                              @JsonProperty("nameRu") String nameRu,
                              @JsonProperty("descriptionEn") String descriptionEn,
                              @JsonProperty("descriptionRu") String descriptionRu,
                              @JsonProperty("fractionsAllowed") boolean fractionsAllowed,
                              @JsonProperty("deleted") boolean deleted,
                              @JsonProperty("units") Set<UnitModel> units,
                              @JsonProperty("key") String key) {
        this.id = id;
        this.tenantId = tenantId;
        this.version = version;
        this.nameEn = nameEn;
        this.nameRu = nameRu;
        this.descriptionEn = descriptionEn;
        this.descriptionRu = descriptionRu;
        this.fractionsAllowed = fractionsAllowed;
        this.deleted = deleted;
        this.units = units;
        this.key = key;
        this.unitsById = units.stream().collect(Collectors.toMap(UnitModel::getId, u -> u, (l, r) -> l));
        this.unitsByKey = units.stream().collect(Collectors.toMap(UnitModel::getKey, u -> u, (l, r) -> l));
    }

    public static Builder builder() {
        return new Builder();
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public long getVersion() {
        return version;
    }

    public String getNameEn() {
        return nameEn;
    }

    public String getNameRu() {
        return nameRu;
    }

    public String getDescriptionEn() {
        return descriptionEn;
    }

    public String getDescriptionRu() {
        return descriptionRu;
    }

    public boolean isFractionsAllowed() {
        return fractionsAllowed;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public Set<UnitModel> getUnits() {
        return units;
    }

    public String getKey() {
        return key;
    }

    @JsonIgnore
    public Optional<UnitModel> unitById(String id) {
        return Optional.ofNullable(unitsById.get(id));
    }

    @JsonIgnore
    public Optional<UnitModel> unitByKey(String key) {
        return Optional.ofNullable(unitsByKey.get(key));
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UnitsEnsembleModel that = (UnitsEnsembleModel) o;
        return version == that.version &&
                fractionsAllowed == that.fractionsAllowed &&
                deleted == that.deleted &&
                Objects.equals(id, that.id) &&
                Objects.equals(tenantId, that.tenantId) &&
                Objects.equals(nameEn, that.nameEn) &&
                Objects.equals(nameRu, that.nameRu) &&
                Objects.equals(descriptionEn, that.descriptionEn) &&
                Objects.equals(descriptionRu, that.descriptionRu) &&
                Objects.equals(units, that.units) &&
                Objects.equals(key, that.key);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu,
                fractionsAllowed, deleted, units, key);
    }

    @Override
    public String toString() {
        return "UnitsEnsembleModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", version=" + version +
                ", nameEn='" + nameEn + '\'' +
                ", nameRu='" + nameRu + '\'' +
                ", descriptionEn='" + descriptionEn + '\'' +
                ", descriptionRu='" + descriptionRu + '\'' +
                ", fractionsAllowed=" + fractionsAllowed +
                ", deleted=" + deleted +
                ", units=" + units +
                ", key='" + key + '\'' +
                '}';
    }

    public static final class Builder {

        private final Set<UnitModel> units = new HashSet<>();

        private String id;
        private TenantId tenantId;
        private Long version;
        private String nameEn;
        private String nameRu;
        private String descriptionEn;
        private String descriptionRu;
        private Boolean fractionsAllowed;
        private Boolean deleted;
        private String key;

        private Builder() {
        }

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder version(long version) {
            this.version = version;
            return this;
        }

        public Builder nameEn(String nameEn) {
            this.nameEn = nameEn;
            return this;
        }

        public Builder nameRu(String nameRu) {
            this.nameRu = nameRu;
            return this;
        }

        public Builder descriptionEn(String descriptionEn) {
            this.descriptionEn = descriptionEn;
            return this;
        }

        public Builder descriptionRu(String descriptionRu) {
            this.descriptionRu = descriptionRu;
            return this;
        }

        public Builder fractionsAllowed(boolean fractionsAllowed) {
            this.fractionsAllowed = fractionsAllowed;
            return this;
        }

        public Builder deleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder addUnit(UnitModel unit) {
            this.units.add(unit);
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public boolean hasChanges(UnitsEnsembleModel unitsEnsemble) {
            if (!Objects.equals(id, unitsEnsemble.getId())) {
                return true;
            }
            if (!Objects.equals(tenantId, unitsEnsemble.getTenantId())) {
                return true;
            }
            if (!Objects.equals(nameEn, unitsEnsemble.getNameEn())) {
                return true;
            }
            if (!Objects.equals(nameRu, unitsEnsemble.getNameRu())) {
                return true;
            }
            if (!Objects.equals(descriptionEn, unitsEnsemble.getDescriptionEn())) {
                return true;
            }
            if (!Objects.equals(descriptionRu, unitsEnsemble.getDescriptionRu())) {
                return true;
            }
            if (!Objects.equals(fractionsAllowed, unitsEnsemble.isFractionsAllowed())) {
                return true;
            }
            if (!Objects.equals(deleted, unitsEnsemble.isDeleted())) {
                return true;
            }
            if (!Objects.equals(units, unitsEnsemble.getUnits())) {
                return true;
            }
            if (!Objects.equals(key, unitsEnsemble.getKey())) {
                return true;
            }
            return false;
        }

        public UnitsEnsembleModel build() {
            Preconditions.checkNotNull(id, "Id is required");
            Preconditions.checkNotNull(tenantId, "Tenant id is required");
            Preconditions.checkNotNull(version, "Version is required");
            Preconditions.checkNotNull(nameEn, "NameEn is required");
            Preconditions.checkNotNull(nameRu, "NameRu is required");
            Preconditions.checkNotNull(descriptionEn, "DescriptionEn is required");
            Preconditions.checkNotNull(descriptionRu, "DescriptionRu is required");
            Preconditions.checkNotNull(fractionsAllowed, "FractionsAllowed is required");
            Preconditions.checkNotNull(deleted, "Deleted is required");
            Preconditions.checkNotNull(key, "Key is required");
            return new UnitsEnsembleModel(id, tenantId, version, nameEn, nameRu, descriptionEn, descriptionRu,
                    fractionsAllowed, deleted, units, key);
        }

    }

}
