package ru.yandex.intranet.d.model.users;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonIgnore;

import ru.yandex.intranet.d.model.TenantId;

/**
 * User.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class UserModel {

    private final String id;
    private final TenantId tenantId;
    private final String passportUid;
    private final String passportLogin;
    private final Long staffId;
    private final Boolean staffDismissed;
    private final Boolean staffRobot;
    private final StaffAffiliation staffAffiliation;
    private final Map<UserServiceRoles, Set<Long>> roles;
    private final String firstNameEn;
    private final String firstNameRu;
    private final String lastNameEn;
    private final String lastNameRu;
    private final boolean dAdmin;
    private final boolean deleted;
    private final String gender;
    private final String workEmail;
    private final String langUi;
    private final String timeZone;

    private UserModel(Builder builder) {
        this.id = builder.id;
        this.tenantId = builder.tenantId;
        this.passportUid = builder.passportUid;
        this.passportLogin = builder.passportLogin;
        this.staffId = builder.staffId;
        this.staffDismissed = builder.staffDismissed;
        this.staffRobot = builder.staffRobot;
        this.staffAffiliation = builder.staffAffiliation;
        this.roles = Map.copyOf(builder.roles);
        this.firstNameEn = builder.firstNameEn;
        this.firstNameRu = builder.firstNameRu;
        this.lastNameEn = builder.lastNameEn;
        this.lastNameRu = builder.lastNameRu;
        this.dAdmin = builder.dAdmin;
        this.deleted = builder.deleted;
        this.gender = builder.gender;
        this.workEmail = builder.workEmail;
        this.langUi = builder.langUi;
        this.timeZone = builder.timeZone;
    }

    public String getId() {
        return id;
    }

    public TenantId getTenantId() {
        return tenantId;
    }

    public Optional<String> getPassportUid() {
        return Optional.ofNullable(passportUid);
    }

    public Optional<String> getPassportLogin() {
        return Optional.ofNullable(passportLogin);
    }

    public Optional<Long> getStaffId() {
        return Optional.ofNullable(staffId);
    }

    public Optional<Boolean> getStaffDismissed() {
        return Optional.ofNullable(staffDismissed);
    }

    public Optional<Boolean> getStaffRobot() {
        return Optional.ofNullable(staffRobot);
    }

    public Optional<StaffAffiliation> getStaffAffiliation() {
        return Optional.ofNullable(staffAffiliation);
    }

    public Map<UserServiceRoles, Set<Long>> getRoles() {
        return roles;
    }

    public String getFirstNameEn() {
        return firstNameEn;
    }

    public String getFirstNameRu() {
        return firstNameRu;
    }

    public String getLastNameEn() {
        return lastNameEn;
    }

    public String getLastNameRu() {
        return lastNameRu;
    }

    public boolean getDAdmin() {
        return dAdmin;
    }

    public boolean isDeleted() {
        return deleted;
    }

    public String getGender() {
        return gender;
    }

    public Optional<String> getWorkEmail() {
        return Optional.ofNullable(workEmail);
    }

    public Optional<String> getLangUi() {
        return Optional.ofNullable(langUi);
    }

    public Optional<String> getTimeZone() {
        return Optional.ofNullable(timeZone);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        UserModel userModel = (UserModel) o;
        return dAdmin == userModel.dAdmin &&
                deleted == userModel.deleted &&
                Objects.equals(id, userModel.id) &&
                Objects.equals(tenantId, userModel.tenantId) &&
                Objects.equals(passportUid, userModel.passportUid) &&
                Objects.equals(passportLogin, userModel.passportLogin) &&
                Objects.equals(staffId, userModel.staffId) &&
                Objects.equals(staffDismissed, userModel.staffDismissed) &&
                Objects.equals(staffRobot, userModel.staffRobot) &&
                staffAffiliation == userModel.staffAffiliation &&
                Objects.equals(roles, userModel.roles) &&
                Objects.equals(gender, userModel.gender) &&
                Objects.equals(firstNameEn, userModel.firstNameEn) &&
                Objects.equals(firstNameRu, userModel.firstNameRu) &&
                Objects.equals(lastNameEn, userModel.lastNameEn) &&
                Objects.equals(lastNameRu, userModel.lastNameRu) &&
                Objects.equals(workEmail, userModel.workEmail) &&
                Objects.equals(langUi, userModel.langUi) &&
                Objects.equals(timeZone, userModel.timeZone);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, tenantId, passportUid, passportLogin, staffId, staffDismissed, staffRobot,
                staffAffiliation, roles, firstNameEn, firstNameRu, lastNameEn, lastNameRu, dAdmin, deleted, gender,
                workEmail, langUi, timeZone);
    }

    @Override
    public String toString() {
        return "UserModel{" +
                "id='" + id + '\'' +
                ", tenantId=" + tenantId +
                ", passportUid='" + passportUid + '\'' +
                ", passportLogin='" + passportLogin + '\'' +
                ", staffId=" + staffId +
                ", staffDismissed=" + staffDismissed +
                ", staffRobot=" + staffRobot +
                ", staffAffiliation=" + staffAffiliation +
                ", roles=" + roles +
                ", firstNameEn='" + firstNameEn + '\'' +
                ", firstNameRu='" + firstNameRu + '\'' +
                ", lastNameEn='" + lastNameEn + '\'' +
                ", lastNameRu='" + lastNameRu + '\'' +
                ", dAdmin=" + dAdmin +
                ", deleted=" + deleted +
                ", gender='" + gender + '\'' +
                ", workEmail='" + workEmail + '\'' +
                ", langUi='" + langUi + '\'' +
                ", timeZone='" + timeZone + '\'' +
                '}';
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonIgnore
    public Builder copyBuilder() {
        return builder()
                .id(id)
                .tenantId(tenantId)
                .passportUid(passportUid)
                .passportLogin(passportLogin)
                .staffId(staffId)
                .staffDismissed(staffDismissed)
                .staffRobot(staffRobot)
                .staffAffiliation(staffAffiliation)
                .roles(roles)
                .gender(gender)
                .firstNameEn(firstNameEn)
                .firstNameRu(firstNameRu)
                .lastNameEn(lastNameEn)
                .lastNameRu(lastNameRu)
                .dAdmin(dAdmin)
                .deleted(deleted)
                .workEmail(workEmail)
                .langUi(langUi)
                .timeZone(timeZone);
    }


    public static class Builder {
        private String id;
        private TenantId tenantId;
        private String passportUid;
        private String passportLogin;
        private Long staffId;
        private Boolean staffDismissed;
        private Boolean staffRobot;
        private StaffAffiliation staffAffiliation;
        private Map<UserServiceRoles, Set<Long>> roles;
        private String firstNameEn;
        private String firstNameRu;
        private String lastNameEn;
        private String lastNameRu;
        private boolean dAdmin;
        private boolean deleted;
        private String gender;
        private String workEmail;
        private String langUi;
        private String timeZone;

        public Builder id(String id) {
            this.id = id;
            return this;
        }

        public Builder tenantId(TenantId tenantId) {
            this.tenantId = tenantId;
            return this;
        }

        public Builder passportUid(String passportUid) {
            this.passportUid = passportUid;
            return this;
        }

        public Builder passportLogin(String passportLogin) {
            this.passportLogin = passportLogin;
            return this;
        }

        public Builder staffId(Long staffId) {
            this.staffId = staffId;
            return this;
        }

        public Builder staffDismissed(Boolean staffDismissed) {
            this.staffDismissed = staffDismissed;
            return this;
        }

        public Builder staffRobot(Boolean staffRobot) {
            this.staffRobot = staffRobot;
            return this;
        }

        public Builder staffAffiliation(StaffAffiliation staffAffiliation) {
            this.staffAffiliation = staffAffiliation;
            return this;
        }

        public Builder roles(Map<UserServiceRoles, Set<Long>> roles) {
            this.roles = roles;
            return this;
        }

        public Builder firstNameEn(String firstNameEn) {
            this.firstNameEn = firstNameEn;
            return this;
        }

        public Builder firstNameRu(String firstNameRu) {
            this.firstNameRu = firstNameRu;
            return this;
        }

        public Builder lastNameEn(String lastNameEn) {
            this.lastNameEn = lastNameEn;
            return this;
        }

        public Builder lastNameRu(String lastNameRu) {
            this.lastNameRu = lastNameRu;
            return this;
        }

        public Builder dAdmin(boolean dAdmin) {
            this.dAdmin = dAdmin;
            return this;
        }

        public Builder deleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public Builder gender(String gender) {
            this.gender = gender;
            return this;
        }

        public Builder workEmail(String workEmail) {
            this.workEmail = workEmail;
            return this;
        }

        public Builder langUi(String langUi) {
            this.langUi = langUi;
            return this;
        }

        public Builder timeZone(String timeZone) {
            this.timeZone = timeZone;
            return this;
        }

        public UserModel build() {
            return new UserModel(this);
        }
    }

}
