package ru.yandex.intranet.d.services.accounts;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.CreateAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsByFolderRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ListAccountsResponseDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyResponseDto;
import ru.yandex.intranet.d.util.result.Result;

/**
 * AccountCreateConflictHelper
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Component
public class AccountCreateConflictHelper {
    private static final long REQUEST_PAGE_LIMIT = 1000L;
    private static final Logger LOG = LoggerFactory.getLogger(AccountCreateConflictHelper.class);

    private final ProvidersIntegrationService providersIntegrationService;

    public AccountCreateConflictHelper(ProvidersIntegrationService providersIntegrationService) {
        this.providersIntegrationService = providersIntegrationService;
    }

    private Mono<Result<List<AccountDto>>> getAccounts(ProviderModel provider, CreateAccountRequestDto requestDto,
                                                       Locale locale) {

        ListAccountsByFolderRequestDto firstPageRequest = new ListAccountsByFolderRequestDto(
                REQUEST_PAGE_LIMIT,
                null,
                true,
                false,
                requestDto.getFolderId(),
                requestDto.getAbcServiceId(),
                requestDto.getAccountsSpaceKey().orElse(null)
        );

        return providersIntegrationService.listAccountsByFolder(provider, firstPageRequest, locale)
                .flatMap(firstPageResult -> firstPageResult.applyMono(firstPageResponse ->
                        unwrapWithLogging(firstPageResponse, provider, "listAccountsByFolder first page")
                        .map(accounts -> withNextPages(provider, locale, firstPageRequest, accounts))
                        .orElse(Mono.just(List.of()))
                ));
    }

    @NotNull
    private Mono<List<AccountDto>> withNextPages(ProviderModel provider, Locale locale,
                                                 ListAccountsByFolderRequestDto firstPageRequest,
                                                 ListAccountsResponseDto accounts) {

        List<AccountDto> resultAccounts = new ArrayList<>(accounts.getAccounts().orElse(List.of()));

        if (accounts.getNextPageToken().isPresent()
                && !accounts.getNextPageToken().get().isEmpty()
                && accounts.getAccounts().isPresent()
                && !accounts.getAccounts().get().isEmpty()) {

            String nextPageToken = accounts.getNextPageToken().get();
            return requestNextPage(provider, firstPageRequest, nextPageToken, locale)
                    .expand(rs -> rs.match(t -> {
                                if (t.getT1().isEmpty() || t.getT2().isEmpty()) {
                                    return Mono.empty();
                                }
                                return requestNextPage(provider, firstPageRequest, t.getT2().get(), locale);
                            },
                            e -> Mono.empty()))
                    .reduce(resultAccounts, (allAccounts, pageResult) -> {
                        pageResult.doOnSuccess(page -> allAccounts.addAll(page.getT1()));
                        return allAccounts;
                    });
        }

        return Mono.just(resultAccounts);
    }

    private Mono<Result<Tuple2<List<AccountDto>, Optional<String>>>> requestNextPage(
            ProviderModel provider, ListAccountsByFolderRequestDto firstPageRequest,
            String nextPageToken, Locale locale) {

        ListAccountsByFolderRequestDto nextPageRequest = new ListAccountsByFolderRequestDto(
                firstPageRequest.getLimit(), nextPageToken, false, firstPageRequest.getIncludeDeleted().orElse(false),
                firstPageRequest.getFolderId(), firstPageRequest.getAbcServiceId(),
                firstPageRequest.getAccountsSpaceKey().orElse(null));

        return providersIntegrationService.listAccountsByFolder(provider, nextPageRequest, locale)
                .map(nextPageResult -> nextPageResult.apply(nextPageResponse ->
                        unwrapWithLogging(nextPageResponse, provider, "listAccountsByFolder next page")
                                .map(accounts -> Tuples.of(accounts.getAccounts().orElse(List.of()),
                                        accounts.getNextPageToken()))
                                .orElse(Tuples.of(List.of(), Optional.empty()))
                ));
    }


    public Mono<Optional<AccountDto>> tryToFindCreatedAccount(ProviderModel provider,
                                                              CreateAccountRequestDto requestDto, Locale locale) {

        return getAccounts(provider, requestDto, locale)
                .map(r -> r.match(Function.identity(), (ec) -> {
                    LOG.warn("Failed to get conflicting accounts: {}", ec);
                    return List.of();
                })).map(accounts -> {
                    Optional<Set<SegmentKeyResponseDto>> responseSegments =
                            requestDto.getAccountsSpaceKey().flatMap(this::toResponseSegments);
                    List<AccountDto> matchedAccounts = accounts.stream().filter(a ->
                            isAccountEqual(provider, requestDto, responseSegments, a))
                            .collect(Collectors.toList());
                    if (matchedAccounts.size() == 1) {
                        return matchedAccounts.stream().findFirst();
                    }
                    return Optional.<AccountDto>empty();
                }).onErrorResume(e -> {
                    LOG.warn("Failed to get conflicting accounts", e);
                    return Mono.just(Optional.empty());
                });
    }

    private Optional<Set<SegmentKeyResponseDto>> toResponseSegments(AccountsSpaceKeyRequestDto requestDto) {
        return requestDto.getSegmentation().map(ss -> ss.stream().map(s ->
                new SegmentKeyResponseDto(s.getSegmentationKey(), s.getSegmentKey()))
                .collect(Collectors.toSet()));
    }

    private boolean isAccountEqual(
            ProviderModel provider,
            CreateAccountRequestDto requestDto,
            Optional<Set<SegmentKeyResponseDto>> responseSegments,
            AccountDto account
    ) {
        if (!provider.isMultipleAccountsPerFolder()) {
            return true;
        }

        Optional<HashSet<SegmentKeyResponseDto>> accountSegments =
                account.getAccountsSpaceKey().flatMap(AccountsSpaceKeyResponseDto::getSegmentation).map(HashSet::new);

        Optional<String> opId = account.getLastUpdate().flatMap(LastUpdateDto::getOperationId)
                .filter(StringUtils::isNoneEmpty);

        return requestDto.getKey().equals(account.getKey()) &&
                requestDto.getDisplayName().equals(account.getDisplayName().filter(StringUtils::isNoneEmpty)) &&
                responseSegments.equals(accountSegments) &&
                Optional.of(requestDto.getAuthor()).equals(account.getLastUpdate().flatMap(LastUpdateDto::getAuthor)) &&
                (opId.isEmpty() || requestDto.getOperationId().equals(opId));
    }

    private <T> Optional<T> unwrapWithLogging(Response<T> response,
                                              ProviderModel provider,
                                              String name) {

        return response.match(new Response.Cases<>() {
            @Override
            public Optional<T> success(T result, String requestId) {
                return Optional.of(result);
            }

            @Override
            public Optional<T> failure(Throwable error) {
                LOG.error("Failure on provider \"{}\", providerId: {}", name, provider.getId(), error);
                return Optional.empty();
            }

            @Override
            public Optional<T> error(ProviderError error, String requestId) {
                LOG.error("Failure on provider \"{}\", providerId: {}, requestId: {}", name, provider.getId(),
                        requestId);
                LOG.error("Failure on provider \"{}\", requestId: {}, error: {}", name, requestId, error);
                return Optional.empty();
            }
        });
    }
}
