package ru.yandex.intranet.d.services.accounts

import kotlinx.coroutines.reactor.awaitSingle
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.datasource.TxSession
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.elapsed
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel
import ru.yandex.intranet.d.model.accounts.OperationChangesModel
import ru.yandex.intranet.d.model.accounts.OperationErrorKind
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel
import ru.yandex.intranet.d.model.accounts.OperationSource
import ru.yandex.intranet.d.model.folders.AccountHistoryModel
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel
import ru.yandex.intranet.d.model.folders.FolderOperationType
import ru.yandex.intranet.d.model.folders.FolderType
import ru.yandex.intranet.d.model.folders.QuotasByAccount
import ru.yandex.intranet.d.model.folders.QuotasByResource
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel
import ru.yandex.intranet.d.services.accounts.model.AccountOperationResult
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService
import ru.yandex.intranet.d.services.resources.AccountsSpacesService
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.uniques.CreateAccountOperation
import ru.yandex.intranet.d.services.uniques.PutAccountOperation
import ru.yandex.intranet.d.services.uniques.RequestUniqueService
import ru.yandex.intranet.d.services.validators.AbcServiceValidator
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.CreateAccountExpandedAnswerDto
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeInputDto
import ru.yandex.intranet.d.web.model.accounts.CreateAccountDto
import ru.yandex.intranet.d.web.model.accounts.PutAccountDto
import ru.yandex.intranet.d.web.model.folders.FrontAccountInputDto
import ru.yandex.intranet.d.web.model.folders.FrontPutAccountDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.time.Instant
import java.util.*

private val logger = KotlinLogging.logger {}

/**
 * Account logic service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class AccountLogicService(private val securityManagerService: SecurityManagerService,
                          private val requestUniqueService: RequestUniqueService,
                          private val accountService: AccountService,
                          private val accountsSpacesService: AccountsSpacesService,
                          private val abcServiceValidator: AbcServiceValidator,
                          private val folderDao: FolderDao,
                          private val providersLoader: ProvidersLoader,
                          private val accountsSpacesDao: AccountsSpacesDao,
                          private val accountsQuotasOperationsDao: AccountsQuotasOperationsDao,
                          private val operationsInProgressDao: OperationsInProgressDao,
                          private val folderOperationLogDao: FolderOperationLogDao,
                          private val accountsQuotasDao: AccountsQuotasDao,
                          private val quotasDao: QuotasDao,
                          private val resourcesLoader: ResourcesLoader,
                          private val accountsSpacesLoader: AccountSpacesLoader,
                          private val tableClient: YdbTableClient,
                          private val operationsObservabilityService: OperationsObservabilityService,
                          private val reserveAccountsService: ReserveAccountsService,
                          private val accountsDao: AccountsDao,
                          @Qualifier("messageSource") private val messages: MessageSource) {

    fun createAccountMono(folderId: String,
                          request: CreateAccountDto,
                          currentUser: YaUserDetails,
                          locale: Locale,
                          idempotencyKey: String?): Mono<Result<AccountOperationResult>> {
        return mono { createAccount(folderId, request, currentUser, locale, idempotencyKey) }
    }

    suspend fun createAccount(folderId: String,
                              request: CreateAccountDto,
                              currentUser: YaUserDetails,
                              locale: Locale,
                              idempotencyKey: String?): Result<AccountOperationResult> {
        val input = FrontAccountInputDto(folderId, request.providerId.orElse(null),
            request.displayName.orElse(null), request.externalKey.orElse(null),
            request.accountsSpaceId.orElse(null), request.freeTier.orElse(null),
            request.reserveType.orElse(null))
        return createAccountImpl(input, currentUser, locale, true, idempotencyKey).apply{ createResult ->
            val operationId = createResult.operation.operationId
            createResult.account.map { a -> AccountOperationResult.success(a, operationId) }
                .orElseGet { AccountOperationResult.inProgress(operationId) }
        }
    }

    fun createAccountMono(accountInputDto: FrontAccountInputDto,
                          currentUser: YaUserDetails,
                          locale: Locale,
                          idempotencyKey: String?): Mono<Result<CreateAccountExpandedAnswerDto>> {
        return mono { createAccount(accountInputDto, currentUser, locale, idempotencyKey) }
    }

    suspend fun createAccount(accountInputDto: FrontAccountInputDto,
                              currentUser: YaUserDetails,
                              locale: Locale,
                              idempotencyKey: String?): Result<CreateAccountExpandedAnswerDto> {
        return createAccountImpl(accountInputDto, currentUser, locale, false, idempotencyKey)
            .apply { createResult -> createResult.response.orElseThrow() }
    }

    fun putAccountMono(folderId: String,
                       accountId: String,
                       update: PutAccountDto,
                       currentUser: YaUserDetails,
                       locale: Locale,
                       idempotencyKey: String?): Mono<Result<AccountOperationResult>> {
        return mono { putAccount(folderId, accountId, update, currentUser, locale, idempotencyKey) }
    }

    fun putAccountMono(accountId: String,
                       update: FrontPutAccountDto,
                       currentUser: YaUserDetails,
                       locale: Locale,
                       idempotencyKey: String?): Mono<Result<AccountOperationResult>> {
        return mono { putAccount(accountId, update, currentUser, locale, idempotencyKey) }
    }

    suspend fun putAccount(folderId: String,
                           accountId: String,
                           update: PutAccountDto,
                           currentUser: YaUserDetails,
                           locale: Locale,
                           idempotencyKey: String?): Result<AccountOperationResult> {
        return putAccountImpl(folderId, accountId, update, currentUser, locale, idempotencyKey, false)
    }

    suspend fun putAccount(accountId: String,
                           update: FrontPutAccountDto,
                           currentUser: YaUserDetails,
                           locale: Locale,
                           idempotencyKey: String?): Result<AccountOperationResult> {
        return putAccountImpl(null, accountId, PutAccountDto(update.reserveType, null), currentUser,
            locale, idempotencyKey, true)
    }

    private suspend fun createAccountImpl(accountInputDto: FrontAccountInputDto,
                                          currentUser: YaUserDetails,
                                          locale: Locale,
                                          inProgressIsOk: Boolean,
                                          idempotencyKey: String?): Result<AccountService.ResultHolder> = binding {
        val validatedUser = accountService.validateUser(currentUser, locale).bind()!!
        val tenantId = Tenants.getTenantId(validatedUser)
        securityManagerService.checkReadPermissions(validatedUser, locale).awaitSingle().bind()
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()
        val (existing, created) = dbSessionRetryable(tableClient) {
            val (existingData, createdData) = rwTxRetryable {
                val existingAccountOperation = meter({ requestUniqueService.checkCreateAccount(txSession,
                    checkedIdempotencyKey, currentUser) }, "New account, check idempotency key")
                if (existingAccountOperation != null) {
                    val folder = meter({ loadFolder(txSession, existingAccountOperation, tenantId) },
                        "New account, load folder")
                    securityManagerService.checkReadPermissions(folder, currentUser, locale, folder)
                        .awaitSingle().bind()
                    val provider = meter({ loadProvider(existingAccountOperation, tenantId) },
                        "New account, load provider")
                    val accountsSpace = meter({ loadAccountsSpace(txSession, existingAccountOperation, tenantId) },
                        "New account, load accounts space")
                    val preparedAnswer = if (existingAccountOperation.account != null) {
                        prepareAnswerForExistingAccount(txSession, existingAccountOperation.account)
                    } else {
                        null
                    }
                    Pair (ExistingAccountData(existingAccountOperation, preparedAnswer, provider, folder,
                        accountsSpace), null)
                } else {
                    accountService.validateAccountInputDto(accountInputDto, locale).bind()
                    val provider = meter({ validateProvider(txSession, accountInputDto, locale, tenantId) },
                        "New account, validate provider").bind()!!
                    val folder = meter({ validateFolder(txSession, accountInputDto, provider, currentUser, locale,
                        tenantId) }, "New account, validate folder").bind()!!
                    val service = meter({ validateService(txSession, folder, locale) },
                        "New account, validate service").bind()!!
                    val accountsSpace = meter({ validateAccountsSpace(txSession, accountInputDto, provider, locale,
                        tenantId) }, "New account, validate accounts space").bind()
                    val expandedAccountsSpaceO = if (accountsSpace != null) {
                        Optional.of(meter({ accountsSpacesService.expand(accountsSpace).awaitSingle() },
                            "New account, expand accounts space"))
                    } else {
                        Optional.empty()
                    }
                    val reserveType = AccountReserveTypeInputDto.toModel(accountInputDto.reserveType.orElse(null))
                    reserveAccountsService.validateReservePermission(reserveType, provider, currentUser, locale).bind()
                    reserveAccountsService.validateReserve(
                        txSession, reserveType, provider.tenantId, provider,
                        service.id, folder.id, accountsSpace?.id, locale
                    ).bind()
                    val createParams = AccountService.CreateParameters(provider, folder, expandedAccountsSpaceO,
                        validatedUser, reserveType)
                    val operation = insertOperationAndHistory(txSession, accountInputDto, createParams, folder,
                        validatedUser, checkedIdempotencyKey)
                    Pair (null, CreatedAccountData(operation, createParams, service.slug))
                }
            }!!
            Pair (existingData, createdData)
        }!!
        return if (created != null) {
            accountService.tryApplyOperation(created.operation, accountInputDto, created.parameters, locale,
                tenantId, inProgressIsOk, created.abcServiceSlug).awaitSingle()
        } else {
            if (existing!!.createOperation.account != null) {
                val createParams = AccountService.CreateParameters(existing.provider, existing.folder,
                    Optional.ofNullable(existing.accountsSpace), validatedUser,
                    existing.createOperation.account!!.reserveType.orElse(null))
                val result = meter({ accountService.toCreateAccountExpandedAnswer(existing.preparedAnswer, locale,
                    createParams, tenantId, existing.createOperation.operation).awaitSingle() },
                    "New account, expand answer")
                Result.success(result)
            } else {
                val op = existing.createOperation.operation
                val isError = op.requestStatus
                    .map { s -> s == AccountsQuotasOperationsModel.RequestStatus.ERROR }.orElse(false)
                val isConflict = op.errorKind.map { k -> k == OperationErrorKind.ALREADY_EXISTS }.orElse(false)
                if (isError) {
                    accountService.generateResultForFailedOperation(op, isConflict, locale) {
                        errorMessage -> if (isConflict) {
                            TypedError.conflict(errorMessage)
                        } else {
                            TypedError.badRequest(errorMessage)
                        }
                    }
                } else {
                    if (inProgressIsOk) {
                        Result.success(AccountService.ResultHolder.inProgress(op))
                    } else {
                        Result.failure(ErrorCollection.builder().addError(TypedError.badRequest(messages
                            .getMessage("errors.provider.account.creation", null, locale))).build())
                    }
                }
            }
        }
    }

    private suspend fun putAccountImpl(folderId: String?,
                                       accountId: String,
                                       update: PutAccountDto,
                                       currentUser: YaUserDetails,
                                       locale: Locale,
                                       idempotencyKey: String?,
                                       front: Boolean): Result<AccountOperationResult> = binding {
        val validatedUser = validateUser(currentUser, locale).bind()!!
        val tenantId = Tenants.getTenantId(validatedUser)
        securityManagerService.checkReadPermissions(validatedUser, locale).awaitSingle().bind()
        val checkedIdempotencyKey = requestUniqueService.validateIdempotencyKey(idempotencyKey, locale).bind()
        val (alreadyDoneEarlier, doneNow) = dbSessionRetryable(tableClient) {
            val (alreadyDoneEarlierData, doneNowData) = rwTxRetryable {
                val existingAccountOperation = meter({ requestUniqueService.checkPutAccount(txSession,
                    checkedIdempotencyKey, validatedUser) }, "Put account, check idempotency key")
                if (existingAccountOperation != null) {
                    val targetAccount = meter({ validateAccount(txSession, accountId, existingAccountOperation,
                        tenantId, locale) }, "Put account, validate account").bind()!!
                    val accountFolder = meter({ loadAccountFolder(txSession, targetAccount) },
                        "Put account, load folder")
                    validateReadPermissions(accountFolder, validatedUser, locale).bind()
                    validateExistingOperationMatch(existingAccountOperation, targetAccount, locale).bind()
                    Pair(existingAccountOperation, null)
                } else {
                    val targetAccount = meter({ validateAccount(txSession, accountId, tenantId, locale) },
                        "Put account, validate account").bind()!!
                    val accountFolder = meter({ loadAccountFolder(txSession, targetAccount) },
                        "Put account, load folder")
                    val accountProvider = meter({ loadAccountProvider(txSession, targetAccount) },
                        "Put account, load provider")
                    validatePutPermissions(accountFolder, accountProvider, validatedUser, locale).bind()
                    if (!front) {
                        validatePublicApiConstraints(targetAccount, accountFolder, folderId, update.version, locale).bind()
                    }
                    val (accountWithUpdate, updated) = meter({ validateUpdate(txSession, targetAccount, update,
                        accountProvider, accountFolder, validatedUser, locale) }, "Put account, validate update")
                        .bind()!!
                    val now = Instant.now()
                    if (updated) {
                        meter({ accountsDao.upsertOneRetryable(txSession, accountWithUpdate).awaitSingle() },
                            "Put account, upsert account")
                        meter({ reserveAccountsService.updateReserveAccount(txSession, targetAccount, accountWithUpdate) },
                            "Put account, update reserve account index")
                    }
                    val operation = preparePutAccountOperation(accountWithUpdate, accountFolder, now, validatedUser)
                    meter({ accountsQuotasOperationsDao.upsertOneRetryable(txSession, operation).awaitSingle() },
                        "Put account, add operation")
                    meter({ folderOperationLogDao.upsertOneRetryable(txSession,
                        preparePutAccountOpLog(targetAccount, accountWithUpdate, operation, validatedUser)).awaitSingle() },
                        "Put account, add history")
                    meter({ folderDao.upsertOneRetryable(txSession,
                        accountFolder.toBuilder().setNextOpLogOrder(accountFolder.nextOpLogOrder + 1L).build()).awaitSingle() },
                        "Put account, update folder")
                    if (idempotencyKey != null) {
                        meter({ requestUniqueService.addPutAccount(txSession, idempotencyKey, operation,
                            operation.createDateTime, currentUser) }, "Put account, add idempotency key")
                    }
                    Pair(null, PutAccountOperation(accountWithUpdate, operation))
                }
            }!!
            Pair (alreadyDoneEarlierData, doneNowData)
        }!!
        if (doneNow != null) {
            preparePutResponse(doneNow, locale)
        } else {
            preparePutResponse(alreadyDoneEarlier!!, locale)
        }
    }

    private fun validateUser(userDetails: YaUserDetails, locale: Locale): Result<YaUserDetails> {
        return userDetails.user
            .filter { it.passportLogin.isPresent && it.passportUid.isPresent }
            .map { Result.success(userDetails) }
            .orElseGet { Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                .getMessage("errors.user.cannot.put.account", null, locale)))
                .build())
            }
    }

    private suspend fun validateUpdate(txSession: YdbTxSession, account: AccountModel, update: PutAccountDto,
                                       provider: ProviderModel, folder: FolderModel, currentUser: YaUserDetails,
                                       locale: Locale): Result<Pair<AccountModel, Boolean>> = binding {
        if (update.reserveType != null && update.reserveType == AccountReserveTypeInputDto.UNKNOWN) {
            Result.failure(ErrorCollection.builder().addError("reserveType", TypedError.invalid(messages
                .getMessage("errors.invalid.account.reserve.type", null, locale))).build())
        } else {
            val newReserveType = AccountReserveTypeInputDto.toModel(update.reserveType)
            if (!Objects.equals(newReserveType, account.reserveType.orElse(null))) {
                reserveAccountsService.validateReserveUpdatePermission(account.reserveType.orElse(null), newReserveType,
                    provider, currentUser, locale).bind()
                reserveAccountsService.validateReserve(txSession, newReserveType, account.tenantId, provider,
                    folder.serviceId, folder.id, account.accountsSpacesId.orElse(null), locale).bind()
                Result.success(Pair(AccountModel.Builder(account)
                    .setReserveType(newReserveType)
                    .setVersion(account.version + 1L)
                    .build(), true))
            } else {
                Result.success(Pair(account, false))
            }
        }
    }

    private fun preparePutAccountOperation(updatedAccount: AccountModel, folder: FolderModel, now: Instant,
                                           currentUser: YaUserDetails): AccountsQuotasOperationsModel {
        return AccountsQuotasOperationsModel.builder()
            .setTenantId(updatedAccount.tenantId)
            .setOperationId(UUID.randomUUID().toString())
            .setLastRequestId(UUID.randomUUID().toString())
            .setCreateDateTime(now)
            .setOperationSource(OperationSource.USER)
            .setOperationType(AccountsQuotasOperationsModel.OperationType.PUT_ACCOUNT)
            .setAuthorUserId(currentUser.user.orElseThrow().id)
            .setAuthorUserUid(currentUser.uid.orElseThrow())
            .setProviderId(updatedAccount.providerId)
            .setAccountsSpaceId(updatedAccount.accountsSpacesId.orElse(null))
            .setUpdateDateTime(now)
            .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
            .setErrorMessage(null)
            .setFullErrorMessage(null)
            .setRequestedChanges(OperationChangesModel.builder()
                .accountId(updatedAccount.id)
                .accountPutParams(OperationChangesModel.AccountPutParams(
                    updatedAccount.reserveType.orElse(null)))
                .build())
            .setOrders(OperationOrdersModel.builder()
                .submitOrder(folder.nextOpLogOrder)
                .build())
            .setErrorKind(null)
            .setLogs(listOf())
            .build()
    }

    private fun preparePutAccountOpLog(oldAccount: AccountModel, newAccount: AccountModel,
                                       operation: AccountsQuotasOperationsModel,
                                       currentUser: YaUserDetails): FolderOperationLogModel {
        return FolderOperationLogModel.builder()
            .setTenantId(newAccount.tenantId)
            .setFolderId(newAccount.folderId)
            .setOperationDateTime(operation.createDateTime)
            .setId(UUID.randomUUID().toString())
            .setProviderRequestId(operation.lastRequestId.orElseThrow())
            .setOperationType(FolderOperationType.PUT_ACCOUNT)
            .setAuthorUserId(currentUser.user.orElseThrow().id)
            .setAuthorUserUid(currentUser.uid.orElseThrow())
            .setAuthorProviderId(null)
            .setSourceFolderOperationsLogId(null)
            .setDestinationFolderOperationsLogId(null)
            .setOldFolderFields(null)
            .setNewFolderFields(null)
            .setOldQuotas(QuotasByResource(emptyMap()))
            .setNewQuotas(QuotasByResource(emptyMap()))
            .setOldProvisions(QuotasByAccount(emptyMap()))
            .setNewProvisions(QuotasByAccount(emptyMap()))
            .setOldBalance(QuotasByResource(emptyMap()))
            .setNewBalance(QuotasByResource(emptyMap()))
            .setActuallyAppliedProvisions(null)
            .setOldAccounts(AccountsHistoryModel(mapOf(oldAccount.id to AccountHistoryModel.builder()
                .version(oldAccount.version)
                .reserveType(oldAccount.reserveType.orElse(null))
                .build())))
            .setNewAccounts(AccountsHistoryModel(mapOf(newAccount.id to AccountHistoryModel.builder()
                .version(newAccount.version)
                .reserveType(newAccount.reserveType.orElse(null))
                .build())))
            .setAccountsQuotasOperationsId(operation.operationId)
            .setQuotasDemandsId(null)
            .setOperationPhase(null)
            .setCommentId(null)
            .setOrder(operation.orders.submitOrder)
            .setDeliveryMeta(null)
            .setTransferMeta(null)
            .build()
    }

    private fun validatePublicApiConstraints(account:AccountModel, folder: FolderModel, folderId: String?,
                                             version: Long?, locale: Locale): Result<Unit> {
        val errors = ErrorCollection.builder()
        if (version == null) {
            errors.addError("version", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (version != null && account.version != version) {
            errors.addError("version", TypedError.versionMismatch(messages
                .getMessage("errors.version.mismatch", null, locale)))
        }
        if (folderId == null) {
            errors.addError("folderId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (folderId != null && folder.id != folderId) {
            errors.addError("folderId", TypedError.notFound(messages
                .getMessage("errors.account.not.found.in.this.folder", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(Unit)
    }

    private fun preparePutResponse(accountOperation: PutAccountOperation,
                                   locale: Locale): Result<AccountOperationResult> {
        val op = accountOperation.operation
        val account = accountOperation.account
        val isError = op.requestStatus
            .map { s -> s == AccountsQuotasOperationsModel.RequestStatus.ERROR }.orElse(false)
        val isInProgress = op.requestStatus.isEmpty || op.requestStatus
            .map { s -> s == AccountsQuotasOperationsModel.RequestStatus.WAITING }.orElse(false)
        return if (isError) {
            val isConflict = op.errorKind.map { k -> k == OperationErrorKind.FAILED_PRECONDITION }.orElse(false)
            accountService.generateResultForFailedOperation(op, isConflict, locale) { errorMessage ->
                if (isConflict) {
                    TypedError.versionMismatch(errorMessage)
                } else {
                    TypedError.invalid(errorMessage)
                }
            }
        } else if (isInProgress) {
            Result.success(AccountOperationResult.inProgress(op.operationId))
        } else {
            Result.success(AccountOperationResult.success(account, op.operationId))
        }
    }

    private suspend fun validateAccount(txSession: YdbTxSession, accountId: String,
                                        existingAccountOperation: PutAccountOperation,
                                        tenantId: TenantId, locale: Locale): Result<AccountModel> {
        if (existingAccountOperation.account.id == accountId) {
            return Result.success(existingAccountOperation.account)
        }
        val accountO = accountsDao.getById(txSession, accountId, tenantId).awaitSingle()
        if (accountO.isEmpty) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.notFound(messages
                .getMessage("errors.account.not.found", null, locale))).build())
        }
        return Result.success(accountO.get())
    }

    private suspend fun validateAccount(txSession: YdbTxSession, accountId: String,
                                        tenantId: TenantId, locale: Locale): Result<AccountModel> {
        val accountO = accountsDao.getById(txSession, accountId, tenantId).awaitSingle()
        if (accountO.isEmpty || accountO.get().isDeleted) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.notFound(messages
                .getMessage("errors.account.not.found", null, locale))).build())
        }
        return Result.success(accountO.get())
    }

    private suspend fun loadAccountFolder(txSession: YdbTxSession, account: AccountModel): FolderModel {
        return folderDao.getById(txSession, account.folderId, account.tenantId).awaitSingle().orElseThrow()
    }

    private suspend fun loadAccountProvider(txSession: YdbTxSession, account: AccountModel): ProviderModel {
        return providersLoader.getProviderById(txSession, account.providerId, account.tenantId).awaitSingle().orElseThrow()
    }

    private suspend fun validateReadPermissions(folder: FolderModel, currentUser: YaUserDetails,
                                                locale: Locale): Result<Unit> = binding {
        securityManagerService.checkReadPermissions(folder, currentUser, locale, folder)
            .awaitSingle().apply { _ -> }.bind()
        securityManagerService.checkReadPermissions(folder.serviceId, currentUser, locale, folder)
            .awaitSingle().apply { _ -> }.bind()
        Result.success(Unit)
    }

    private suspend fun validatePutPermissions(folder: FolderModel, provider: ProviderModel, currentUser: YaUserDetails,
                                               locale: Locale): Result<Unit> = binding {
        securityManagerService.checkReadPermissions(folder, currentUser, locale, folder)
            .awaitSingle().apply { _ -> }.bind()
        securityManagerService.checkReadPermissions(folder.serviceId, currentUser, locale, folder)
            .awaitSingle().apply { _ -> }.bind()
        securityManagerService.checkProvisionAndAccountPermissions(folder.serviceId, provider, currentUser, locale)
            .awaitSingle().bind()
        Result.success(Unit)
    }

    private fun validateExistingOperationMatch(existingAccountOperation: PutAccountOperation,
                                               targetAccount: AccountModel, locale: Locale): Result<Unit> {
        if (existingAccountOperation.account.id != targetAccount.id) {
            return Result.failure(ErrorCollection.builder().addError("idempotencyKey", TypedError.invalid(messages
                .getMessage("errors.idempotency.key.mismatch", null, locale))).build())
        }
        return Result.success(Unit)
    }

    private suspend fun TxSession.loadProvider(existingAccountOperation: CreateAccountOperation,
                                               tenantId: TenantId): ProviderModel {
        val providerId = if (existingAccountOperation.account != null) {
            existingAccountOperation.account.providerId
        } else {
            existingAccountOperation.operation.providerId
        }
        return providersLoader.getProviderById(txSession, providerId, tenantId).awaitSingle().orElseThrow()
    }

    private suspend fun loadFolder(txSession: YdbTxSession,
                                   existingAccountOperation: CreateAccountOperation,
                                   tenantId: TenantId): FolderModel {
        val folderId = if (existingAccountOperation.account != null) {
            existingAccountOperation.account.folderId
        } else {
            existingAccountOperation.operation.requestedChanges.accountCreateParams.orElseThrow().folderId
        }
        return folderDao.getById(txSession, folderId, tenantId).awaitSingle().orElseThrow()
    }

    private suspend fun loadAccountsSpace(txSession: YdbTxSession,
                                          existingAccountOperation: CreateAccountOperation,
                                          tenantId: TenantId): ExpandedAccountsSpaces<AccountSpaceModel>? {
        val account = existingAccountOperation.account
        if (account != null) {
            val accountsSpaceIdO = account.accountsSpacesId
            return if (accountsSpaceIdO.isPresent) {
                val accountsSpace = accountsSpacesLoader.getAccountSpaces(txSession, tenantId, account.providerId,
                    accountsSpaceIdO.get()).awaitSingle().orElseThrow()[0]
                accountsSpacesService.expand(accountsSpace).awaitSingle()
            } else {
                null
            }
        } else {
            val operation = existingAccountOperation.operation
            val accountsSpaceIdO = operation.accountsSpaceId
            return if (accountsSpaceIdO.isPresent) {
                val accountsSpace = accountsSpacesLoader.getAccountSpaces(txSession, tenantId, operation.providerId,
                    accountsSpaceIdO.get()).awaitSingle().orElseThrow()[0]
                accountsSpacesService.expand(accountsSpace).awaitSingle()
            } else {
                null
            }
        }
    }

    private suspend fun prepareAnswerForExistingAccount(txSession: YdbTxSession,
                                                        account: AccountModel): AccountService.CreateAccountResult {
        val provisions = meter({ accountsQuotasDao.getAllByAccountIds(txSession, account.tenantId, setOf(account.id))
            .awaitSingle() }, "New account, load provisions")
        if (provisions.isEmpty()) {
            return AccountService.CreateAccountResult(account)
        }
        val resourceIds = provisions.map {p -> p.resourceId}.toSet()
        val quotas = meter({ quotasDao.getByProviderFoldersResources(txSession, account.tenantId,
            setOf(account.folderId), account.providerId, resourceIds).awaitSingle() }, "New account, load quotas")
        val resources = meter({ resourceIds.chunked(500) { p -> p.map { i -> Tuples.of(i, account.tenantId) } }
            .map { p -> resourcesLoader.getResourcesByIds(txSession, p).awaitSingle() }.flatten() },
            "New account, load resources")
        return AccountService.CreateAccountResult(account, provisions, quotas, quotas,
            resources.associateBy { k -> k.id })
    }

    private suspend fun insertOperationAndHistory(txSession: YdbTxSession,
                                                  accountInputDto: FrontAccountInputDto,
                                                  createParams: AccountService.CreateParameters,
                                                  folder: FolderModel,
                                                  currentUser: YaUserDetails,
                                                  idempotencyKey: String?): AccountService.Operation {
        val operation = meter({ accountsQuotasOperationsDao.upsertOneRetryable(txSession,
            accountService.getOperation(accountInputDto, createParams, folder)).awaitSingle() },
            "New account, add operation")
        operationsObservabilityService.observeOperationSubmitted(operation)
        val operationInProgress = meter({ operationsInProgressDao.upsertOneRetryable(txSession,
            accountService.getOperationInProgress(operation, createParams)).awaitSingle() },
            "New account, add operation in progress")
        meter({ folderOperationLogDao.upsertOneRetryable(txSession,
            accountService.getSubmitFolderLog(createParams, operation, accountInputDto)).awaitSingle() },
            "New account, add history")
        meter({ folderDao.upsertOneRetryable(txSession,
            folder.toBuilder().setNextOpLogOrder(folder.nextOpLogOrder + 1L).build()).awaitSingle() },
            "New account, update folder")
        if (idempotencyKey != null) {
            meter({ requestUniqueService.addCreateAccount(txSession, idempotencyKey, operation,
                operation.createDateTime, currentUser) }, "New account, add idempotency key")
        }
        return AccountService.Operation(operation, operationInProgress)
    }

    private suspend fun validateProvider(txSession: YdbTxSession,
                                         accountInputDto: FrontAccountInputDto,
                                         locale: Locale,
                                         tenantId: TenantId): Result<ProviderModel> {
        val providerO = providersLoader.getProviderById(txSession, accountInputDto.providerId, tenantId).awaitSingle()
        if (providerO.isEmpty || providerO.get().isDeleted) {
            return Result.failure(ErrorCollection.builder().addError("providerId", TypedError.notFound(messages
                .getMessage("errors.provider.not.found", null, locale))).build())
        }
        return accountService.validateProviderSettings(providerO.get(), accountInputDto, locale)
    }

    private suspend fun validateFolder(txSession: YdbTxSession,
                                       accountInputDto: FrontAccountInputDto,
                                       provider: ProviderModel,
                                       currentUser: YaUserDetails,
                                       locale: Locale,
                                       tenantId: TenantId): Result<FolderModel> = binding {
        val folderO = folderDao.getById(txSession, accountInputDto.folderId, tenantId).awaitSingle()
        if (folderO.isEmpty || folderO.get().isDeleted) {
            return Result.failure(ErrorCollection.builder().addError("folderId", TypedError.invalid(messages
                .getMessage("errors.folder.not.found", null, locale))).build())
        }
        val folder = folderO.get()
        if (folder.folderType == FolderType.PROVIDER_RESERVE) {
            return Result.failure(ErrorCollection.builder().addError("folderId", TypedError.invalid(messages
                .getMessage("errors.account.creation.in.reserve.folder.is.not.allowed", null, locale))).build())
        }
        securityManagerService.checkReadPermissions(folder.serviceId, currentUser, locale, folder)
            .awaitSingle().bind()
        securityManagerService.checkProvisionAndAccountPermissions(folder.serviceId, provider, currentUser, locale)
            .awaitSingle().bind()
        return Result.success(folder)
    }

    private suspend fun validateService(txSession: YdbTxSession,
                                        folder: FolderModel,
                                        locale: Locale): Result<ServiceWithStatesModel> {
        return abcServiceValidator.validateAbcService(Optional.of(folder.serviceId), locale, txSession, "folderId",
            AbcServiceValidator.ALLOWED_SERVICE_STATES, AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES, false)
            .awaitSingle().toResult()
    }

    private suspend fun validateAccountsSpace(txSession: YdbTxSession,
                                              accountInputDto: FrontAccountInputDto,
                                              provider: ProviderModel,
                                              locale: Locale,
                                              tenantId: TenantId): Result<AccountSpaceModel?> {
        if (accountInputDto.accountsSpaceId == null) {
            return Result.success(null)
        }
        val accountsSpaceO = accountsSpacesDao.getById(txSession, accountInputDto.accountsSpaceId, tenantId)
            .awaitSingle()
        if (accountsSpaceO.isEmpty || accountsSpaceO.get().isDeleted) {
            return Result.failure(ErrorCollection.builder().addError("accountsSpaceId", TypedError.invalid(messages
                .getMessage("errors.accounts.space.not.found", null, locale))).build())
        }
        val accountsSpace = accountsSpaceO.get()
        if (accountsSpace.providerId != provider.id) {
            return Result.failure(ErrorCollection.builder().addError("accountsSpaceId", TypedError.invalid(messages
                .getMessage("errors.wrong.provider.for.accounts.space", null, locale))).build())
        }
        if (accountsSpace.isReadOnly) {
            return Result.failure(ErrorCollection.builder().addError("accountsSpaceId", TypedError.invalid(messages
                .getMessage("errors.provider.account.space.is.read.only", null, locale))).build())
        }
        return Result.success(accountsSpace)
    }

    private inline fun <T> meter(statement: () -> T,
                                  label: String): T {
        return elapsed(statement)
                { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }

    private data class ExistingAccountData(val createOperation: CreateAccountOperation,
                                           val preparedAnswer: AccountService.CreateAccountResult?,
                                           val provider: ProviderModel,
                                           val folder: FolderModel,
                                           val accountsSpace: ExpandedAccountsSpaces<AccountSpaceModel>?)

    private data class CreatedAccountData(val operation: AccountService.Operation,
                                          val parameters: AccountService.CreateParameters,
                                          val abcServiceSlug: String)

}
