package ru.yandex.intranet.d.services.accounts;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.yandex.ydb.table.transaction.TransactionMode;
import io.grpc.Status;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuples;
import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasOperationsDao;
import ru.yandex.intranet.d.dao.accounts.OperationsInProgressDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.datasource.model.WithTxId;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.resources.ResourcesByKeysLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountReserveType;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasOperationsModel;
import ru.yandex.intranet.d.model.accounts.OperationChangesModel;
import ru.yandex.intranet.d.model.accounts.OperationErrorKind;
import ru.yandex.intranet.d.model.accounts.OperationInProgressModel;
import ru.yandex.intranet.d.model.accounts.OperationOrdersModel;
import ru.yandex.intranet.d.model.accounts.OperationSource;
import ru.yandex.intranet.d.model.folders.AccountHistoryModel;
import ru.yandex.intranet.d.model.folders.AccountsHistoryModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.folders.OperationPhase;
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel;
import ru.yandex.intranet.d.model.folders.ProvisionsByResource;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.AccountsSettingsModel;
import ru.yandex.intranet.d.model.providers.ProviderId;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.model.resources.types.ResourceTypeId;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.accounts.model.AccountOperationFailureMeta;
import ru.yandex.intranet.d.services.integration.providers.ProviderError;
import ru.yandex.intranet.d.services.integration.providers.ProvidersIntegrationService;
import ru.yandex.intranet.d.services.integration.providers.Response;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.AccountsSpaceKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.CreateAccountRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ErrorMessagesDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.LastUpdateDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ProvisionDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.ResourceComplexKey;
import ru.yandex.intranet.d.services.integration.providers.rest.model.SegmentKeyRequestDto;
import ru.yandex.intranet.d.services.integration.providers.rest.model.UserIdDto;
import ru.yandex.intranet.d.services.operations.OperationsObservabilityService;
import ru.yandex.intranet.d.services.quotas.ExpandedProviderBuilder;
import ru.yandex.intranet.d.services.quotas.ExternalAccountUrlFactory;
import ru.yandex.intranet.d.services.quotas.ProvisionOperationFailureMeta;
import ru.yandex.intranet.d.services.quotas.QuotaSums;
import ru.yandex.intranet.d.services.resources.ExpandedAccountsSpaces;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.AsyncMetrics;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.ResultTx;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.CreateAccountExpandedAnswerDto;
import ru.yandex.intranet.d.web.model.ResourceDto;
import ru.yandex.intranet.d.web.model.accounts.AccountReserveTypeInputDto;
import ru.yandex.intranet.d.web.model.folders.FrontAccountInputDto;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedAccount;
import ru.yandex.intranet.d.web.model.folders.front.ExpandedProvider;
import ru.yandex.intranet.d.web.model.folders.front.ProviderPermission;
import ru.yandex.intranet.d.web.model.folders.front.ResourceTypeDto;
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;
import ru.yandex.intranet.d.web.util.ModelDtoConverter;

import static java.util.function.Function.identity;
import static ru.yandex.intranet.d.dao.Tenants.withTenantId;
import static ru.yandex.intranet.d.web.util.ModelDtoConverter.providerDtoFromModel;
import static ru.yandex.intranet.d.web.util.ModelDtoConverter.resourceTypeDtoFromModel;

/**
 * Account service
 *
 * @author Denis Blokhin <denblo@yandex-team.ru>
 */
@Component
public class AccountService {
    private static final Logger LOG = LoggerFactory.getLogger(AccountService.class);

    private final ProvidersIntegrationService providersIntegrationService;
    private final AccountsDao accountsDao;
    private final YdbTableClient tableClient;
    private final MessageSource messages;
    private final SecurityManagerService securityManagerService;
    private final AccountsQuotasOperationsDao accountsQuotasOperationsDao;
    private final OperationsInProgressDao operationsInProgressDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final FolderDao folderDao;
    private final ResourcesDao resourcesDao;
    private final AccountCreateConflictHelper createConflictHelper;
    private final AccountsQuotasDao accountsQuotasDao;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;
    private final QuotasDao quotasDao;
    private final ResourcesByKeysLoader resourcesByKeysLoader;
    private final ResourceTypesLoader resourceTypesLoader;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final OperationsObservabilityService operationsObservabilityService;
    private final ReserveAccountsService reserveAccountsService;

    @SuppressWarnings("ParameterNumber")
    public AccountService(ProvidersIntegrationService providersIntegrationService,
                          AccountsDao accountsDao, YdbTableClient tableClient,
                          @Qualifier("messageSource") MessageSource messages,
                          SecurityManagerService securityManagerService,
                          AccountsQuotasOperationsDao accountsQuotasOperationsDao,
                          OperationsInProgressDao operationsInProgressDao,
                          FolderOperationLogDao folderOperationLogDao,
                          FolderDao folderDao, ResourcesDao resourcesDao,
                          AccountCreateConflictHelper createConflictHelper,
                          AccountsQuotasDao accountsQuotasDao,
                          UnitsEnsemblesLoader unitsEnsemblesLoader,
                          QuotasDao quotasDao,
                          ResourcesByKeysLoader resourcesByKeysLoader,
                          ResourceTypesLoader resourceTypesLoader,
                          ResourceSegmentationsLoader resourceSegmentationsLoader,
                          ResourceSegmentsLoader resourceSegmentsLoader,
                          OperationsObservabilityService operationsObservabilityService,
                          ReserveAccountsService reserveAccountsService) {
        this.providersIntegrationService = providersIntegrationService;
        this.accountsDao = accountsDao;
        this.tableClient = tableClient;
        this.messages = messages;
        this.securityManagerService = securityManagerService;
        this.accountsQuotasOperationsDao = accountsQuotasOperationsDao;
        this.operationsInProgressDao = operationsInProgressDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.folderDao = folderDao;
        this.resourcesDao = resourcesDao;
        this.createConflictHelper = createConflictHelper;
        this.accountsQuotasDao = accountsQuotasDao;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
        this.quotasDao = quotasDao;
        this.resourcesByKeysLoader = resourcesByKeysLoader;
        this.resourceTypesLoader = resourceTypesLoader;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.operationsObservabilityService = operationsObservabilityService;
        this.reserveAccountsService = reserveAccountsService;
    }

    public Mono<Result<ResultHolder>> tryApplyOperation(Operation operation,
                                                        FrontAccountInputDto accountInputDto,
                                                        CreateParameters params,
                                                        Locale locale,
                                                        TenantId tenantId,
                                                        boolean inProgressIsOk,
                                                        String abcServiceSlug) {
        return createAccountInProvider(accountInputDto, params, operation, locale, inProgressIsOk, abcServiceSlug)
                .flatMap(r -> r.andThenMono(accountDto -> accountDto.map(dto ->
                                doOperationApplication(dto, operation, params, tenantId, locale, inProgressIsOk))
                        .orElseGet(() -> Mono.just(Result.success(ResultHolder.inProgress(operation.operation))))))
                .map(r -> r.match(
                        Result::success,
                        e -> {
                            if (inProgressIsOk) {
                                // Add operation id into details
                                return Result.<ResultHolder>failure(ErrorCollection.builder().add(e)
                                        .addDetail("operationMeta", new AccountOperationFailureMeta(operation
                                                .operation.getOperationId())).build());
                            }
                            return Result.<ResultHolder>failure(e);
                        }))
                .onErrorResume(e -> {
                    if (inProgressIsOk) {
                        // Failed to process response, operation may be not complete,
                        // user expects operation id, error is just logged
                        LOG.warn("Failed to process CreateAccount result", e);
                        return Mono.just(Result.success(ResultHolder.inProgress(operation.operation)));
                    }
                    // Failed to process response, return error to user
                    return Mono.error(e);
                });
    }

    private Mono<Result<ResultHolder>> doOperationApplication(AccountDto dto, Operation operation,
                                                              CreateParameters params, TenantId tenantId,
                                                              Locale locale, boolean inProgressIsOk) {
        return createAccountModel(dto, params, operation, locale).map(v ->
                filterCreateAccountResult(v, inProgressIsOk)).flatMap(result -> result.andThenMono(accountO ->
                        accountO.map(account -> toCreateAccountExpandedAnswer(account, locale, params,
                                        tenantId, operation.operation).map(Result::success))
                                .orElseGet(() -> Mono.just(Result.success(ResultHolder
                                        .inProgress(operation.operation))))));
    }

    private Result<Optional<CreateAccountResult>> filterCreateAccountResult(Result<CreateAccountResult> input,
                                                                            boolean inProgressIsOk) {
        return input.match(r -> Result.success(Optional.of(r)), e -> {
            if (inProgressIsOk) {
                // Failed to process response, operation is not complete,
                // user expects operation id, error is just logged
                LOG.warn("Failed to process CreateAccount result: {}", e);
                return Result.success(Optional.empty());
            }
            // Failed to process response, return error to user
            return Result.failure(e);
        });
    }

    public Mono<ResultHolder> toCreateAccountExpandedAnswer(
            CreateAccountResult accountWithDefaults,
            Locale locale,
            CreateParameters params,
            TenantId tenantId,
            AccountsQuotasOperationsModel operation
    ) {
        AccountModel account = accountWithDefaults.account;
        List<AccountsQuotasModel> accountsQuotas = accountWithDefaults.accountsQuotas;
        List<QuotaModel> folderQuotas = accountWithDefaults.newFolderQuotas;
        Map<String, ResourceModel> defaultResourcesById = accountWithDefaults.resourcesById;

        if (accountsQuotas.isEmpty()) {
            return Mono.just(ResultHolder.success(new CreateAccountExpandedAnswerDto(
                    new ExpandedProvider(account.getProviderId(), List.of(), List.of(new ExpandedAccount(
                        ru.yandex.intranet.d.web.model.AccountDto.fromModel(account),
                        List.of(), null, null
                    )), Set.of()),
                    providerDtoFromModel(params.getProvider(), locale), List.of(), List.of(), List.of()
            ), operation, accountWithDefaults.account));
        }

        Set<String> unitsEnsembleIds = new HashSet<>();
        Set<String> resourceTypeIds = new HashSet<>();
        Set<String> segmentationIds = new HashSet<>();
        Set<String> segmentIds = new HashSet<>();
        for (ResourceModel resource : defaultResourcesById.values()) {
            unitsEnsembleIds.add(resource.getUnitsEnsembleId());
            resourceTypeIds.add(resource.getResourceTypeId());
            for (ResourceSegmentSettingsModel segment: resource.getSegments()) {
                segmentationIds.add(segment.getSegmentationId());
                segmentIds.add(segment.getSegmentId());
            }
        }
        return
            unitsEnsemblesLoader.getUnitsEnsemblesByIdsImmediate(withTenantId(unitsEnsembleIds, tenantId))
                    .flatMap(unitsEnsembles ->
            resourceTypesLoader.getResourceTypesByIdsImmediate(withTenantId(resourceTypeIds, tenantId))
                    .flatMap(resourceTypes ->
            resourceSegmentationsLoader.getResourceSegmentationsByIdsImmediate(withTenantId(segmentationIds, tenantId))
                    .flatMap(segmentations ->
            resourceSegmentsLoader.getResourceSegmentsByIdsImmediate(withTenantId(segmentIds, tenantId))
        .map(segments -> {
            Map<String, UnitsEnsembleModel> unitsEnsembleMap = unitsEnsembles.stream()
                    .collect(Collectors.toMap(UnitsEnsembleModel::getId, identity()));
            Map<String, ResourceSegmentationModel> segmentationsById = segmentations.stream()
                    .collect(Collectors.toMap(ResourceSegmentationModel::getId, identity()));
            Map<String, ResourceSegmentModel> segmentsById = segments.stream()
                    .collect(Collectors.toMap(ResourceSegmentModel::getId, identity()));

            Map<ResourceTypeId, List<QuotaSums>> quotasByResourceTypeId =
                    folderQuotas.stream().collect(Collectors.groupingBy(
                            quotaModel -> new ResourceTypeId(
                                    defaultResourcesById.get(quotaModel.getResourceId()).getResourceTypeId()
                            ),
                            Collectors.mapping(QuotaSums::from, Collectors.toList())
                    ));
            Map<String, QuotaModel> quotasByResourceId = folderQuotas.stream()
                    .collect(Collectors.toMap(QuotaModel::getResourceId, identity()));
            List<ResourceDto> resources = accountWithDefaults.resourcesById.values().stream().map(resourceModel ->
                            new ResourceDto(resourceModel, locale, unitsEnsembleMap, segmentationsById, segmentsById))
                    .collect(Collectors.toList());
            List<AccountsSpaceDto> accountsSpaces = List.of();
            if (params.accountSpace.isPresent()) {
                ExpandedAccountsSpaces<AccountSpaceModel> space = params.accountSpace.get();
                accountsSpaces = List.of(ModelDtoConverter.toDto(space, locale));
            }
            List<ResourceTypeDto> resourceTypeDtos = resourceTypes.stream().map(resourceType ->
                    resourceTypeDtoFromModel(resourceType, locale, unitsEnsembleMap)
            ).collect(Collectors.toList());
            ExternalAccountUrlFactory externalAccountUrlFactory =
                    params.provider.getAccountsSettings().getExternalAccountUrlTemplates() == null ? null :
                            new ExternalAccountUrlFactory(
                                    params.provider.getAccountsSettings().getExternalAccountUrlTemplates(),
                                    params.provider.getServiceId(),
                                    segmentationsById,
                                    segmentsById,
                                    accountsSpaces
                            );

            return ResultHolder.success(new CreateAccountExpandedAnswerDto(new ExpandedProviderBuilder(
                    locale, defaultResourcesById, unitsEnsembleMap, externalAccountUrlFactory)
                    .toExpandedProvider(
                            new ProviderId(account.getProviderId()),
                            quotasByResourceTypeId,
                            List.of(account),
                            Map.of(account.getId(), accountsQuotas),
                            quotasByResourceId,
                            // If account has been created, then user has both permissions for this provider
                            Set.of(ProviderPermission.CAN_UPDATE_PROVISION,
                                    ProviderPermission.CAN_MANAGE_ACCOUNT),
                            false
                    ),
                    providerDtoFromModel(params.provider, locale), //provider
                    resources, //resources
                    accountsSpaces, //accountsSpaces
                    resourceTypeDtos //resourceTypes
            ), operation, accountWithDefaults.account);
        }))));
    }

    public Result<Void> validateAccountInputDto(FrontAccountInputDto accountInputDto, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (accountInputDto.getAccountName() != null && accountInputDto.getAccountName().trim().isEmpty()) {
            errors.addError("accountName", TypedError.invalid(messages
                    .getMessage("errors.account.name.is.required", null, locale)));
        }
        if (accountInputDto.getFolderId() == null) {
            errors.addError("folderId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (accountInputDto.getProviderId() == null) {
            errors.addError("providerId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        }
        if (accountInputDto.getFolderId() != null && !Uuids.isValidUuid(accountInputDto.getFolderId())) {
            errors.addError("folderId", TypedError.invalid(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        }
        if (accountInputDto.getProviderId() != null && !Uuids.isValidUuid(accountInputDto.getProviderId())) {
            errors.addError("providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.not.found", null, locale)));
        }
        if (accountInputDto.getAccountsSpaceId() != null && !Uuids.isValidUuid(accountInputDto.getAccountsSpaceId())) {
            errors.addError("accountsSpaceId", TypedError.invalid(messages
                    .getMessage("errors.accounts.space.not.found", null, locale)));
        }
        if (accountInputDto.getFreeTier() != null && accountInputDto.getFreeTier()) {
            // TODO Remove this check after free tier implementation
            errors.addError("freeTier", TypedError.invalid(messages
                    .getMessage("errors.free.tier.not.supported.yet", null, locale)));
        }
        if (accountInputDto.getReserveType().isPresent()
                && AccountReserveTypeInputDto.UNKNOWN.equals(accountInputDto.getReserveType().get())) {
            errors.addError("reserveType", TypedError.invalid(messages
                    .getMessage("errors.invalid.account.reserve.type", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return  Result.success(null);
    }

    public Result<ProviderModel> validateProviderSettings(ProviderModel provider,
                                                           FrontAccountInputDto accountInputDto, Locale locale) {
        final ErrorCollection.Builder builder = ErrorCollection.builder();
        if (provider.isReadOnly()) {
            builder.addError("providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.is.read.only", null, locale)));
        }

        if (!provider.isManaged()) {
            builder.addError("providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.is.not.managed", null, locale)));
        }

        if (!provider.isAccountsSpacesSupported() && accountInputDto.getAccountsSpaceId() != null) {
            builder.addError("accountsSpaceId", TypedError.invalid(messages
                    .getMessage("errors.provider.not.support.accounts.spaces", null, locale)));
        }
        if (provider.isAccountsSpacesSupported() && StringUtils.isEmpty(accountInputDto.getAccountsSpaceId())) {
            builder.addError("accountsSpaceId", TypedError.invalid(messages
                    .getMessage("errors.provider.account.required.accounts.space", null, locale)));
        }
        final AccountsSettingsModel accountsSettings = provider.getAccountsSettings();
        if (!accountsSettings.isKeySupported() && accountInputDto.getAccountKey() != null) {
            builder.addError("accountKey", TypedError.invalid(messages
                    .getMessage("errors.provider.not.support.account.keys", null, locale)));
        }
        if (accountsSettings.isKeySupported() && StringUtils.isEmpty(accountInputDto.getAccountKey())) {
            builder.addError("accountKey", TypedError.invalid(messages
                    .getMessage("errors.provider.account.required.key", null, locale)));
        }
        if (accountInputDto.getAccountName() != null && !accountsSettings.isDisplayNameSupported()) {
            builder.addError("accountName", TypedError.invalid(messages
                    .getMessage("errors.provider.not.support.display.name", null, locale)));
        } else if (accountInputDto.getAccountName() == null && accountsSettings.isDisplayNameSupported()) {
            builder.addError("accountName", TypedError.invalid(messages
                    .getMessage("errors.account.name.is.required", null, locale)));
        }

        final ErrorCollection errorCollection = builder.build();
        if (errorCollection.hasAnyErrors()) {
            return Result.failure(errorCollection);
        }

        return Result.success(provider);
    }

    public Result<YaUserDetails> validateUser(YaUserDetails userDetails, Locale locale) {
        return userDetails.getUser()
                .filter(u -> u.getPassportLogin().isPresent() && u.getPassportUid().isPresent())
                .map(u -> Result.success(userDetails))
                .orElseGet(() -> Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                        .getMessage("errors.user.cannot.create.account", null, locale)))
                        .build()));
    }

    private Result<FolderModel> validateFolder(Optional<FolderModel> folderModelO, Locale locale) {
        if (folderModelO.isEmpty()) {
            return Result.failure(ErrorCollection.builder()
                    .addError("folderId", TypedError.badRequest(messages
                            .getMessage("errors.folder.not.found", null, locale)))
                    .build());
        }
        if (folderModelO.get().getFolderType().equals(FolderType.PROVIDER_RESERVE)) {
            return Result.failure(ErrorCollection.builder()
                    .addError("folderId", TypedError.badRequest(messages
                            .getMessage("errors.account.creation.in.reserve.folder.is.not.allowed", null, locale)))
                    .build());
        }
        return Result.success(folderModelO.get());
    }

    private Mono<Result<Optional<AccountDto>>> createAccountInProvider(FrontAccountInputDto inputDto,
                                                                       CreateParameters params,
                                                                       Operation history,
                                                                       Locale locale,
                                                                       boolean inProgressIsOk,
                                                                       String abcServiceSlug) {
        UserModel currentUserModel = params.userDetails.getUser().orElseThrow();
        CreateAccountRequestDto requestDto = new CreateAccountRequestDto(
                inputDto.getAccountKey(),
                inputDto.getAccountName(),
                params.folder.getId(),
                params.folder.getServiceId(),
                new UserIdDto(currentUserModel.getPassportUid().orElseThrow(),
                        currentUserModel.getPassportLogin().orElseThrow()),
                history.operation.getOperationId(),
                params.accountSpace.map(this::toSpaceRequestDto).orElse(null),
                inputDto.getFreeTier() != null ? inputDto.getFreeTier() : false,
                abcServiceSlug);
        // Do request to provider, log time spent
        return meter(providersIntegrationService.createAccount(params.provider, requestDto, locale),
                "New account, provider request")
                // Wrap result into Optional
                .map(Optional::of)
                // On exception fallback
                .onErrorResume(e -> {
                    operationsObservabilityService.observeOperationTransientFailure(history.operation);
                    if (inProgressIsOk) {
                        // Operation is unfinished, operation id is expected by a user, error is logged only
                        LOG.warn("CreateAccount request failed", e);
                        return Mono.just(Optional.empty());
                    }
                    // Actual error is expected by a user
                    return Mono.error(e);
                // No exception during provider request, process request result
                }).flatMap(o -> o.map(r -> r.match(
                        // There was an actual response from provider
                        response -> {
                            // Convert provider response to result, either successful or not
                            Result<Optional<AccountDto>> result = processAccountDtoResponse(response, locale)
                                    // Wrap AccountDto into optional
                                    .apply(Optional::of);
                            // Extract provider error, if any
                            Optional<ProviderError> providerErrorO = getProviderError(response);
                            if (providerErrorO.isEmpty()) {
                                // No provider error, either it is success or there was an exception during request
                                return result.match(
                                        // Success, return AccountDto, operation will be finished later in the flow
                                        a -> Mono.just(Result.success(a)),
                                        // There was some error but we have no error code
                                        e -> {
                                            operationsObservabilityService
                                                    .observeOperationTransientFailure(history.operation);
                                            if (inProgressIsOk) {
                                                // Operation is unfinished, operation id is expected by a user,
                                                // error is logged only
                                                LOG.warn("CreateAccount request failed {}", e);
                                                return Mono.just(Result.success(Optional.<AccountDto>empty()));
                                            }
                                            // Actual error is expected by a user
                                            return Mono.just(Result.<Optional<AccountDto>>failure(e));
                                        });
                            }
                            ProviderError providerError = providerErrorO.get();
                            Mono<Optional<AccountDto>> conflictingAccount = Mono.just(Optional.empty());
                            if (providerError.isConflict()) {
                                // Error code is Conflict, try to find conflicting account
                                // and check if it matches the account we are creating,
                                // empty means either there is no such account or we don't know
                                conflictingAccount = meter(createConflictHelper.tryToFindCreatedAccount(
                                        params.provider, requestDto, locale), "New account, provider read request");
                            }
                            return conflictingAccount.flatMap(accountO -> {
                                if (accountO.isPresent()) {
                                    // Matching conflicting account was found, return it instead
                                    return Mono.just(Result.success(accountO));
                                }
                                // Extract error message text
                                String errorMessage = getProviderErrorMessage(providerError, locale);
                                if (providerError.isRetryable()) {
                                    operationsObservabilityService
                                            .observeOperationTransientFailure(history.operation);
                                    // An operation is retryable, save error message
                                    return meter(updateOperationErrorMessage(history.operation, errorMessage),
                                            "New account, update operation")
                                            .then()
                                            .onErrorResume(e -> {
                                                // Ignore errors while saving error message, just log them
                                                LOG.warn("Failed to update account operation state", e);
                                                return Mono.empty();
                                            })
                                            .then(Mono.fromSupplier(() -> inProgressIsOk
                                                    // Operation is unfinished, operation id is expected by a user
                                                    ? Result.success(Optional.<AccountDto>empty())
                                                    // Return error from provider to user
                                                    : result));
                                }
                                boolean conflict = providerError.isConflict();
                                // An operation is non-retryable, finish the operation
                                return meter(markOperationAsFailed(history, errorMessage, locale, conflict),
                                        "New account, finish operation").flatMap(markResult -> markResult.match(
                                        // An operation is successfully finished, return provider error to user
                                        op -> Mono.just(result),
                                        // Unable to finish an operation
                                        e -> {
                                            LOG.warn("Failed to mark operation as failed: {}", e);
                                            return inProgressIsOk
                                                    // Operation is unfinished, operation id is expected by a user
                                                    ? Mono.just(Result.success(Optional.<AccountDto>empty()))
                                                    // Return error from provider to user
                                                    : Mono.just(result);
                                        })).onErrorResume(e -> {
                                            // Failed to finish an operation
                                            LOG.warn("Failed to mark operation as failed", e);
                                            return inProgressIsOk
                                                // Operation is unfinished, operation id is expected by a user
                                                ? Mono.just(Result.success(Optional.empty()))
                                                // Return error from provider to user
                                                : Mono.just(result);
                                        });
                            });
                        },
                        // Failed to send request to provider, probably TVM issue
                        errors -> {
                            operationsObservabilityService.observeOperationTransientFailure(history.operation);
                            if (inProgressIsOk) {
                                // Operation is unfinished, operation id is expected by a user, error is logged only
                                LOG.warn("Failed to send CreateAccount request: {}", errors);
                                return Mono.just(Result.success(Optional.<AccountDto>empty()));
                            }
                            // Actual error is expected by a user
                            return Mono.just(Result.<Optional<AccountDto>>failure(errors));
                        }
                // Operation is unfinished, operation id is expected by a user
                )).orElseGet(() -> Mono.just(Result.success(Optional.empty()))))
                // If there was any unexpected error
                .onErrorResume(e -> {
                    operationsObservabilityService.observeOperationTransientFailure(history.operation);
                    if (inProgressIsOk) {
                        // Unsure if an operation is finished, operation id is expected by a user, error is logged only
                        LOG.warn("Failed to process CreateAccount request", e);
                        return Mono.just(Result.success(Optional.empty()));
                    }
                    // Actual error is expected by a user
                    return Mono.error(e);
                });
    }


    private Mono<AccountsQuotasOperationsModel> updateOperationErrorMessage(AccountsQuotasOperationsModel operation,
                                                                            String errorMessage) {
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, ts ->
                        accountsQuotasOperationsDao.upsertOneRetryable(ts, prepareOperationWithMessage(operation,
                                errorMessage))
                ));
    }

    private AccountsQuotasOperationsModel prepareOperationWithMessage(AccountsQuotasOperationsModel operation,
                                                                      String errorMessage) {
        return new AccountsQuotasOperationsModel.Builder(operation)
                .setUpdateDateTime(Instant.now())
                .setErrorMessage(errorMessage)
                .build();
    }

    private Mono<Result<AccountsQuotasOperationsModel>> markOperationAsFailed(Operation history, String message,
                                                                              Locale locale, boolean conflict) {
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingCompResultTxRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                        ts -> accountsQuotasOperationsDao.getByIdStartTx(ts, history.operation.getOperationId(),
                                history.operation.getTenantId())
                                .map(tx -> ResultTx.success(tx.get(), tx.getTransactionId())),
                        (ts, accountOpO) -> accountOpO.map(op -> getFailedOperation(op, message, conflict))
                                .map(op -> accountsQuotasOperationsDao.upsertOneRetryable(ts, op)
                                                .then(removeOperationInProgress(ts, history.operationInProgress))
                                                .thenReturn(Result.success(op))
                                ).orElseGet(() -> Mono.just(Result.failure(ErrorCollection.builder()
                                        .addError(TypedError.notFound(messages.getMessage(
                                                "errors.operation.not.found", null, locale))).build()))
                                )
                                .map(r -> new WithTxId<>(r, ts.getId())),
                        (ts, b) -> ts.commitTransaction().thenReturn(new WithTxId<>(b, null))
                )).map(r -> r.andThen(identity()));
    }

    private Optional<ProviderError> getProviderError(Response<AccountDto> res) {
        return res.match(new Response.Cases<>() {
            @Override
            public Optional<ProviderError> success(AccountDto result, String requestId) {
                return Optional.empty();
            }

            @Override
            public Optional<ProviderError> failure(Throwable error) {
                return Optional.empty();
            }

            @Override
            public Optional<ProviderError> error(ProviderError error, String requestId) {
                return Optional.of(error);
            }
        });
    }

    public FolderOperationLogModel getSubmitFolderLog(CreateParameters params,
                                                       AccountsQuotasOperationsModel operation,
                                                       FrontAccountInputDto inputDto) {
        return FolderOperationLogModel.builder()
                .setTenantId(params.folder.getTenantId())
                .setFolderId(params.folder.getId())
                .setOperationDateTime(operation.getCreateDateTime())
                .setId(UUID.randomUUID().toString())
                .setProviderRequestId(operation.getLastRequestId().orElseThrow())
                .setOperationType(FolderOperationType.CREATE_ACCOUNT)
                .setAuthorUserId(params.userDetails.getUser().orElseThrow().getId())
                .setAuthorUserUid(params.userDetails.getUid().orElseThrow())
                .setAuthorProviderId(null)
                .setSourceFolderOperationsLogId(null)
                .setDestinationFolderOperationsLogId(null)
                .setOldFolderFields(null)
                .setOldQuotas(new QuotasByResource(Map.of()))
                .setNewQuotas(new QuotasByResource(Map.of()))
                .setOldProvisions(new QuotasByAccount(Map.of()))
                .setNewProvisions(new QuotasByAccount(Map.of()))
                .setOldBalance(new QuotasByResource(Map.of()))
                .setNewBalance(new QuotasByResource(Map.of()))
                .setOldFolderFields(null)
                .setActuallyAppliedProvisions(null)
                .setNewAccounts(new AccountsHistoryModel(Map.of(operation.getRequestedChanges()
                                .getAccountCreateParams().orElseThrow().getAccountId(),
                        AccountHistoryModel.builder()
                                .version(0L)
                                .providerId(operation.getProviderId())
                                .accountsSpacesId(operation.getAccountsSpaceId().orElse(null))
                                .outerAccountIdInProvider(null)
                                .outerAccountKeyInProvider(inputDto.getAccountKey())
                                .folderId(params.folder.getId())
                                .displayName(inputDto.getAccountName())
                                .deleted(false)
                                .lastReceivedVersion(null)
                                .reserveType(AccountReserveTypeInputDto
                                        .toModel(inputDto.getReserveType().orElse(null)))
                                .build()
                )))
                .setAccountsQuotasOperationsId(operation.getOperationId())
                .setQuotasDemandsId(null)
                .setOperationPhase(OperationPhase.SUBMIT)
                .setOrder(operation.getOrders().getSubmitOrder())
                .build();
    }

    private FolderOperationLogModel getCloseFolderLog(
            CreateParameters params,
            AccountsQuotasOperationsModel completeOperation,
            CreateAccountResult accountWithDefaults
    ) {
        AccountModel account = accountWithDefaults.account;
        QuotasByResource newQuotasByResource = new QuotasByResource(accountWithDefaults.newFolderQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getQuota)));
        QuotasByResource oldFolderQuotasByResource = new QuotasByResource(accountWithDefaults.oldFolderQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getQuota)));
        QuotasByResource newBalancesByResource = new QuotasByResource(accountWithDefaults.newFolderQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getBalance)));
        QuotasByResource oldBalancesByResource = new QuotasByResource(accountWithDefaults.oldFolderQuotas.stream()
                .collect(Collectors.toMap(QuotaModel::getResourceId, QuotaModel::getQuota)));
        QuotasByAccount quotasByAccount = new QuotasByAccount(accountWithDefaults.accountsQuotas.isEmpty() ? Map.of() :
                Map.of(account.getId(), new ProvisionsByResource(
                        accountWithDefaults.accountsQuotas.stream().collect(Collectors.toMap(
                                AccountsQuotasModel::getResourceId,
                                q -> new ProvisionHistoryModel(
                                        q.getProvidedQuota(),
                                        q.getLastReceivedProvisionVersion().orElse(0L)
                                )
                        )))));
        return FolderOperationLogModel.builder()
                .setTenantId(params.folder.getTenantId())
                .setFolderId(params.folder.getId())
                .setOperationDateTime(completeOperation.getUpdateDateTime().orElseThrow())
                .setId(UUID.randomUUID().toString())
                .setProviderRequestId(completeOperation.getLastRequestId().orElseThrow())
                .setOperationType(FolderOperationType.CREATE_ACCOUNT)
                .setAuthorUserId(params.userDetails.getUser().orElseThrow().getId())
                .setAuthorUserUid(params.userDetails.getUid().orElseThrow())
                .setAuthorProviderId(null)
                .setSourceFolderOperationsLogId(null)
                .setDestinationFolderOperationsLogId(null)
                .setOldFolderFields(null)
                .setNewFolderFields(null)
                .setOldAccounts(null)
                .setActuallyAppliedProvisions(null)
                .setNewAccounts(new AccountsHistoryModel(Map.of(account.getId(),
                        AccountHistoryModel.builder()
                                .version(0L)
                                .providerId(account.getProviderId())
                                .accountsSpacesId(account.getAccountsSpacesId().orElse(null))
                                .outerAccountIdInProvider(account.getOuterAccountIdInProvider())
                                .outerAccountKeyInProvider(account.getOuterAccountKeyInProvider().orElse(null))
                                .folderId(account.getFolderId())
                                .displayName(account.getDisplayName().orElse(null))
                                .deleted(account.isDeleted())
                                .lastReceivedVersion(account.getLastReceivedVersion().orElse(null))
                                .reserveType(account.getReserveType().orElse(null))
                                .build()
                )))
                .setAccountsQuotasOperationsId(completeOperation.getOperationId())
                .setQuotasDemandsId(null)
                .setOperationPhase(OperationPhase.CLOSE)
                .setOrder(completeOperation.getOrders().getCloseOrder().orElseThrow())
                .setOldQuotas(oldFolderQuotasByResource)
                .setNewQuotas(newQuotasByResource)
                .setOldProvisions(new QuotasByAccount(Map.of()))
                .setNewProvisions(quotasByAccount)
                .setOldBalance(oldBalancesByResource)
                .setNewBalance(newBalancesByResource)
                .build();
    }

    public AccountsQuotasOperationsModel getOperation(FrontAccountInputDto inputDto, CreateParameters params,
                                                       FolderModel folder) {
        return AccountsQuotasOperationsModel.builder()
                .setTenantId(folder.getTenantId())
                .setOperationId(UUID.randomUUID().toString())
                .setLastRequestId(UUID.randomUUID().toString())
                .setCreateDateTime(Instant.now())
                .setOperationSource(OperationSource.USER)
                .setOperationType(AccountsQuotasOperationsModel.OperationType.CREATE_ACCOUNT)
                .setAuthorUserId(params.userDetails.getUser().orElseThrow().getId())
                .setAuthorUserUid(params.userDetails.getUid().orElseThrow())
                .setProviderId(params.provider.getId())
                .setAccountsSpaceId(params.accountSpace.map(as -> as.getAccountsSpaces().getId()).orElse(null))
                .setUpdateDateTime(null)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.WAITING)
                .setErrorMessage(null)
                .setRequestedChanges(OperationChangesModel.builder()
                        .accountCreateParams(new OperationChangesModel.AccountCreateParams(
                                inputDto.getAccountKey(), inputDto.getAccountName(), folder.getId(),
                                UUID.randomUUID().toString(),
                                inputDto.getFreeTier() != null ? inputDto.getFreeTier() : false,
                                AccountReserveTypeInputDto.toModel(inputDto.getReserveType().orElse(null))))
                        .build())
                .setOrders(OperationOrdersModel.builder().submitOrder(folder.getNextOpLogOrder()).build())
                .setErrorKind(null)
                .build();
    }

    private AccountsQuotasOperationsModel getCompleteOperation(AccountsQuotasOperationsModel operation,
                                                               FolderModel folder) {
        AccountsQuotasOperationsModel result = new AccountsQuotasOperationsModel.Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.OK)
                .setUpdateDateTime(Instant.now())
                .setOrders(OperationOrdersModel.builder(operation.getOrders())
                        .closeOrder(folder.getNextOpLogOrder())
                        .build())
                .setErrorKind(null)
                .build();
        operationsObservabilityService.observeOperationFinished(result);
        return result;
    }

    private AccountsQuotasOperationsModel getFailedOperation(AccountsQuotasOperationsModel operation,
                                                             String errorMessage, boolean conflict) {
        AccountsQuotasOperationsModel result = new AccountsQuotasOperationsModel.Builder(operation)
                .setRequestStatus(AccountsQuotasOperationsModel.RequestStatus.ERROR)
                .setUpdateDateTime(Instant.now())
                .setErrorMessage(errorMessage)
                .setErrorKind(conflict ? OperationErrorKind.ALREADY_EXISTS : OperationErrorKind.INVALID_ARGUMENT)
                .build();
        operationsObservabilityService.observeOperationFinished(result);
        return result;
    }

    public <T> Result<T> generateResultForFailedOperation(AccountsQuotasOperationsModel operation,
                                                          boolean isConflict, Locale locale,
                                                          Function<String, TypedError> typedErrorFromMessage) {
        ErrorCollection.Builder errorBuilder = ErrorCollection.builder();
        if (operation.getFullErrorMessage().isPresent()) {
            errorBuilder.add(operation.getFullErrorMessage().get().getErrorCollection(locale).toErrorCollection());
        } else {
            String errorMessage = operation.getErrorMessage().orElseGet(() -> {
                if (isConflict) {
                    return messages.getMessage("errors.grpc.code.failed.precondition", null, locale);
                } else {
                    return messages.getMessage("errors.bad.request", null, locale);
                }
            });
            errorBuilder.addError(typedErrorFromMessage.apply(errorMessage));
        }

        return Result.failure(errorBuilder
                .addDetail("operationMeta", new ProvisionOperationFailureMeta(operation.getOperationId()))
                .build());
    }

    public OperationInProgressModel getOperationInProgress(AccountsQuotasOperationsModel operationsModel,
                                                           CreateParameters params) {
        return new OperationInProgressModel(params.folder.getTenantId(),
                operationsModel.getOperationId(),
                params.folder.getId(), null, 0L);
    }

    private Result<AccountDto> processAccountDtoResponse(Response<AccountDto> response, Locale locale) {
        return response.match(new Response.Cases<>() {
            @Override
            public Result<AccountDto> success(AccountDto result, String requestId) {
                if (result.getAccountId().isEmpty()) {
                    return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                            .getMessage("errors.no.account.id.received.from.provider", null, locale)))
                            .build());
                }
                return Result.success(result);
            }

            @Override
            public Result<AccountDto> failure(Throwable error) {
                LOG.error("Failure on provider account creation", error);

                return Result.failure(
                        ErrorCollection.builder().addError(TypedError.badRequest(messages
                                .getMessage("errors.provider.account.creation",
                                        null, locale)))
                                .build());
            }

            @Override
            public Result<AccountDto> error(ProviderError error, String requestId) {
                LOG.error("Error on provider account creation error: {}, requestId: {}", error, requestId);

                if (error.isConflict()) {
                    return Result.failure(ErrorCollection.builder()
                            .addError(TypedError.conflict(getProviderErrorMessage(error, locale)))
                            .build());
                }
                return Result.failure(ErrorCollection.builder()
                        .addError(TypedError.badRequest(getProviderErrorMessage(error, locale)))
                        .build());
            }
        });
    }

    private String getProviderErrorMessage(ProviderError error, Locale locale) {
        return error.match(new ProviderError.Cases<>() {

            @Override
            public String httpError(int statusCode) {
                return messages.getMessage("errors.provider.account.creation.status.code",
                        new Object[]{statusCode}, locale);
            }

            @Override
            public String httpExtendedError(int statusCode, ErrorMessagesDto errors) {
                String message = errors.getMessage().map(m -> ", message: " + m).orElse("") +
                        errors.getFieldErrors().filter(fe -> !fe.isEmpty()).map(fe -> fe.entrySet().stream()
                                .map(e -> e.getKey() + ": " + e.getValue())
                                .collect(Collectors.joining(", ", ", ", ""))).orElse("");

                return messages.getMessage("errors.provider.account.creation.status.code.with.message",
                        new Object[]{statusCode, message}, locale);
            }

            @Override
            public String grpcError(Status.Code statusCode, String message) {
                return messages.getMessage("errors.provider.account.creation.status.code.with.message",
                        new Object[]{statusCode, message}, locale);
            }

            @Override
            public String grpcExtendedError(Status.Code statusCode, String message,
                                            Map<String, String> badRequestDetails) {
                String resultMessage = ", message: " + messages +
                        (badRequestDetails.isEmpty() ? "" : badRequestDetails.entrySet().stream()
                                .map(e -> e.getKey() + ": " + e.getValue())
                                .collect(Collectors.joining(", ", ", ", "")));
                return messages.getMessage("errors.provider.account.creation.status.code.with.message",
                        new Object[]{statusCode, resultMessage}, locale);
            }
        });
    }

    private Mono<Result<CreateAccountResult>> createAccountModel(
            AccountDto accountDto, CreateParameters params, Operation hr, Locale locale
    ) {

        String displayName = params.provider.getAccountsSettings().isDisplayNameSupported() ?
                accountDto.getDisplayName().orElse(null) : null;

        String outerKey = params.provider.getAccountsSettings().isKeySupported() ?
                accountDto.getKey().orElse(null) : null;

        final AccountModel accountModel = new AccountModel.Builder()
                .setId(hr.operation.getRequestedChanges().getAccountCreateParams().orElseThrow().getAccountId())
                .setAccountsSpacesId(params.accountSpace.map(as -> as.getAccountsSpaces().getId()).orElse(null))
                .setDeleted(false)
                .setDisplayName(displayName)
                .setFolderId(params.folder.getId())
                .setProviderId(params.provider.getId())
                .setTenantId(params.provider.getTenantId())
                .setVersion(0L)
                .setLastAccountUpdate(Instant.now())
                .setLastReceivedVersion(accountDto.getAccountVersion().orElse(null))
                .setOuterAccountIdInProvider(accountDto.getAccountId().orElseThrow())
                .setOuterAccountKeyInProvider(outerKey)
                .setLatestSuccessfulAccountOperationId(accountDto.getLastUpdate()
                        .flatMap(LastUpdateDto::getOperationId)
                        .orElse(null))
                .setFreeTier(accountDto.isFreeTier().orElse(false))
                .setReserveType(params.getAccountReserveType().orElse(null))
                .build();

        return tableClient.usingSessionMonoRetryable(session ->
                session.usingCompResultTxRetryable(TransactionMode.SERIALIZABLE_READ_WRITE,
                        ts -> meter(folderDao
                                .getByIdStartTx(ts, params.folder.getId(), params.folder.getTenantId()),
                                        "New account, get folder")
                                .map(tx -> ResultTx.success(tx.get(), tx.getTransactionId())),
                        (ts, folderO) -> validateFolder(folderO, locale)
                                .applyMono(folder -> meter(reserveAccountsService.adjustForReserveConflictMono(ts,
                                                accountModel), "New account, adjust for reserve conflict")
                                .flatMap(adjustedAccount ->
                                        meter(accountsDao.upsertOneRetryable(ts, adjustedAccount),
                                                "New account, add account")
                                .flatMap(account -> meter(reserveAccountsService.addReserveAccountMono(ts, account)
                                        .thenReturn(account), "New account, add reserve index")
                                .flatMap(a -> meter(addDefaultQuotas(ts, account, accountDto, locale),
                                        "New account, adding default quotas")
                                .flatMap(accountWithDefaults -> meter(accountsQuotasOperationsDao
                                        .upsertOneRetryable(ts, getCompleteOperation(hr.operation, folder)),
                                                "New account, update operation")
                                .flatMap(completeOperation -> meter(folderOperationLogDao
                                        .upsertOneRetryable(ts, getCloseFolderLog(params, completeOperation,
                                                accountWithDefaults)), "New account, second add history")
                                .flatMap(updatedOperation -> meter(folderDao
                                        .upsertOneRetryable(ts, folder.toBuilder()
                                                .setNextOpLogOrder(folder.getNextOpLogOrder() + 1L)
                                                .build()), "New account, second update folder")
                                .flatMap(updatedFolder -> meter(removeOperationInProgress(ts, hr.operationInProgress)
                                        .thenReturn(accountWithDefaults), "New account, remove operation in progress")
                                ))))))))
                                .map(result -> new WithTxId<>(result, ts.getId())),
                        (ts, b) -> meter(ts.commitTransaction().thenReturn(new WithTxId<>(b, null)),
                                "New account, second commit")
                )).map(r -> r.andThen(identity()));
    }

    private Mono<CreateAccountResult> addDefaultQuotas(
            YdbTxSession ts, AccountModel account, AccountDto accountDto, Locale locale
    ) {
        if (accountDto.getProvisions().isEmpty() || accountDto.getProvisions().get().isEmpty()) {
            return Mono.just(new CreateAccountResult(account));
        }
        List<ProvisionDto> provisions = new ArrayList<>();
        for (ProvisionDto provisionDto : accountDto.getProvisions().get()) {
            if (provisionDto.getResourceKey().isEmpty()) {
                LOG.warn("Received provision with empty resource key . " + provisionDto);
                continue;
            }
            if (provisionDto.getProvidedAmount().isEmpty()) {
                LOG.warn("Received provision with empty provided amount . " + provisionDto);
                continue;
            }
            if (provisionDto.getProvidedAmountUnitKey().isEmpty()) {
                LOG.warn("Received provision with empty provided unit. " + provisionDto);
                continue;
            }
            provisions.add(provisionDto);
        }
        if (provisions.isEmpty()) {
            return Mono.just(new CreateAccountResult(account));
        }
        List<ResourceComplexKey> resourceComplexKeys = provisions.stream()
                .map(ProvisionDto::getResourceKey)
                .map(Optional::get)
                .map(ResourceComplexKey::new)
                .collect(Collectors.toList());
        return resourcesByKeysLoader.getResources(
                ts, account.getTenantId(), account.getProviderId(), resourceComplexKeys, locale
        ).map(resourcesResultsByComplexKey -> {
            Map<ResourceComplexKey, ResourceModel> resourcesByComplexKey = new HashMap<>();
            resourcesResultsByComplexKey.forEach((resourceComplexKey, resourceModelResult) ->
                    resourceModelResult.doOnSuccess(resourceModel -> resourcesByComplexKey.put(
                            resourceComplexKey, resourceModel
                    )));
            return resourcesByComplexKey;
        }).flatMap(resourcesByComplexKey -> quotasDao.getByProviderFoldersResources(ts,
                account.getTenantId(),
                Set.of(account.getFolderId()),
                account.getProviderId(),
                resourcesByComplexKey.values().stream().map(ResourceModel::getId).collect(Collectors.toSet())
        ).flatMap(folderQuotas -> unitsEnsemblesLoader.getUnitsEnsemblesByIds(ts,
                resourcesByComplexKey.values().stream()
                        .map(ResourceModel::getUnitsEnsembleId)
                        .distinct()
                        .map(resourceId -> Tuples.of(resourceId, account.getTenantId()))
                        .collect(Collectors.toList())
        ).flatMap(unitsEnsembles -> {
            Map<String, UnitsEnsembleModel> unitsEnsemblesById =
                    unitsEnsembles.stream().collect(Collectors.toMap(UnitsEnsembleModel::getId, identity()));
            List<AccountsQuotasModel> providedQuotas = new ArrayList<>();
            boolean accountsSpaceIsPresent = account.getAccountsSpacesId().isPresent();
            String accountsSpaceId = account.getAccountsSpacesId().orElse(null);

            Map<String, QuotaModel> oldQuotasByResourceId =
                    folderQuotas.stream().collect(Collectors.toMap(QuotaModel::getResourceId, identity()));
            List<QuotaModel> newQuotas = new ArrayList<>();
            List<QuotaModel> oldQuotas = new ArrayList<>();

            Map<String, ResourceModel> defaultResourcesById = new HashMap<>();
            for (ProvisionDto provision : provisions) {
                // provision уже отфильтрован, есть resourceKey, ProvidedAmount и getProvidedAmountUnitKey
                ResourceModel resource = resourcesByComplexKey.get(
                        new ResourceComplexKey(provision.getResourceKey().orElseThrow()));
                if (resource == null) {
                    LOG.warn("Resource not found by key. " + provision.getResourceKey().orElseThrow());
                    continue;
                }
                if (accountsSpaceIsPresent && !Objects.equals(accountsSpaceId, resource.getAccountsSpacesId())) {
                    LOG.warn("Received resource from another accounts space. " + resource + "; " + account);
                    continue;
                }
                String resourceId = resource.getId();
                defaultResourcesById.put(resourceId, resource);
                UnitsEnsembleModel unitsEnsemble = unitsEnsemblesById.get(resource.getUnitsEnsembleId());

                Optional<Long> providedQuota = convertQuotaFromApi(
                        provision.getProvidedAmount().orElseThrow(),
                        provision.getProvidedAmountUnitKey().orElseThrow(),
                        resource,
                        unitsEnsemble
                );
                if (providedQuota.isEmpty()) {
                    continue;
                }
                long provided = providedQuota.orElseThrow();
                long allocated = provision.getAllocatedAmount().flatMap(allocatedQuota -> {
                    if (provision.getAllocatedAmountUnitKey().isEmpty()) {
                        LOG.warn("Received allocated quota without unit. " + provision + "; " + accountDto);
                        return Optional.empty();
                    }
                    return convertQuotaFromApi(
                            allocatedQuota,
                            provision.getAllocatedAmountUnitKey().orElseThrow(),
                            resource,
                            unitsEnsemble
                    );
                }).orElse(0L);

                providedQuotas.add(new AccountsQuotasModel.Builder()
                        .setTenantId(account.getTenantId())
                        .setAccountId(account.getId())
                        .setResourceId(resourceId)
                        .setProvidedQuota(provided)
                        .setAllocatedQuota(allocated)
                        .setFolderId(account.getFolderId())
                        .setProviderId(account.getProviderId())
                        .setLastProvisionUpdate(account.getLastAccountUpdate())
                        .setLatestSuccessfulProvisionOperationId(
                                account.getLatestSuccessfulAccountOperationId().orElse(null))
                        .setLastReceivedProvisionVersion(null)
                        .build());

                long defaultQuota = resource.getDefaultQuota().orElse(0L);
                Optional<QuotaModel> oldQuotaModel = Optional.ofNullable(oldQuotasByResourceId.get(resourceId));
                long oldQuota = oldQuotaModel.map(QuotaModel::getQuota).orElse(0L);
                long oldBalance = oldQuotaModel.map(QuotaModel::getBalance).orElse(0L);
                long oldFrozenQuota = oldQuotaModel.map(QuotaModel::getFrozenQuota).orElse(0L);
                QuotaModel newQuota = QuotaModel.builder()
                        .tenantId(account.getTenantId())
                        .folderId(account.getFolderId())
                        .providerId(account.getProviderId())
                        .resourceId(resourceId)
                        .quota(oldQuota + defaultQuota)
                        .balance(oldBalance + defaultQuota - provided)
                        .frozenQuota(oldFrozenQuota)
                        .build();
                newQuotas.add(newQuota);
                oldQuotaModel.ifPresent(oldQuotas::add);
            }
            return accountsQuotasDao.upsertAllRetryable(ts, providedQuotas)
                    .then(Mono.defer(() -> quotasDao.upsertAllRetryable(ts, newQuotas))
            ).thenReturn(new CreateAccountResult(
                    account, providedQuotas, newQuotas, oldQuotas, defaultResourcesById));
        })));
    }

    private Optional<Long> convertQuotaFromApi(
            long quotaValue, String quotaUnitKey, ResourceModel resource, UnitsEnsembleModel unitsEnsemble
    ) {
        Optional<UnitModel> unit = unitsEnsemble.unitByKey(quotaUnitKey);
        if (unit.isEmpty()) {
            LOG.error("Unit not found by key. '" + quotaUnitKey + "'");
            return Optional.empty();
        }
        Optional<Long> newProvidedInBaseUnit =
                Units.convertFromApi(quotaValue, resource, unitsEnsemble, unit.orElseThrow());
        if (newProvidedInBaseUnit.isEmpty()) {
            LOG.warn("Received value cannot be converted to a base unit. "
                    + quotaValue + "; " + unit);
            return Optional.empty();
        }
        return newProvidedInBaseUnit;
    }

    private Mono<Void> removeOperationInProgress(YdbTxSession ts, OperationInProgressModel oip) {
        return operationsInProgressDao.deleteOneRetryable(ts, new WithTenant<>(oip.getTenantId(), oip.getKey()));
    }

    private AccountsSpaceKeyRequestDto toSpaceRequestDto(ExpandedAccountsSpaces<AccountSpaceModel> accountSpaceModel) {
        return new AccountsSpaceKeyRequestDto(accountSpaceModel.getAccountsSpaces().getSegments()
                .stream()
                .map(s -> new SegmentKeyRequestDto(
                        accountSpaceModel.getSegmentations().get(s.getSegmentationId()).getKey(),
                        accountSpaceModel.getSegments().get(s.getSegmentId()).getKey()))
                .collect(Collectors.toList()));
    }

    public static class CreateParameters {
        private final ProviderModel provider;
        private final FolderModel folder;
        private final Optional<ExpandedAccountsSpaces<AccountSpaceModel>> accountSpace;
        private final YaUserDetails userDetails;
        @Nullable
        private final AccountReserveType accountReserveType;

        public CreateParameters(ProviderModel provider, FolderModel folder,
                                Optional<ExpandedAccountsSpaces<AccountSpaceModel>> accountSpace,
                                YaUserDetails userDetails,
                                @Nullable AccountReserveType accountReserveType) {
            this.provider = provider;
            this.folder = folder;
            this.accountSpace = accountSpace;
            this.userDetails = userDetails;
            this.accountReserveType = accountReserveType;
        }

        public ProviderModel getProvider() {
            return provider;
        }

        public FolderModel getFolder() {
            return folder;
        }

        public Optional<ExpandedAccountsSpaces<AccountSpaceModel>> getAccountSpace() {
            return accountSpace;
        }

        public YaUserDetails getUserDetails() {
            return userDetails;
        }

        public Optional<AccountReserveType> getAccountReserveType() {
            return Optional.ofNullable(accountReserveType);
        }
    }

    public static class Operation {
        private final AccountsQuotasOperationsModel operation;
        private final OperationInProgressModel operationInProgress;

        public Operation(AccountsQuotasOperationsModel operation, OperationInProgressModel operationInProgress) {

            this.operation = operation;
            this.operationInProgress = operationInProgress;
        }

        public AccountsQuotasOperationsModel getOperation() {
            return operation;
        }

        public OperationInProgressModel getOperationInProgress() {
            return operationInProgress;
        }
    }


    public Mono<Result<AccountModel>> getAccountAndCheckPermissions(String accountId, TenantId tenantId,
                                                                    YaUserDetails currentUser, Locale locale) {
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res -> res.andThenMono(u ->
                tableClient.usingSessionMonoRetryable(session ->
                        accountsDao.getById(
                                session.asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY), accountId, tenantId
                        ).map(account -> account.isEmpty() || account.get().isDeleted() ? accountNotFound(locale) :
                                Result.success(account.get())
                        )
                )));
    }

    private Mono<List<ResourceModel>> getAllProviderResourcesInAccountsSpace(String providerId,
                                                                             TenantId tenantId,
                                                                             String accountsSpaceId) {
        return tableClient.usingSessionMonoRetryable(session ->
                resourcesDao.getAllByProvider(session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY),
                        providerId, tenantId, false)
                        .map(rs -> rs.stream()
                                .filter(res -> Objects.equals(accountsSpaceId, res.getAccountsSpacesId()))
                                .collect(Collectors.toList())
                        )
        );
    }

    public Mono<Result<List<ResourceModel>>> getAvailableResources(String accountId,
                                                                   YaUserDetails currentUser,
                                                                   Locale locale) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return getAccountAndCheckPermissions(accountId, tenantId, currentUser, locale)
                .flatMap(r -> r.applyMono(a ->
                        getAllProviderResourcesInAccountsSpace(a.getProviderId(), a.getTenantId(),
                                a.getAccountsSpacesId().orElse(null))
                ));
    }

    private Result<AccountModel> accountNotFound(Locale locale) {
        return Result.failure(ErrorCollection.builder().addError(
                TypedError.notFound(messages.getMessage("errors.account.not.found", null, locale)))
                .build());
    }

    private static <T> Mono<T> meter(Mono<T> mono, String label) {
        return AsyncMetrics.metric(mono,
                (millis, success) -> LOG.info("{}: duration = {} ms, success = {}", label, millis, success));
    }

    public static class CreateAccountResult {
        private final AccountModel account;
        private final List<AccountsQuotasModel> accountsQuotas;
        private final List<QuotaModel> newFolderQuotas;
        private final List<QuotaModel> oldFolderQuotas;
        private final Map<String, ResourceModel> resourcesById;

        public CreateAccountResult(AccountModel account) {
            this(account, List.of(), List.of(), List.of(), Map.of());
        }

        public CreateAccountResult(
                AccountModel account,
                List<AccountsQuotasModel> accountsQuotas,
                List<QuotaModel> newFolderQuotas,
                List<QuotaModel> oldFolderQuotas,
                Map<String, ResourceModel> resourcesById
        ) {
            this.account = account;
            this.accountsQuotas = accountsQuotas;
            this.newFolderQuotas = newFolderQuotas;
            this.oldFolderQuotas = oldFolderQuotas;
            this.resourcesById = resourcesById;
        }

        public AccountModel getAccount() {
            return account;
        }

        public List<AccountsQuotasModel> getAccountsQuotas() {
            return accountsQuotas;
        }

        public List<QuotaModel> getNewFolderQuotas() {
            return newFolderQuotas;
        }

        public List<QuotaModel> getOldFolderQuotas() {
            return oldFolderQuotas;
        }

        public Map<String, ResourceModel> getResourcesById() {
            return resourcesById;
        }
    }

    public static final class ResultHolder {

        private final CreateAccountExpandedAnswerDto response;
        private final AccountsQuotasOperationsModel operation;
        private final AccountModel account;

        private ResultHolder(CreateAccountExpandedAnswerDto response,
                             AccountsQuotasOperationsModel operation,
                             AccountModel account) {
            this.response = response;
            this.operation = operation;
            this.account = account;
        }

        public static ResultHolder success(CreateAccountExpandedAnswerDto response,
                                           AccountsQuotasOperationsModel operation,
                                           AccountModel account) {
            return new ResultHolder(response, operation, account);
        }

        public static ResultHolder inProgress(AccountsQuotasOperationsModel operation) {
            return new ResultHolder(null, operation, null);
        }

        public Optional<CreateAccountExpandedAnswerDto> getResponse() {
            return Optional.ofNullable(response);
        }

        public AccountsQuotasOperationsModel getOperation() {
            return operation;
        }

        public Optional<AccountModel> getAccount() {
            return Optional.ofNullable(account);
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ResultHolder that = (ResultHolder) o;
            return Objects.equals(response, that.response) &&
                    Objects.equals(operation, that.operation) &&
                    Objects.equals(account, that.account);
        }

        @Override
        public int hashCode() {
            return Objects.hash(response, operation, account);
        }

        @Override
        public String toString() {
            return "ResultHolder{" +
                    "response=" + response +
                    ", operation=" + operation +
                    ", account=" + account +
                    '}';
        }

    }

}
