package ru.yandex.intranet.d.services.accounts;

import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.accounts.AccountByFolderAndProviderPagingFrom;
import ru.yandex.intranet.d.dao.accounts.AccountByFolderPagingFrom;
import ru.yandex.intranet.d.dao.accounts.AccountsDao;
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.datasource.model.YdbSession;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.loaders.providers.AccountSpacesLoader;
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader;
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader;
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader;
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader;
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.accounts.AccountModel;
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel;
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel;
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel;
import ru.yandex.intranet.d.services.accounts.model.AccountByFolderAndProviderContinuationToken;
import ru.yandex.intranet.d.services.accounts.model.AccountByFolderContinuationToken;
import ru.yandex.intranet.d.services.accounts.model.AccountsByFolderResult;
import ru.yandex.intranet.d.services.security.SecurityManagerService;
import ru.yandex.intranet.d.util.ObjectMapperHolder;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.paging.ContinuationTokens;
import ru.yandex.intranet.d.util.paging.Page;
import ru.yandex.intranet.d.util.paging.PageRequest;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.security.model.YaUserDetails;

/**
 * Accounts read service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class AccountsReadService {

    private final MessageSource messages;
    private final ObjectReader byFolderContinuationTokenReader;
    private final ObjectWriter byFolderContinuationTokenWriter;
    private final ObjectReader byProviderContinuationTokenReader;
    private final ObjectWriter byProviderContinuationTokenWriter;
    private final SecurityManagerService securityManagerService;
    private final YdbTableClient tableClient;
    private final FolderDao folderDao;
    private final AccountsDao accountsDao;
    private final ProvidersLoader providersLoader;
    private final AccountSpacesLoader accountSpacesLoader;
    private final ResourceSegmentationsLoader resourceSegmentationsLoader;
    private final ResourceSegmentsLoader resourceSegmentsLoader;
    private final AccountsQuotasDao accountsQuotasDao;
    private final ResourcesLoader resourcesLoader;
    private final UnitsEnsemblesLoader unitsEnsemblesLoader;

    @SuppressWarnings("ParameterNumber")
    public AccountsReadService(@Qualifier("messageSource") MessageSource messages,
                               @Qualifier("continuationTokensJsonObjectMapper") ObjectMapperHolder objectMapper,
                               SecurityManagerService securityManagerService,
                               YdbTableClient tableClient,
                               FolderDao folderDao,
                               AccountsDao accountsDao,
                               ProvidersLoader providersLoader, AccountSpacesLoader accountSpacesLoader,
                               ResourceSegmentationsLoader resourceSegmentationsLoader,
                               ResourceSegmentsLoader resourceSegmentsLoader, AccountsQuotasDao accountsQuotasDao,
                               ResourcesLoader resourcesLoader, UnitsEnsemblesLoader unitsEnsemblesLoader) {
        this.messages = messages;
        this.byFolderContinuationTokenReader = objectMapper.getObjectMapper()
                .readerFor(AccountByFolderContinuationToken.class);
        this.byFolderContinuationTokenWriter = objectMapper.getObjectMapper()
                .writerFor(AccountByFolderContinuationToken.class);
        this.byProviderContinuationTokenReader = objectMapper.getObjectMapper()
                .readerFor(AccountByFolderAndProviderContinuationToken.class);
        this.byProviderContinuationTokenWriter = objectMapper.getObjectMapper()
                .writerFor(AccountByFolderAndProviderContinuationToken.class);
        this.securityManagerService = securityManagerService;
        this.tableClient = tableClient;
        this.folderDao = folderDao;
        this.accountsDao = accountsDao;
        this.providersLoader = providersLoader;
        this.accountSpacesLoader = accountSpacesLoader;
        this.resourceSegmentationsLoader = resourceSegmentationsLoader;
        this.resourceSegmentsLoader = resourceSegmentsLoader;
        this.accountsQuotasDao = accountsQuotasDao;
        this.resourcesLoader = resourcesLoader;
        this.unitsEnsemblesLoader = unitsEnsemblesLoader;
    }

    public Mono<Result<AccountModel>> getById(String folderId, String accountId, YaUserDetails currentUser,
                                              Locale locale) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res ->
                res.andThen(v -> validateFolderIdAccountId(folderId, accountId, locale)).andThenMono(u ->
                        tableClient.usingSessionMonoRetryable(session ->
                                getFolder(session, folderId, currentUser, locale, tenantId).flatMap(folderR ->
                                        folderR.andThenMono(folder -> getAccount(session, accountId, locale,
                                                tenantId, folder))))));
    }

    public Mono<Result<Page<AccountModel>>> getForProvider(String folderId, String providerId, PageRequest pageRequest,
                                                           YaUserDetails currentUser, Locale locale) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res ->
                res.andThen(v -> validateFolderIdProviderId(folderId, providerId, locale)).andThenMono(u -> {
                    Result<PageRequest.Validated<AccountByFolderAndProviderContinuationToken>> pageValidation
                            = pageRequest.validate(byProviderContinuationTokenReader, messages, locale);
                    return pageValidation
                            .andThenDo(p -> validateByProviderContinuationToken(p, locale)).andThenMono(p -> {
                                int limit = p.getLimit();
                                AccountByFolderAndProviderPagingFrom from = p.getContinuationToken().map(t ->
                                        new AccountByFolderAndProviderPagingFrom(t.getAccountsSpaceId().orElse(null),
                                                t.getAccountId())).orElse(null);
                                return tableClient.usingSessionMonoRetryable(session -> getFolderAndProvider(session,
                                        folderId, providerId, currentUser, locale, tenantId).flatMap(tr ->
                                        tr.andThenMono(t -> accountsDao.getByFolderAndProvider(immediateTx(session),
                                                        tenantId, t.getT1().getId(),
                                                        t.getT2().getId(), from, limit + 1, false)
                                                .map(values -> values.size() > limit
                                                        ? Page.page(values.subList(0, limit),
                                                        prepareByProviderToken(values.get(limit - 1)))
                                                        : Page.lastPage(values))
                                                .map(Result::success))));
                            });
                }));
    }

    public Mono<Result<Page<AccountModel>>> getForFolder(String folderId, PageRequest pageRequest,
                                                         YaUserDetails currentUser, Locale locale) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(currentUser, locale).flatMap(res ->
                res.andThen(v -> validateFolderId(folderId, locale)).andThenMono(u -> {
                    Result<PageRequest.Validated<AccountByFolderContinuationToken>> pageValidation
                            = pageRequest.validate(byFolderContinuationTokenReader, messages, locale);
                    return pageValidation
                            .andThenDo(p -> validateByFolderContinuationToken(p, locale)).andThenMono(p -> {
                                int limit = p.getLimit();
                                AccountByFolderPagingFrom from = p.getContinuationToken().map(t ->
                                        new AccountByFolderPagingFrom(t.getProviderId(),
                                                t.getAccountsSpaceId().orElse(null), t.getAccountId()))
                                        .orElse(null);
                                return tableClient.usingSessionMonoRetryable(session -> getFolder(session,
                                        folderId, currentUser, locale, tenantId).flatMap(tr ->
                                        tr.andThenMono(t -> accountsDao.getByFolder(immediateTx(session),
                                                        tenantId, t.getId(), from, limit + 1, false)
                                                .map(values -> values.size() > limit
                                                        ? Page.page(values.subList(0, limit),
                                                        prepareByFolderToken(values.get(limit - 1)))
                                                        : Page.lastPage(values))
                                                .map(Result::success))));
                            });
                }));
    }

    private Result<Void> validateFolderIdAccountId(String folderId, String accountId, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(folderId)) {
            errors.addError(TypedError.notFound(messages
                            .getMessage("errors.folder.not.found", null, locale)));
        }
        if (!Uuids.isValidUuid(accountId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.account.not.found", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Result<Void> validateFolderIdProviderId(String folderId, String providerId, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(folderId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        }
        if (!Uuids.isValidUuid(providerId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.provider.not.found", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Result<Void> validateFolderId(String folderId, Locale locale) {
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(folderId)) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Mono<Result<FolderModel>> getFolder(YdbSession session, String folderId, YaUserDetails currentUser,
                                                Locale locale, TenantId tenantId) {
        return folderDao.getById(immediateTx(session), folderId, tenantId).flatMap(folderO -> {
            if (folderO.isEmpty() || folderO.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError
                        .notFound(messages.getMessage("errors.folder.not.found", null, locale))).build();
                return Mono.just(Result.failure(error));
            }
            return securityManagerService.checkReadPermissions(folderO.get(), currentUser, locale, folderO.get());
        });
    }

    private Mono<Result<Tuple2<FolderModel, ProviderModel>>> getFolderAndProvider(YdbSession session,
                                                                                  String folderId,
                                                                                  String providerId,
                                                                                  YaUserDetails currentUser,
                                                                                  Locale locale,
                                                                                  TenantId tenantId) {
        return providersLoader.getProviderById(immediateTx(session), providerId, tenantId).flatMap(providerO ->
                folderDao.getById(immediateTx(session), folderId, tenantId).flatMap(folderO -> {
            ErrorCollection.Builder errors = ErrorCollection.builder();
            if (providerO.isEmpty() || providerO.get().isDeleted()) {
                errors.addError(TypedError
                        .notFound(messages.getMessage("errors.provider.not.found", null, locale)));
            }
            if (folderO.isEmpty() || folderO.get().isDeleted()) {
                errors.addError(TypedError
                        .notFound(messages.getMessage("errors.folder.not.found", null, locale)));
            }
            if (errors.hasAnyErrors()) {
                return Mono.just(Result.failure(errors.build()));
            }
            return securityManagerService.checkReadPermissions(folderO.orElseThrow(), currentUser, locale,
                    Tuples.of(folderO.orElseThrow(), providerO.orElseThrow()));
        }));
    }

    private Mono<Result<AccountModel>> getAccount(YdbSession session, String accountId, Locale locale,
                                                  TenantId tenantId, FolderModel folder) {
        return accountsDao.getByIdWithDeleted(immediateTx(session), accountId, tenantId)
                .map(accountO -> {
            if (accountO.isEmpty() || accountO.get().isDeleted()) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError
                        .notFound(messages.getMessage("errors.account.not.found", null, locale))).build();
                return Result.failure(error);
            }
            if (!accountO.get().getFolderId().equals(folder.getId())) {
                ErrorCollection error = ErrorCollection.builder().addError(TypedError.notFound(messages
                        .getMessage("errors.account.not.found.in.this.folder", null, locale))).build();
                return Result.failure(error);

            }
            return Result.success(accountO.get());
        });
    }

    private YdbTxSession immediateTx(YdbSession session) {
        return session.asTxCommitRetryable(TransactionMode.STALE_READ_ONLY);
    }

    private Result<Void> validateByProviderContinuationToken(
            PageRequest.Validated<AccountByFolderAndProviderContinuationToken> pageRequest, Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(pageRequest.getContinuationToken().get().getAccountId())) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.invalid.continuation.token", null, locale)));
        }
        if (pageRequest.getContinuationToken().get().getAccountsSpaceId().isPresent() &&
                !Uuids.isValidUuid(pageRequest.getContinuationToken().get().getAccountsSpaceId().get())) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.invalid.continuation.token", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private Result<Void> validateByFolderContinuationToken(
            PageRequest.Validated<AccountByFolderContinuationToken> pageRequest, Locale locale) {
        if (pageRequest.getContinuationToken().isEmpty()) {
            return Result.success(null);
        }
        ErrorCollection.Builder errors = ErrorCollection.builder();
        if (!Uuids.isValidUuid(pageRequest.getContinuationToken().get().getProviderId())) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.invalid.continuation.token", null, locale)));
        }
        if (!Uuids.isValidUuid(pageRequest.getContinuationToken().get().getAccountId())) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.invalid.continuation.token", null, locale)));
        }
        if (pageRequest.getContinuationToken().get().getAccountsSpaceId().isPresent() &&
                !Uuids.isValidUuid(pageRequest.getContinuationToken().get().getAccountsSpaceId().get())) {
            errors.addError(TypedError.notFound(messages
                    .getMessage("errors.invalid.continuation.token", null, locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private String prepareByProviderToken(AccountModel lastItem) {
        return ContinuationTokens.encode(new AccountByFolderAndProviderContinuationToken(lastItem
                        .getAccountsSpacesId().orElse(null), lastItem.getId()),
                byProviderContinuationTokenWriter);
    }

    private String prepareByFolderToken(AccountModel lastItem) {
        return ContinuationTokens.encode(new AccountByFolderContinuationToken(lastItem.getProviderId(), lastItem
                        .getAccountsSpacesId().orElse(null), lastItem.getId()),
                byFolderContinuationTokenWriter);
    }

    public Mono<Result<AccountsByFolderResult>> getAccountsByFolder(String folderId,
                                                                    String providerId,
                                                                    YaUserDetails currentUser,
                                                                    Locale locale) {
        TenantId tenantId = Tenants.getTenantId(currentUser);
        return securityManagerService.checkReadPermissions(folderId, currentUser, locale, null)
                .flatMap(res -> res.andThen(u ->
                        validateFolderIdProviderId(folderId, providerId, locale)).applyMono(v ->
                        getAccountsByFolderResultMono(folderId, providerId, tenantId)));
    }

    private Mono<AccountsByFolderResult> getAccountsByFolderResultMono(String folderId, String providerId,
                                                                       TenantId tenantId) {
        return tableClient.usingSessionMonoRetryable(session -> accountsDao.getByFoldersForProvider(session
                        .asTxCommitRetryable(TransactionMode.ONLINE_READ_ONLY), tenantId, providerId, folderId, false)
                .flatMap(accounts -> {
                    Set<String> accountsIds = accounts.stream()
                            .map(AccountModel::getId)
                            .collect(Collectors.toSet());

                    if (accountsIds.isEmpty()) {
                        return Mono.just(AccountsByFolderResult.EMPTY);
                    }

                    return accountSpacesLoader.getAllByProviderId(tenantId, providerId).flatMap(accountSpaces -> {
                        Set<String> accountSpacesSet = accounts.stream()
                                .map(AccountModel::getAccountsSpacesId)
                                .filter(Optional::isPresent)
                                .map(Optional::get)
                                .collect(Collectors.toSet());

                        List<AccountSpaceModel> filteredAccountSpaces = accountSpaces.stream()
                                .filter(accountSpaceModel -> accountSpacesSet.contains(accountSpaceModel.getId()))
                                .collect(Collectors.toList());

                        Set<String> segmentIds = new HashSet<>();
                        Set<String> segmentationIds = new HashSet<>();

                        filteredAccountSpaces.stream()
                                .flatMap(s -> s.getSegments().stream())
                                .forEach(s -> {
                                    segmentIds.add(s.getSegmentId());
                                    segmentationIds.add(s.getSegmentationId());
                                });

                        List<Tuple2<String, TenantId>> segmentationsIdsTuple = segmentationIds.stream()
                                .map(id -> Tuples.of(id, tenantId))
                                .collect(Collectors.toList());

                        List<Tuple2<String, TenantId>> segmentIdsTuple = segmentIds.stream()
                                .map(id -> Tuples.of(id, tenantId))
                                .collect(Collectors.toList());


                        return resourceSegmentationsLoader.getResourceSegmentationsByIdsImmediate(segmentationsIdsTuple)
                                .flatMap(segmentations -> resourceSegmentsLoader.getResourceSegmentsByIdsImmediate(
                                        segmentIdsTuple).flatMap(segments ->
                                        accountsQuotasDao.getAllByAccountIds(session.asTxCommitRetryable(
                                                        TransactionMode.ONLINE_READ_ONLY), tenantId, accountsIds)
                                                .flatMap(accountsQuotas -> getAccountsByFolderResultMono(tenantId,
                                                        accounts, filteredAccountSpaces, segmentations, segments,
                                                        accountsQuotas))));
                    });
                }));
    }

    private Mono<AccountsByFolderResult> getAccountsByFolderResultMono(TenantId tenantId, List<AccountModel> accounts,
                                                                       List<AccountSpaceModel> filteredAccountSpaces,
                                                                       List<ResourceSegmentationModel> segmentations,
                                                                       List<ResourceSegmentModel> segments,
                                                                       List<AccountsQuotasModel> accountsQuotas) {
        Set<String> resourcesSet = accountsQuotas.stream()
                .map(AccountsQuotasModel::getResourceId)
                .collect(Collectors.toSet());

        List<Tuple2<String, TenantId>> resourcesIds = resourcesSet.stream()
                .map(resourceId -> Tuples.of(resourceId, tenantId))
                .collect(Collectors.toList());

        return resourcesLoader.getResourcesByIdsImmediate(resourcesIds).flatMap(resources -> {
            Set<String> unitsEnsembleSet = resources.stream()
                    .map(ResourceModel::getUnitsEnsembleId)
                    .collect(Collectors.toSet());

            List<Tuple2<String, TenantId>> unitEnsembleIds = unitsEnsembleSet.stream()
                    .map(unitEnsembleId -> Tuples.of(unitEnsembleId, tenantId))
                    .collect(Collectors.toList());

            return unitsEnsemblesLoader.getUnitsEnsemblesByIdsImmediate(unitEnsembleIds).flatMap(unitEnsembles ->
                    Mono.just(new AccountsByFolderResult(accounts, filteredAccountSpaces, accountsQuotas, segmentations,
                            segments, resources, unitEnsembles)));
        });
    }

}
