package ru.yandex.intranet.d.services.accounts

import kotlinx.coroutines.reactor.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao
import ru.yandex.intranet.d.dao.accounts.ProviderReserveAccountsDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.datasource.model.YdbTxSession
import ru.yandex.intranet.d.kotlin.AccountId
import ru.yandex.intranet.d.kotlin.AccountsSpacesId
import ru.yandex.intranet.d.kotlin.FolderId
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ResourceTypeId
import ru.yandex.intranet.d.kotlin.SegmentId
import ru.yandex.intranet.d.kotlin.SegmentationId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.kotlin.UnitsEnsembleId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.WithTenant
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountReserveType
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.accounts.ProviderReserveAccountKey
import ru.yandex.intranet.d.model.accounts.ProviderReserveAccountModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.units.UnitModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.services.quotas.ExternalAccountUrlFactory
import ru.yandex.intranet.d.services.quotas.QuotasHelper
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.units.UnitsComparator
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.AccountDto
import ru.yandex.intranet.d.web.model.AmountDto
import ru.yandex.intranet.d.web.model.ProviderDto
import ru.yandex.intranet.d.web.model.ResourceDto
import ru.yandex.intranet.d.web.model.folders.FolderDto
import ru.yandex.intranet.d.web.model.folders.FrontProviderReserveAccountsDto
import ru.yandex.intranet.d.web.model.folders.FrontReserveAccountsDto
import ru.yandex.intranet.d.web.model.folders.front.ExpandedAccount
import ru.yandex.intranet.d.web.model.folders.front.ExpandedAccountResource
import ru.yandex.intranet.d.web.model.folders.front.ResourceTypeDto
import ru.yandex.intranet.d.web.model.providers.ProviderReserveAccountDto
import ru.yandex.intranet.d.web.model.providers.ProviderReserveAccountsDto
import ru.yandex.intranet.d.web.model.resources.AccountsSpaceDto
import ru.yandex.intranet.d.web.model.services.ServiceMinimalDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.intranet.d.web.util.ModelDtoConverter
import java.math.BigDecimal
import java.math.MathContext
import java.util.*

/**
 * Reserve accounts service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class ReserveAccountsService(
    private val providerReserveAccountsDao: ProviderReserveAccountsDao,
    private val securityManagerService: SecurityManagerService,
    private val servicesDao: ServicesDao,
    private val providersLoader: ProvidersLoader,
    private val accountsDao: AccountsDao,
    private val accountsSpacesDao: AccountsSpacesDao,
    private val resourceSegmentationsLoader: ResourceSegmentationsLoader,
    private val resourceSegmentsLoader: ResourceSegmentsLoader,
    private val accountsQuotasDao: AccountsQuotasDao,
    private val resourcesLoader: ResourcesLoader,
    private val resourceTypesLoader: ResourceTypesLoader,
    private val unitsEnsemblesLoader: UnitsEnsemblesLoader,
    private val folderDao: FolderDao,
    private val quotasDao: QuotasDao,
    private val tableClient: YdbTableClient,
    @Qualifier("messageSource") private val messages: MessageSource
) {

    fun addReserveAccountMono(txSession: YdbTxSession, account: AccountModel): Mono<Void?> {
        return mono {
            addReserveAccount(txSession, account)
            null
        }
    }

    fun adjustForReserveConflictMono(txSession: YdbTxSession, account: AccountModel): Mono<AccountModel> {
        return mono {
            adjustForReserveConflict(txSession, account)
        }
    }

    fun removeReserveAccountsMono(txSession: YdbTxSession, accounts: List<AccountModel>): Mono<Void?> {
        return mono {
            removeReserveAccounts(txSession, accounts)
            null
        }
    }

    fun findReserveAccountsMono(serviceId: Long?, providerId: ProviderId?, user: YaUserDetails,
                                locale: Locale): Mono<Result<FrontReserveAccountsDto>> {
        return mono {
            findReserveAccounts(serviceId, providerId, user, locale)
        }
    }

    fun findReserveAccountsApiMono(providerId: ProviderId, user: YaUserDetails,
                                   locale: Locale): Mono<Result<ProviderReserveAccountsDto>> {
        return mono {
            findReserveAccountsApi(providerId, user, locale)
        }
    }

    suspend fun validateReserve(txSession: YdbTxSession,
                                accountReserveType: AccountReserveType?,
                                tenantId: TenantId,
                                provider: ProviderModel,
                                serviceId: ServiceId,
                                folderId: FolderId,
                                accountsSpaceId: AccountsSpacesId?,
                                locale: Locale): Result<Unit> {
        if (accountReserveType == null) {
            return Result.success(Unit)
        }
        if (accountReserveType == AccountReserveType.PROVIDER) {
            val reserveAccounts = dbSessionRetryable(tableClient) {
                providerReserveAccountsDao.getAllByProvider(txSession, tenantId, provider.id)
            }!!
            if (reserveAccounts.isEmpty()) {
                return Result.success(Unit)
            }
            // MDS needs multiple reserves because each account may contain only one of possible resources
            // and there are no accounts spaces for MDS
            val multipleReservesAllowed = provider.accountsSettings.multipleReservesAllowed.orElse(false)
            if (accountsSpaceId == null && !multipleReservesAllowed) {
                return alreadyExists(locale)
            }
            if (accountsSpaceId != null) {
                val reserveAccountsSpaces = reserveAccounts.mapNotNull { it.key.accountsSpaceId }.toSet()
                if (reserveAccountsSpaces.contains(accountsSpaceId) && !multipleReservesAllowed) {
                    return alreadyExists(locale)
                }
            }
            val accountsModels = loadAccounts(reserveAccounts)
            val folders = loadFolders(accountsModels)
            val serviceIds = folders.map { it.serviceId }.distinct()
            if (serviceIds.size > 1 || !serviceIds.contains(serviceId)) {
                return Result.failure(ErrorCollection.builder().addError(TypedError.conflict(messages
                    .getMessage("errors.multiple.services.with.reserve.accounts", null, locale))).build())
            }
            val multipleAccountsPerFolderAllowed = provider.isMultipleAccountsPerFolder
            // S3 supports at most one account per folder so we need multiple folders to support multiple reserves
            val multipleFoldersWithReserveAllowed = multipleReservesAllowed && !multipleAccountsPerFolderAllowed
            val foldersIds = folders.map { it.id }.distinct()
            if ((folders.size > 1 || !foldersIds.contains(folderId)) && !multipleFoldersWithReserveAllowed) {
                return Result.failure(ErrorCollection.builder().addError(TypedError.conflict(messages
                    .getMessage("errors.multiple.folders.with.reserve.accounts", null, locale))).build())
            }
        }
        return Result.success(Unit)
    }

    private fun alreadyExists(locale: Locale): Result<Unit> =
        Result.failure(ErrorCollection.builder().addError(TypedError.conflict(messages
            .getMessage("errors.provider.reserve.account.already.exists", null, locale))).build())

    suspend fun validateReservePermission(accountReserveType: AccountReserveType?,
                                          provider: ProviderModel,
                                          user: YaUserDetails,
                                          locale: Locale): Result<Unit> {
        if (accountReserveType == null) {
            return Result.success(Unit)
        }
        if (accountReserveType == AccountReserveType.PROVIDER) {
            val noPermissions = securityManagerService.filterProvidersWithProviderAdminRole(listOf(provider), user)
                .awaitSingle().isEmpty() && !user.user.map { it.dAdmin }.orElse(false)
            if (noPermissions) {
                return Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                    .getMessage("errors.not.enough.permissions.to.create.reserve.account", null, locale))).build())
            }
        }
        return Result.success(Unit)
    }

    suspend fun validateReserveUpdatePermission(oldAccountReserveType: AccountReserveType?,
                                                newAccountReserveType: AccountReserveType?,
                                                provider: ProviderModel,
                                                user: YaUserDetails,
                                                locale: Locale): Result<Unit> {
        if (oldAccountReserveType == null && newAccountReserveType == null) {
            return Result.success(Unit)
        }
        if (oldAccountReserveType != newAccountReserveType
            && (oldAccountReserveType == AccountReserveType.PROVIDER || newAccountReserveType == AccountReserveType.PROVIDER)
        ) {
            val noPermissions = securityManagerService.filterProvidersWithProviderAdminRole(listOf(provider), user)
                .awaitSingle().isEmpty() && !user.user.map { it.dAdmin }.orElse(false)
            if (noPermissions) {
                return Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                    .getMessage("errors.not.enough.permissions.to.create.reserve.account", null, locale))).build())
            }
        }
        return Result.success(Unit)
    }

    suspend fun addReserveAccount(txSession: YdbTxSession, account: AccountModel) {
        if (account.reserveType.isEmpty) {
            return
        }
        if (account.reserveType.get() == AccountReserveType.PROVIDER) {
            addProviderReserveAccount(txSession, account)
        }
    }

    suspend fun removeReserveAccounts(txSession: YdbTxSession, accounts: Collection<AccountModel>) {

        val providerReservesToRemove = accounts.filter { it.reserveType.isPresent
            && it.reserveType.orElseThrow() == AccountReserveType.PROVIDER }
        removeProviderReserveAccounts(txSession, providerReservesToRemove)
    }

    suspend fun updateReserveAccount(txSession: YdbTxSession, oldAccount: AccountModel, newAccount: AccountModel) {
        val oldReserveType = oldAccount.reserveType.orElse(null)
        val newReserveType = newAccount.reserveType.orElse(null)
        if (oldReserveType == null && newReserveType == null) {
            return
        }
        if (Objects.equals(oldReserveType, newReserveType)) {
            return
        }
        if (oldReserveType == AccountReserveType.PROVIDER && newReserveType != AccountReserveType.PROVIDER) {
            removeProviderReserveAccount(txSession, newAccount)
        } else if (newReserveType == AccountReserveType.PROVIDER && oldReserveType != AccountReserveType.PROVIDER) {
            addProviderReserveAccount(txSession, newAccount)
        }
    }

    suspend fun addProviderReserveAccount(txSession: YdbTxSession, account: AccountModel) {
        if (AccountReserveType.PROVIDER != account.reserveType.orElse(null)) {
            throw IllegalStateException("Not a provider reserve account")
        }
        providerReserveAccountsDao.upsertOneRetryable(txSession, ProviderReserveAccountModel(
            key = ProviderReserveAccountKey(
                tenantId = account.tenantId,
                providerId = account.providerId,
                accountsSpaceId = account.accountsSpacesId.orElse(null),
                accountId = account.id
            )
        ))
    }

    suspend fun removeProviderReserveAccount(txSession: YdbTxSession, account: AccountModel) {
        providerReserveAccountsDao.deleteByIdRetryable(txSession, ProviderReserveAccountKey(
            tenantId = account.tenantId,
            providerId = account.providerId,
            accountsSpaceId = account.accountsSpacesId.orElse(null),
            accountId = account.id
        ))
    }

    suspend fun removeProviderReserveAccounts(txSession: YdbTxSession, accounts: Collection<AccountModel>) {
        providerReserveAccountsDao.deleteByIdsRetryable(txSession, accounts.map { ProviderReserveAccountKey(
            tenantId = it.tenantId,
            providerId = it.providerId,
            accountsSpaceId = it.accountsSpacesId.orElse(null),
            accountId = it.id
        )})
    }

    suspend fun existsProviderReserveAccount(txSession: YdbTxSession, tenantId: TenantId, providerId: ProviderId,
                                             accountsSpaceId: AccountsSpacesId?): Boolean {
        return if (accountsSpaceId == null) {
            providerReserveAccountsDao.existsByProvider(txSession, tenantId, providerId)
        } else {
            providerReserveAccountsDao.existsByProviderAccountSpace(txSession, tenantId, providerId,
                accountsSpaceId)
        }
    }

    suspend fun adjustForReserveConflict(txSession: YdbTxSession, account: AccountModel): AccountModel {
        if (account.reserveType.isEmpty) {
            return account
        } else {
            if (account.isDeleted) {
                return AccountModel.Builder(account).setReserveType(null).build()
            }
            return if (account.reserveType.get() == AccountReserveType.PROVIDER) {
                val reserveExists = existsProviderReserveAccount(txSession, account.tenantId, account.providerId,
                        account.accountsSpacesId.orElse(null))
                if (reserveExists) {
                    AccountModel.Builder(account).setReserveType(null).build()
                } else {
                    account
                }
            } else {
                account
            }
        }
    }

    suspend fun findReserveAccountsApi(providerId: ProviderId, user: YaUserDetails,
                                       locale: Locale): Result<ProviderReserveAccountsDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val provider = validateProvider(providerId, locale, false).bind()!!
        val reserveAccounts = findProviderReserveAccounts(provider)
        return Result.success(ProviderReserveAccountsDto(reserveAccounts
            .map { ProviderReserveAccountDto(it.key.accountId, it.key.accountsSpaceId) }))
    }

    suspend fun findReserveAccounts(serviceId: Long?, providerId: ProviderId?, user: YaUserDetails,
                                    locale: Locale): Result<FrontReserveAccountsDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        // Service will be used when there will be other reserve types
        validateService(serviceId, locale).bind()
        val provider = validateProvider(providerId, locale, true).bind()
        val reserveAccounts = findProviderReserveAccounts(provider)
        val accounts = loadAccounts(reserveAccounts)
        val folders = loadFolders(accounts)
        val services = loadServices(folders, locale)
        val providers = loadProviders(provider, accounts)
        val providersDto = prepareProviders(providers, locale)
        val accountsSpaces = loadAccountsSpaces(accounts)
        val accountsSpacesSegmentations = loadAccountsSpacesSegmentations(accountsSpaces)
        val accountsSpacesSegments = loadAccountsSpacesSegments(accountsSpaces)
        val accountsSpacesDto = prepareAccountsSpaces(accountsSpaces, accountsSpacesSegmentations,
            accountsSpacesSegments, locale)
        val accountsProvisions = loadAccountsProvisions(accounts)
        val resources = loadProvisionsResources(accountsProvisions)
        val resourceTypes = loadResourceTypes(resources)
        val unitsEnsembles = loadUnitsEnsembles(resources, resourceTypes)
        val resourcesSegmentations = loadResourceSegmentations(resources)
        val resourcesSegments = loadResourceSegments(resources)
        val quotas = loadQuotas(accounts, accountsProvisions)
        val resourceTypesDto = prepareResourceTypes(resourceTypes, unitsEnsembles, locale)
        val resourcesDto = prepareResources(resources, resourcesSegmentations, resourcesSegments, unitsEnsembles,
            locale)
        return Result.success(prepareReserveAccountsDto(services, accounts, providers, folders, resourcesSegmentations,
            resourcesSegments, accountsProvisions, resources, quotas, unitsEnsembles, providersDto, accountsSpacesDto,
            resourceTypesDto, resourcesDto, locale))
    }

    suspend fun findProviderReserveAccount(txSession: YdbTxSession, provider: ProviderModel,
                                           accountsSpace: AccountSpaceModel?): ProviderReserveAccountModel? {
        return if (accountsSpace != null) {
            providerReserveAccountsDao.getAllByProviderAccountsSpace(txSession,
                Tenants.DEFAULT_TENANT_ID, provider.id, accountsSpace.id).firstOrNull()
        } else {
            providerReserveAccountsDao.getAllByProvider(txSession,
                Tenants.DEFAULT_TENANT_ID, provider.id).firstOrNull()
        }
    }

    private fun prepareReserveAccountsDto(services: Map<ServiceId, ServiceMinimalDto>,
                                          accounts: List<AccountModel>,
                                          providers: List<ProviderModel>,
                                          folders: List<FolderDto>,
                                          resourcesSegmentations: Map<SegmentationId, ResourceSegmentationModel>,
                                          resourceSegments: Map<SegmentId, ResourceSegmentModel>,
                                          accountsProvisions: Map<AccountId, List<AccountsQuotasModel>>,
                                          resources: List<ResourceModel>,
                                          quotas: List<QuotaModel>,
                                          unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
                                          providersDto: Map<ProviderId, ProviderDto>,
                                          accountsSpacesDto: Map<AccountsSpacesId, AccountsSpaceDto>,
                                          resourceTypesDto: Map<ResourceTypeId, ResourceTypeDto>,
                                          resourcesDto: Map<ResourceId, ResourceDto>,
                                          locale: Locale): FrontReserveAccountsDto {
        val providersById = providers.associateBy { it.id }
        val resourcesById = resources.associateBy { it.id }
        val quotasByKey = quotas.associateBy { it.toKey() }
        val foldersById = folders.associateBy { it.id }
        val sortedUnitsCache = mutableMapOf<UnitsEnsembleId, List<UnitModel>>()
        val allowedSortedUnitsCache = mutableMapOf<ResourceId, List<UnitModel>>()
        val minAllowedUnitsCache = mutableMapOf<ResourceId, UnitModel>()
        val zeroAmountsCache = mutableMapOf<ResourceId, AmountDto>()
        val result = accounts.groupBy { it.providerId }.mapValues { e ->
            val providerAccounts = e.value
            val providerReserveServiceId = providerAccounts.map { foldersById[it.folderId]!!.serviceId }.firstOrNull()
            FrontProviderReserveAccountsDto(providerAccounts.map { account ->
                val provider = providersById[account.providerId]!!
                val folder = foldersById[account.folderId]!!
                val provisions = accountsProvisions[account.id] ?: emptyList()
                val provisionsResources = provisions.map { it.resourceId }.distinct().map { resourcesById[it]!! }
                val accountUrlFactory = ExternalAccountUrlFactory(provider.accountsSettings.externalAccountUrlTemplates,
                    folder.serviceId, resourcesSegmentations, resourceSegments, accountsSpacesDto.values.toList())
                val externalAccountUrlsPair = accountUrlFactory.generateUrl(account, provisionsResources)
                val urlForSegments = externalAccountUrlsPair?.first
                val generatedUrls = externalAccountUrlsPair?.second
                val expandedResources = provisions.mapNotNull { provision ->
                    val quota = quotasByKey[QuotaModel
                        .Key(provision.folderId, provision.providerId, provision.resourceId)]
                    val resource = resourcesById[provision.resourceId]!!
                    val unitsEnsemble = unitsEnsembles[resource.unitsEnsembleId]!!
                    prepareExpandedResource(provision, quota, unitsEnsemble, resource, sortedUnitsCache,
                        allowedSortedUnitsCache, minAllowedUnitsCache, zeroAmountsCache, locale)
                }
                ExpandedAccount(AccountDto.fromModel(account), expandedResources, generatedUrls, urlForSegments)
            }, providerReserveServiceId)}
        return FrontReserveAccountsDto(
            providerReserveAccounts = result,
            providers = providersDto,
            services = services,
            folders = foldersById,
            accountSpaces = accountsSpacesDto,
            resourceTypes = resourceTypesDto,
            resources = resourcesDto
        )
    }

    private fun prepareExpandedResource(provision: AccountsQuotasModel,
                                        folderQuota: QuotaModel?,
                                        unitsEnsemble: UnitsEnsembleModel,
                                        resource: ResourceModel,
                                        sortedUnitsCache: MutableMap<UnitsEnsembleId, List<UnitModel>>,
                                        allowedSortedUnitsCache: MutableMap<ResourceId, List<UnitModel>>,
                                        minAllowedUnitsCache: MutableMap<ResourceId, UnitModel>,
                                        zeroAmountsCache: MutableMap<ResourceId, AmountDto>,
                                        locale: Locale): ExpandedAccountResource? {
        val provided = provision.providedQuota ?: 0L
        val allocated = provision.allocatedQuota ?: 0L
        val quota = folderQuota?.quota ?: 0L
        if (provided == 0L && allocated == 0L && quota == 0L) {
            return null
        }
        val providedNotAllocated = if (provided != 0L || allocated != 0L) {
            QuotasHelper.toProvidedAndNotAllocated(provision.providedQuota, provision.allocatedQuota)
        } else {
            BigDecimal.ZERO
        }
        val providedRatio = if (quota != 0L) {
            BigDecimal.valueOf(provided).divide(BigDecimal.valueOf(quota), MathContext.DECIMAL64)
        } else {
            BigDecimal.ZERO
        }
        val allocatedRatio = if (quota != 0L) {
            BigDecimal.valueOf(allocated).divide(BigDecimal.valueOf(quota), MathContext.DECIMAL64)
        } else {
            BigDecimal.ZERO
        }
        val sortedUnits = sortedUnitsCache
            .computeIfAbsent(unitsEnsemble.id) { unitsEnsemble.units.sortedWith(UnitsComparator.INSTANCE) }
        val allowedSortedUnits = allowedSortedUnitsCache
            .computeIfAbsent(resource.id) { QuotasHelper.getAllowedUnits(resource, sortedUnits) }
        val baseUnit = unitsEnsemble.unitById(resource.baseUnitId).orElseThrow()
        val defaultUnit = unitsEnsemble.unitById(resource.resourceUnits.defaultUnitId).orElseThrow()
        val minAllowedUnit = minAllowedUnitsCache
            .computeIfAbsent(resource.id) { QuotasHelper.getMinAllowedUnit(resource.resourceUnits.allowedUnitIds,
                sortedUnits).orElse(baseUnit) }
        val providedAmount = if (provided != 0L) {
            val providedValue = BigDecimal.valueOf(provided)
            val providedForEditUnit = QuotasHelper.convertToReadable(providedValue, allowedSortedUnits, baseUnit).unit
            QuotasHelper.getAmountDto(providedValue, allowedSortedUnits, baseUnit, providedForEditUnit, defaultUnit,
                minAllowedUnit, locale)
        } else {
            zeroAmountsCache.computeIfAbsent(resource.id) { QuotasHelper
                .zeroAmount(defaultUnit, defaultUnit, minAllowedUnit, locale) }
        }
        val allocatedAmount = if (allocated != 0L) {
            val allocatedValue = BigDecimal.valueOf(allocated)
            val allocatedForEditUnit = QuotasHelper.convertToReadable(allocatedValue, allowedSortedUnits, baseUnit).unit
            QuotasHelper.getAmountDto(allocatedValue, allowedSortedUnits, baseUnit,
                allocatedForEditUnit, defaultUnit, minAllowedUnit, locale)
        } else {
            zeroAmountsCache.computeIfAbsent(resource.id) { QuotasHelper
                .zeroAmount(defaultUnit, defaultUnit, minAllowedUnit, locale) }
        }
        val providedNotAllocatedAmount = if (providedNotAllocated.compareTo(BigDecimal.ZERO) != 0) {
            val providedNotAllocatedForEditUnit = QuotasHelper.convertToReadable(providedNotAllocated,
                allowedSortedUnits, baseUnit).unit
            QuotasHelper.getAmountDto(providedNotAllocated, allowedSortedUnits, baseUnit,
                providedNotAllocatedForEditUnit, defaultUnit, minAllowedUnit, locale)
        } else {
            zeroAmountsCache.computeIfAbsent(resource.id) { QuotasHelper
                .zeroAmount(defaultUnit, defaultUnit, minAllowedUnit, locale) }
        }
        return ExpandedAccountResource(provision.resourceId, providedAmount, providedRatio, allocatedAmount,
            allocatedRatio, providedNotAllocatedAmount)
    }

    private suspend fun findProviderReserveAccounts(provider: ProviderModel?): List<ProviderReserveAccountModel> {
        return if (provider != null) {
            dbSessionRetryable(tableClient) {
                providerReserveAccountsDao.getAllByProvider(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                    provider.id)
            }!!
        } else {
            dbSessionRetryable(tableClient) {
                providerReserveAccountsDao.getAllByTenant(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID)
            }!!
        }
    }

    private suspend fun loadAccounts(reserveAccounts: List<ProviderReserveAccountModel>): List<AccountModel> {
        if (reserveAccounts.isEmpty()) {
            return emptyList()
        }
        return dbSessionRetryable(tableClient) {
            reserveAccounts.chunked(1000).flatMap { p -> accountsDao.getByIds(roStaleSingleRetryableCommit(),
                p.map { it.key.accountId }, Tenants.DEFAULT_TENANT_ID).awaitSingle() }
        }!!
    }

    private suspend fun validateService(serviceId: Long?, locale: Locale): Result<ServiceMinimalModel?> {
        if (serviceId == null) {
            return Result.success(null)
        }
        val serviceO = dbSessionRetryable(tableClient) {
            servicesDao.getByIdMinimal(roStaleSingleRetryableCommit(), serviceId)
                .awaitSingle().get()
        }!!
        if (serviceO.isEmpty) {
            return Result.failure(ErrorCollection.builder().addError("serviceId", TypedError.invalid(messages
                .getMessage("errors.service.not.found", null, locale))).build())
        }
        return Result.success(serviceO.get())
    }

    private suspend fun loadServices(
        folders: Collection<FolderDto>,
        locale: Locale
    ): Map<ServiceId, ServiceMinimalDto> {
        if (folders.isEmpty()) {
            return emptyMap()
        }
        val serviceIds = folders.map { it.serviceId }.distinct()
        val services = dbSessionRetryable(tableClient) {
            servicesDao.getByIdsMinimal(roStaleSingleRetryableCommit(), serviceIds)
                .awaitSingle()
        }!!.map { model -> ModelDtoConverter.toDto(model, locale) }.associateBy { it.id }
        return services
    }

    private suspend fun validateProvider(providerId: ProviderId?, locale: Locale, nullable: Boolean): Result<ProviderModel?> {
        if (nullable && providerId == null) {
            return Result.success(null)
        } else if (!nullable && providerId == null) {
            return Result.failure(ErrorCollection.builder().addError("providerId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale))).build())
        }
        if (!Uuids.isValidUuid(providerId)) {
            return Result.failure(ErrorCollection.builder().addError("providerId", TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale))).build())
        }
        val providerO = providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID).awaitSingle()
        if (providerO.isEmpty || providerO.get().isDeleted) {
            return Result.failure(ErrorCollection.builder().addError("providerId", TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale))).build())
        }
        return Result.success(providerO.get())
    }

    private suspend fun loadProviders(provider: ProviderModel?,
                                      accounts: List<AccountModel>): List<ProviderModel> {
        if (provider != null) {
            return listOf(provider)
        }
        if (accounts.isEmpty()) {
            return listOf()
        }
        val providerIds = accounts.map { it.providerId }.distinct()
        return providerIds.chunked(1000).flatMap { p -> providersLoader
            .getProvidersByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
    }

    private fun prepareProviders(providers: List<ProviderModel>, locale: Locale): Map<ProviderId, ProviderDto> {
        return providers.associateBy { it.id }.mapValues { ModelDtoConverter.providerDtoFromModel(it.value, locale) }
    }

    private suspend fun loadAccountsSpaces(accounts: List<AccountModel>): List<AccountSpaceModel> {
        val accountSpaceIds = accounts.mapNotNull { it.accountsSpacesId.orElse(null) }.distinct()
        if (accountSpaceIds.isEmpty()) {
            return emptyList()
        }
        return dbSessionRetryable(tableClient) {
            accountSpaceIds.chunked(1000).flatMap { p -> accountsSpacesDao
                .getByIds(roStaleSingleRetryableCommit(), p, Tenants.DEFAULT_TENANT_ID).awaitSingle() }
        }!!
    }

    private suspend fun loadAccountsSpacesSegmentations(
        accountsSpaces: List<AccountSpaceModel>
    ): Map<SegmentationId, ResourceSegmentationModel> {
        val segmentationIds = accountsSpaces.flatMap { it.segments.map { s -> s.segmentationId } }.distinct()
        if (segmentationIds.isEmpty()) {
            return emptyMap()
        }
        return segmentationIds.chunked(1000).flatMap { p -> resourceSegmentationsLoader
            .getResourceSegmentationsByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.associateBy { it.id }
    }

    private suspend fun loadAccountsSpacesSegments(
        accountsSpaces: List<AccountSpaceModel>
    ): Map<SegmentId, ResourceSegmentModel> {
        val segmentIds = accountsSpaces.flatMap { it.segments.map { s -> s.segmentId } }.distinct()
        if (segmentIds.isEmpty()) {
            return emptyMap()
        }
        return segmentIds.chunked(1000).flatMap { p -> resourceSegmentsLoader
            .getResourceSegmentsByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.associateBy { it.id }
    }

    private suspend fun loadResourceSegmentations(
        resources: List<ResourceModel>
    ): Map<SegmentationId, ResourceSegmentationModel> {
        val segmentationIds = resources.flatMap { it.segments.map { s -> s.segmentationId } }.distinct()
        if (segmentationIds.isEmpty()) {
            return emptyMap()
        }
        return segmentationIds.chunked(1000).flatMap { p -> resourceSegmentationsLoader
            .getResourceSegmentationsByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.associateBy { it.id }
    }

    private suspend fun loadResourceSegments(
        resources: List<ResourceModel>
    ): Map<SegmentId, ResourceSegmentModel> {
        val segmentIds = resources.flatMap { it.segments.map { s -> s.segmentId } }.distinct()
        if (segmentIds.isEmpty()) {
            return emptyMap()
        }
        return segmentIds.chunked(1000).flatMap { p -> resourceSegmentsLoader
            .getResourceSegmentsByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.associateBy { it.id }
    }

    private fun prepareAccountsSpaces(accountsSpaces: List<AccountSpaceModel>,
                                      segmentationsById: Map<SegmentationId, ResourceSegmentationModel>,
                                      segmentsById: Map<SegmentId, ResourceSegmentModel>,
                                      locale: Locale): Map<AccountsSpacesId, AccountsSpaceDto> {
        return accountsSpaces.associateBy { it.id }.mapValues { ModelDtoConverter.toDto(it.value, segmentationsById,
            segmentsById, locale) }
    }

    private suspend fun loadAccountsProvisions(accounts: List<AccountModel>): Map<AccountId, List<AccountsQuotasModel>> {
        if (accounts.isEmpty()) {
            return emptyMap()
        }
        val accountsIds = accounts.map { it.id }.toSet()
        return dbSessionRetryable(tableClient) {
            accountsQuotasDao.getAllByAccountIds(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                accountsIds).awaitSingle()
        }!!.groupBy { it.accountId }
    }

    private suspend fun loadProvisionsResources(
        provisionsByAccount: Map<String, List<AccountsQuotasModel>>): List<ResourceModel> {
        val resourceIds = provisionsByAccount.values.flatMap { it.map { r -> r.resourceId } }.distinct()
        if (resourceIds.isEmpty()) {
            return listOf()
        }
        return resourceIds.chunked(1000).flatMap { p -> resourcesLoader
            .getResourcesByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
    }

    private suspend fun loadResourceTypes(resources: List<ResourceModel>): List<ResourceTypeModel> {
        if (resources.isEmpty()) {
            return emptyList()
        }
        val resourceTypesIds = resources.map { it.resourceTypeId }.distinct()
        return resourceTypesIds.chunked(1000).flatMap { p -> resourceTypesLoader
            .getResourceTypesByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
    }

    private suspend fun loadUnitsEnsembles(resources: List<ResourceModel>,
                                           resourceTypes: List<ResourceTypeModel>): Map<UnitsEnsembleId, UnitsEnsembleModel> {
        val unitsEnsembleIds = (resources.map { it.unitsEnsembleId } + resourceTypes.map { it.unitsEnsembleId }).distinct()
        if (unitsEnsembleIds.isEmpty()) {
            return emptyMap()
        }
        return unitsEnsembleIds.chunked(1000).flatMap { p -> unitsEnsemblesLoader
            .getUnitsEnsemblesByIdsImmediate(p.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
            .associateBy { it.id }
    }

    private fun prepareResourceTypes(resourceTypes: List<ResourceTypeModel>,
                                     unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
                                     locale: Locale): Map<String, ResourceTypeDto> {
        return resourceTypes.associateBy { it.id }.mapValues { ModelDtoConverter
            .resourceTypeDtoFromModel(it.value, locale, unitsEnsembles) }
    }

    private fun prepareResources(resources: List<ResourceModel>,
                                 segmentationsById: Map<SegmentationId, ResourceSegmentationModel>,
                                 segmentsById: Map<SegmentId, ResourceSegmentModel>,
                                 unitsEnsembles: Map<UnitsEnsembleId, UnitsEnsembleModel>,
                                 locale: Locale): Map<String, ResourceDto> {
        return resources.associateBy { it.id }.mapValues {
            ResourceDto(it.value, locale, unitsEnsembles, segmentationsById, segmentsById) }
    }

    private suspend fun loadFolders(accounts: List<AccountModel>): List<FolderDto> {
        if (accounts.isEmpty()) {
            return emptyList()
        }
        val folderIds = accounts.map { it.folderId }.distinct()
        return dbSessionRetryable(tableClient) {
            folderIds.chunked(1000).flatMap { p -> folderDao.getByIds(roStaleSingleRetryableCommit(), p,
                Tenants.DEFAULT_TENANT_ID).awaitSingle() }
        }!!.map { model -> ModelDtoConverter.toDto(model) }
    }

    private suspend fun loadQuotas(accounts: List<AccountModel>,
                                   provisions: Map<AccountId, List<AccountsQuotasModel>>): List<QuotaModel> {
        val quotaKeys = mutableListOf<WithTenant<QuotaModel.Key>>()
        accounts.forEach { account ->
            val accountProvisions = provisions[account.id] ?: emptyList()
            accountProvisions.forEach { provision ->
                quotaKeys.add(WithTenant(provision.tenantId, QuotaModel.Key(provision.folderId, provision.providerId,
                    provision.resourceId)))
            }
        }
        if (quotaKeys.isEmpty()) {
            return emptyList()
        }
        return dbSessionRetryable(tableClient) {
            quotaKeys.chunked(1000).flatMap { p -> quotasDao
                .getByKeys(roStaleSingleRetryableCommit(), p).awaitSingle() }
        }!!
    }

}
