package ru.yandex.intranet.d.services.admin

import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.runBlocking
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import ru.yandex.intranet.d.kotlin.AccountsQuotasDaoAdapter
import ru.yandex.intranet.d.kotlin.FolderOperationLogDaoAdapter
import ru.yandex.intranet.d.kotlin.QuotasDaoAdapter
import ru.yandex.intranet.d.kotlin.ResourcesDaoAdapter
import ru.yandex.intranet.d.kotlin.TableClientAdapter
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.model.TenantId
import ru.yandex.intranet.d.model.accounts.AccountsQuotasModel
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel
import ru.yandex.intranet.d.model.folders.FolderOperationType
import ru.yandex.intranet.d.model.folders.ProvisionHistoryModel
import ru.yandex.intranet.d.model.folders.ProvisionsByResource
import ru.yandex.intranet.d.model.folders.QuotasByAccount
import ru.yandex.intranet.d.model.folders.QuotasByResource
import ru.yandex.intranet.d.model.quotas.QuotaModel
import ru.yandex.intranet.d.model.resources.ResourceBaseIdentity
import ru.yandex.intranet.d.services.admin.JobState.RUNNING
import ru.yandex.intranet.d.services.quotas.QuotaIdentity
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.validators.ProviderValidator
import ru.yandex.intranet.d.util.MdcTaskDecorator
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.quotas.ClearedQuotaDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.time.Instant
import java.util.*
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.RejectedExecutionException
import javax.annotation.PreDestroy
import kotlin.math.max

/**
 * AdminService.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 29-10-2021
 */
private const val INSERT_PARTITION_SIZE = 1000

@Component
class AdminService constructor(
    @Qualifier("messageSource") private val messages: MessageSource,
    private val providerValidator: ProviderValidator,
    private val tableClient: TableClientAdapter,
    private val securityManagerService: SecurityManagerService,
    private val quotasDao: QuotasDaoAdapter,
    private val accountsQuotasDao: AccountsQuotasDaoAdapter,
    private val folderOperationLogDao: FolderOperationLogDaoAdapter,
    private val resourcesDao: ResourcesDaoAdapter,
) {
    private val jobsStatusesByJobId = ConcurrentHashMap<String, JobStatus>()
    private val cleanQuotasExecutor = ThreadPoolTaskExecutor().apply {
        setQueueCapacity(2)
        maxPoolSize = 1
        corePoolSize = 1
        isDaemon = true
        threadNamePrefix = "clean-quotas-pool-"
        setTaskDecorator(MdcTaskDecorator())
        initialize()
    }

    @PreDestroy
    fun destroy() {
        cleanQuotasExecutor.destroy()
    }

    fun getJobStatus(
        jobId: String, currentUser: YaUserDetails, locale: Locale
    ): Result<JobStatus> = binding {
        securityManagerService.checkDictionaryWritePermissions(currentUser, locale).bind()
        val status: JobStatus? = jobsStatusesByJobId[jobId]
        return if (status != null) {
            Result.success(status)
        } else {
            Result.failure(
                ErrorCollection.builder().addError(TypedError.notFound("Job not found.")).build()
            )
        }
    }

    fun runCleanQuotasMono(
        providerId: String, resourceTypeId: String?, tenantId: TenantId, currentUser: YaUserDetails, locale: Locale
    ): Mono<Result<ClearedQuotaDto>> = mono {
        runCleanQuotas(
            currentUser = currentUser,
            locale = locale,
            resourceTypeId = resourceTypeId,
            tenantId = tenantId,
            providerId = providerId,
        )
    }

    suspend fun runCleanQuotas(
        currentUser: YaUserDetails, locale: Locale, resourceTypeId: String?, tenantId: TenantId, providerId: String
    ): Result<ClearedQuotaDto> = binding {
        securityManagerService.checkDictionaryWritePermissions(currentUser, locale).bind()
        val provider = providerValidator.validateProvider(tenantId, providerId, locale).awaitSingle().bind()!!
        if (provider.isSyncEnabled) {
            return Result.failure(
                ErrorCollection.builder().addError(TypedError.badRequest("Sync is enabled for provider.")).build()
            )
        }

        val jobId = UUID.randomUUID().toString()
        return try {
            cleanQuotasExecutor.execute {
                jobsStatusesByJobId[jobId] = JobStatus(RUNNING)
                val result = runBlocking {
                    cleanQuotas(locale, resourceTypeId, tenantId, providerId)
                }
                jobsStatusesByJobId[jobId] = result.toJobStatus()
            }
            Result.success(ClearedQuotaDto(jobId))
        } catch (e: RejectedExecutionException) {
            Result.failure(
                ErrorCollection.builder().addError(
                    TypedError.tooManyRequests(
                        messages.getMessage("errors.too.many.requests", null, locale)
                    )
                ).build()
            )
        }
    }

    private suspend fun cleanQuotas(
        locale: Locale, resourceTypeId: String?, tenantId: TenantId, providerId: String
    ): Result<ClearedQuotaDto> = binding {
        val resourceIds = loadResourceIds(tenantId, providerId, resourceTypeId, locale).bind()!!

        val quotas = tableClient.inTransactionOne { tx ->
            quotasDao.getAllByResourceIds(tx, tenantId, resourceIds)
        }!!
        if (quotas.isEmpty()) {
            return Result.failure(notFoundError("errors.quota.not.found", locale))
        }

        val accountsQuotas = tableClient.inTransactionOne { tx ->
            accountsQuotasDao.getAllByResourceIds(tx, tenantId, resourceIds)
        }!!

        val quotasByFolder = groupByFolder(quotas, accountsQuotas, locale).bind()!!
        forPartitions(quotasByFolder, INSERT_PARTITION_SIZE) {
            val quotasPart = ArrayList<QuotaModel>()
            val accountsQuotasPart = ArrayList<AccountsQuotasModel>()
            val operationLogsPart = ArrayList<FolderOperationLogModel>()
            for (folder in it) {
                quotasPart.addAll(folder.quotasInFolder)
                accountsQuotasPart.addAll(folder.accountsQuotasInFolder)
                operationLogsPart.add(
                    createFolderOperationLogToRemove(
                        tenantId, folder.folderId, folder.quotasInFolder, folder.accountsQuotasInFolder
                    )
                )
            }
            tableClient.inTransactionMany { tx ->
                quotasDao.deleteQuotasByModel(tx, quotasPart)
                accountsQuotasDao.removeAllModelsRetryable(tx, accountsQuotasPart)
                folderOperationLogDao.upsertAllRetryable(tx, operationLogsPart)
            }
        }

        return Result.success(ClearedQuotaDto(quotas.size, accountsQuotas.size))
    }

    private suspend fun loadResourceIds(
        tenantId: TenantId,
        providerId: String,
        resourceTypeId: String?,
        locale: Locale
    ): Result<List<String>> {
        val resourceIds: List<String>
        if (resourceTypeId != null) {
            resourceIds = tableClient.inTransactionOne { tx ->
                resourcesDao.getAllByBaseIdentities(
                    tx, listOf(ResourceBaseIdentity(tenantId, providerId, resourceTypeId)), false
                )
            }!!.map { it.id }
            if (resourceIds.isEmpty()) {
                return Result.failure(notFoundError("errors.resource.type.not.found", locale))
            }
        } else {
            resourceIds = tableClient.inTransactionOne { tx ->
                resourcesDao.getAllByProvider(tx, tenantId, providerId, false)
            }!!.map { it.id }
            if (resourceIds.isEmpty()) {
                return Result.failure(notFoundError("errors.provider.not.found", locale))
            }
        }
        return Result.success(resourceIds)
    }

    private inline fun forPartitions(
        quotasByFolder: List<QuotasInFolder>,
        @Suppress("SameParameterValue") // API
        partitionSize: Int,
        action: (List<QuotasInFolder>) -> Unit
    ) {
        val partition = ArrayList<QuotasInFolder>()
        var size = 0
        for (folder in quotasByFolder) {
            if (size + folder.size() > partitionSize && partition.isNotEmpty()) {
                action(partition)
                partition.clear()
                size = 0
            }
            partition.add(folder)
            size += folder.size()
        }
        if (partition.isNotEmpty()) {
            action(partition)
        }
    }

    private fun groupByFolder(
        quotas: List<QuotaModel>, accountsQuotas: List<AccountsQuotasModel>, locale: Locale,
    ): Result<List<QuotasInFolder>> {
        val accountQuotaByFolderAndResourceIds = accountsQuotas.groupBy { accountsQuota ->
            QuotaIdentity(accountsQuota.folderId, accountsQuota.resourceId)
        }.toMutableMap()

        val quotasByFolder = quotas.groupBy { it.folderId }
        val accountQuotasByFolder = quotasByFolder.entries.associateBy({ it.key }, { entry ->
            val quotasInFolder = entry.value
            val accountsQuotasInFolder: MutableList<AccountsQuotasModel> = ArrayList()
            for (quota: QuotaModel in quotasInFolder) {
                accountQuotaByFolderAndResourceIds.remove(
                    QuotaIdentity(quota.folderId, quota.resourceId)
                )?.let {
                    accountsQuotasInFolder.addAll(it)
                }
            }
            return@associateBy accountsQuotasInFolder
        })

        val accountsQuotasWithoutQuotasInFolder: MutableList<AccountsQuotasModel> = ArrayList()
        for (accountQuotaList: List<AccountsQuotasModel> in accountQuotaByFolderAndResourceIds.values) {
            for (accountQuota: AccountsQuotasModel in accountQuotaList) {
                if (accountQuota.providedQuota == 0L) {
                    accountQuotasByFolder.getOrDefault(accountQuota.folderId, ArrayList()).add(accountQuota)
                } else {
                    accountsQuotasWithoutQuotasInFolder.add(accountQuota)
                }
            }
        }

        //If account quota has provided quota, then should be quota on the folder
        if (accountsQuotasWithoutQuotasInFolder.isNotEmpty()) {
            return Result.failure(
                notFoundError(
                    "errors.quota.account.quota.has.no.folder.quota", locale,
                    accountsQuotasWithoutQuotasInFolder
                )
            )
        }

        val result = accountQuotasByFolder.entries.stream().map {
            QuotasInFolder(it.key, quotasByFolder.getOrDefault(it.key, ArrayList()), it.value) }
            .toList()

        println(quotasByFolder.values.stream().mapToInt{ it.size }.sum())
        println(accountQuotasByFolder.values.stream().mapToInt{ it.size }.sum())

        return Result.success(result)
    }

    private fun createFolderOperationLogToRemove(
        tenantId: TenantId, folderId: String,
        quotas: List<QuotaModel>, accountsQuotasList: List<AccountsQuotasModel>
    ): FolderOperationLogModel {
        val zeroQuotasByResource = QuotasByResource(quotas.associateBy({ it.resourceId }, { 0L }))
        val builder = FolderOperationLogModel.builder()
            .setId(UUID.randomUUID().toString())
            .setTenantId(tenantId)
            .setFolderId(folderId)
            .setOperationDateTime(Instant.now())
            .setOperationType(FolderOperationType.CLEAN_QUOTAS_BY_ADMIN)
            .setOldQuotas(QuotasByResource(quotas.associateBy({ it.resourceId }, { it.quota })))
            .setNewQuotas(zeroQuotasByResource)
            .setOldBalance(QuotasByResource(quotas.associateBy({ it.resourceId }, { it.balance })))
            .setNewBalance(zeroQuotasByResource)
        val accountsQuotasMap = accountsQuotasList
            .groupBy { accountQuota -> accountQuota.accountId }
            .mapValues { it.value.associateBy { accountQuota -> accountQuota.resourceId } }
        val oldProvisions = accountsQuotasMap.mapValues { byAccountId ->
            ProvisionsByResource(byAccountId.value.mapValues { byResourceId ->
                ProvisionHistoryModel(
                    byResourceId.value.providedQuota,
                    byResourceId.value.lastReceivedProvisionVersion.orElse(null)
                )
            })
        }
        val newProvisions = accountsQuotasMap.mapValues { byAccountId ->
            ProvisionsByResource(byAccountId.value.mapValues { byResourceId ->
                ProvisionHistoryModel(
                    0L,
                    byResourceId.value.lastReceivedProvisionVersion.orElse(0L) + 1L
                )
            })
        }
        builder
            .setOldProvisions(QuotasByAccount(oldProvisions))
            .setNewProvisions(QuotasByAccount(newProvisions))
        return builder.build()
    }

    private fun notFoundError(messageCode: String, locale: Locale, vararg args: Any) =
        ErrorCollection.builder().addError(
            TypedError.notFound(messages.getMessage(messageCode, args, locale))
        ).build()

    private data class QuotasInFolder(
        val folderId: String,
        val quotasInFolder: List<QuotaModel>,
        val accountsQuotasInFolder: List<AccountsQuotasModel>
    ) {
        fun size() = max(quotasInFolder.size, accountsQuotasInFolder.size)
    }
}

enum class JobState {
    RUNNING, SUCCESS, FAILURE;
}

data class JobStatus(
    val state: JobState,
    val message: String? = null,
)

fun <T> Result<T>.toJobStatus() =
    this.match({ JobStatus(JobState.SUCCESS, it.toString()) }, { JobStatus(JobState.FAILURE, it.toString()) })
