package ru.yandex.intranet.d.services.aggregates

import com.fasterxml.jackson.databind.ObjectReader
import com.fasterxml.jackson.databind.ObjectWriter
import kotlinx.coroutines.reactor.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.aggregates.ServiceAggregatesDao
import ru.yandex.intranet.d.dao.aggregates.ServiceDenormalizedAggregatesDao
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.kotlin.ProviderId
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ResourceTypeId
import ru.yandex.intranet.d.kotlin.SegmentId
import ru.yandex.intranet.d.kotlin.SegmentationId
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.resources.segmentations.ResourceSegmentationsLoader
import ru.yandex.intranet.d.loaders.resources.segments.ResourceSegmentsLoader
import ru.yandex.intranet.d.loaders.resources.types.ResourceTypesLoader
import ru.yandex.intranet.d.loaders.units.UnitsEnsemblesLoader
import ru.yandex.intranet.d.model.WithTenant
import ru.yandex.intranet.d.model.aggregates.AggregateBundle
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateKey
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateModel
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateUsageModel
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.providers.AggregationSettings
import ru.yandex.intranet.d.model.providers.FreeProvisionAggregationMode
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.providers.UsageMode
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.ResourceSegmentSettingsModel
import ru.yandex.intranet.d.model.resources.segmentations.ResourceSegmentationModel
import ru.yandex.intranet.d.model.resources.segments.ResourceSegmentModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.units.UnitModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.services.quotas.QuotasHelper
import ru.yandex.intranet.d.services.resources.ExpandedResources
import ru.yandex.intranet.d.services.resources.ResourceTreeUtils
import ru.yandex.intranet.d.services.resources.ResourceUtils
import ru.yandex.intranet.d.services.resources.SelectionTreeResourceDto
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.units.UnitsComparator
import ru.yandex.intranet.d.util.ObjectMapperHolder
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.paging.ContinuationTokens
import ru.yandex.intranet.d.util.paging.PageRequest
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.converters.toSelectionResourceTreeNodeDto
import ru.yandex.intranet.d.web.model.AmountDto
import ru.yandex.intranet.d.web.model.ProviderDto
import ru.yandex.intranet.d.web.model.ResourceDto
import ru.yandex.intranet.d.web.model.aggregation.AggregateAmountsDto
import ru.yandex.intranet.d.web.model.aggregation.AggregateForProviderDto
import ru.yandex.intranet.d.web.model.aggregation.AggregateForResourceDto
import ru.yandex.intranet.d.web.model.aggregation.AggregateForResourceTypeDto
import ru.yandex.intranet.d.web.model.aggregation.AggregateResourceSegmentDto
import ru.yandex.intranet.d.web.model.aggregation.FindServiceTotalsAggregateDto
import ru.yandex.intranet.d.web.model.aggregation.FindServiceTotalsRequestDto
import ru.yandex.intranet.d.web.model.aggregation.FindServiceTotalsResponseDto
import ru.yandex.intranet.d.web.model.aggregation.FindServiceTotalsSingleExpandedFilterDto
import ru.yandex.intranet.d.web.model.aggregation.FindSubtreeTotalAggregateDto
import ru.yandex.intranet.d.web.model.aggregation.FindSubtreeTotalRequestDto
import ru.yandex.intranet.d.web.model.aggregation.FindSubtreeTotalResponseDto
import ru.yandex.intranet.d.web.model.aggregation.FreeProvisionAggregationModeDto
import ru.yandex.intranet.d.web.model.aggregation.GetProvidersByServiceRequestDto
import ru.yandex.intranet.d.web.model.aggregation.GetResourceSelectionDto
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeAmountsAggregateDto
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeAmountsAggregateFolderDto
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeAmountsRequestDto
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeAmountsResponseDto
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.ALLOCATED
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.BALANCE
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.PROVIDED
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.QUOTA
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.TRANSFERABLE
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.UNALLOCATED
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.UNDERUTILIZED_ESTIMATION
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingField.UNUSED_ESTIMATION
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingOrder
import ru.yandex.intranet.d.web.model.aggregation.RankSubtreeSortingParamsDto
import ru.yandex.intranet.d.web.model.aggregation.ResourceUsageModeDto
import ru.yandex.intranet.d.web.model.aggregation.ServiceAggregateForProviderDto
import ru.yandex.intranet.d.web.model.aggregation.ServiceAggregateForResourceDto
import ru.yandex.intranet.d.web.model.aggregation.ServiceAggregateForResourceTypeDto
import ru.yandex.intranet.d.web.model.aggregation.ServiceAggregateResourceSegmentDto
import ru.yandex.intranet.d.web.model.aggregation.api.AggregateAmountApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.AggregateAmountsApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.FindSubtreeTotalAggregateApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.FindSubtreeTotalApiRequestDto
import ru.yandex.intranet.d.web.model.aggregation.api.FindSubtreeTotalApiResponseDto
import ru.yandex.intranet.d.web.model.aggregation.api.RankSubtreeAmountsAggregateApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.RankSubtreeAmountsApiRequestDto
import ru.yandex.intranet.d.web.model.aggregation.api.RankSubtreeAmountsApiResponseDto
import ru.yandex.intranet.d.web.model.resources.ResourceSelectionListResponseDto
import ru.yandex.intranet.d.web.model.resources.ResourceSelectionListSegmentDto
import ru.yandex.intranet.d.web.model.resources.ResourceSelectionListSegmentationDto
import ru.yandex.intranet.d.web.model.resources.SelectionResourceTreeNodeDto
import ru.yandex.intranet.d.web.model.resources.directory.segmentations.SegmentationUISettingsDto
import ru.yandex.intranet.d.web.model.units.front.FrontUnitsEnsembleDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.intranet.d.web.util.ModelDtoConverter
import ru.yandex.intranet.d.web.util.ModelDtoConverter.resourceTypeDtoFromModel
import java.math.BigDecimal
import java.math.BigInteger
import java.util.*

@Component
class QueryAggregatesService(private val securityManagerService: SecurityManagerService,
                             private val servicesDao: ServicesDao,
                             private val resourcesLoader: ResourcesLoader,
                             private val providersLoader: ProvidersLoader,
                             private val resourceTypesLoader: ResourceTypesLoader,
                             private val segmentationsLoader: ResourceSegmentationsLoader,
                             private val segmentsLoader: ResourceSegmentsLoader,
                             private val unitsEnsemblesLoader: UnitsEnsemblesLoader,
                             private val tableClient: YdbTableClient,
                             private val serviceAggregatesDao: ServiceAggregatesDao,
                             private val serviceDenormalizedAggregatesDao: ServiceDenormalizedAggregatesDao,
                             private val folderDao: FolderDao,
                             private val resourcesDao: ResourcesDao,
                             private val resourceUtils: ResourceUtils,
                             private val resourceTreeUtils: ResourceTreeUtils,
                             private val usageService: QueryAggregatesUsageService,
                             @Qualifier("continuationTokensJsonObjectMapper") private val objectMapper: ObjectMapperHolder,
                             @Qualifier("messageSource") private val messages: MessageSource
) {

    private val continuationTokenReader: ObjectReader = objectMapper.objectMapper
        .readerFor(RankSubtreeAmountsPageToken::class.java)
    private val continuationTokenWriter: ObjectWriter = objectMapper.objectMapper
        .writerFor(RankSubtreeAmountsPageToken::class.java)

    fun findSubtreeTotalMono(request: FindSubtreeTotalRequestDto,
                             user: YaUserDetails, locale: Locale): Mono<Result<FindSubtreeTotalResponseDto>> {
        return mono { findSubtreeTotal(request, user, locale) }
    }

    fun rankSubtreeAmountsMono(request: RankSubtreeAmountsRequestDto,
                               user: YaUserDetails, locale: Locale): Mono<Result<RankSubtreeAmountsResponseDto>> {
        return mono { rankSubtreeAmounts(request, user, locale) }
    }

    fun findServiceTotalsMono(request: FindServiceTotalsRequestDto,
                              user: YaUserDetails, locale: Locale): Mono<Result<FindServiceTotalsResponseDto>> {
        return mono { findServiceTotals(request, user, locale) }
    }

    fun findSubtreeTotalApiMono(request: FindSubtreeTotalApiRequestDto,
                                user: YaUserDetails, locale: Locale): Mono<Result<FindSubtreeTotalApiResponseDto>> {
        return mono { findSubtreeTotalApi(request, user, locale) }
    }

    fun rankSubtreeAmountsApiMono(request: RankSubtreeAmountsApiRequestDto,
                                  user: YaUserDetails, locale: Locale): Mono<Result<RankSubtreeAmountsApiResponseDto>> {
        return mono { rankSubtreeAmountsApi(request, user, locale) }
    }

    fun getProvidersByServiceMono(
        request: GetProvidersByServiceRequestDto,
        user: YaUserDetails, locale: Locale): Mono<Result<List<ProviderDto>>> {
        return mono { getProvidersByService(request, user, locale) }
    }

    fun getResourceSelectionTreeMono(
        request: GetResourceSelectionDto,
        user: YaUserDetails, locale: Locale): Mono<Result<SelectionResourceTreeNodeDto>> {
        return mono { getResourceSelectionTree(request, user, locale) }
    }

    fun getResourceSelectionListMono(
        request: GetResourceSelectionDto,
        user: YaUserDetails, locale: Locale): Mono<Result<ResourceSelectionListResponseDto>> {
        return mono { getResourceSelectionList(request, user, locale) }
    }

    private suspend fun findSubtreeTotal(request: FindSubtreeTotalRequestDto,
                                         user: YaUserDetails, locale: Locale): Result<FindSubtreeTotalResponseDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val aggregate = getServiceAggregate(validated.rootService, validated.resource)
        val usage = usageService.getServiceUsage(validated.rootService, validated.resource, aggregate) { a -> getBundle(a) }
        val response = prepareSubtreeTotalResponse(validated, aggregate, usage, locale)
        return Result.success(response)
    }

    private suspend fun rankSubtreeAmounts(request: RankSubtreeAmountsRequestDto,
                                           user: YaUserDetails, locale: Locale): Result<RankSubtreeAmountsResponseDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val aggregates = getDenormalizedAggregates(validated.rootService, validated.resource, preValidated.pageToken,
            preValidated.limit + 1, preValidated.sortingParams)
        val response = prepareSubtreeAmountsResponse(validated, preValidated, aggregates, locale)
        return Result.success(response)
    }

    private suspend fun findServiceTotals(request: FindServiceTotalsRequestDto,
                                          user: YaUserDetails, locale: Locale): Result<FindServiceTotalsResponseDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val aggregates = getServiceAggregates(validated)
        val response = prepareServiceTotalsResponse(aggregates, validated, locale)
        return Result.success(response)
    }

    private suspend fun findSubtreeTotalApi(request: FindSubtreeTotalApiRequestDto,
                                            user: YaUserDetails, locale: Locale): Result<FindSubtreeTotalApiResponseDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateApiRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val aggregate = getServiceAggregate(validated.rootService, validated.resource)
        val includeUsage = request.includeUsage ?: false
        val includeUsageRaw = request.includeUsageRaw ?: false
        val usage = if (includeUsage) {
            usageService.getServiceUsage(validated.rootService, validated.resource, aggregate) { a -> getBundle(a) }
        } else {
            null
        }
        val response = prepareSubtreeTotalApiResponse(validated, aggregate, usage, includeUsageRaw)
        return Result.success(response)
    }

    private suspend fun rankSubtreeAmountsApi(request: RankSubtreeAmountsApiRequestDto,
                                              user: YaUserDetails, locale: Locale): Result<RankSubtreeAmountsApiResponseDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateApiRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val aggregates = getDenormalizedAggregates(validated.rootService, validated.resource, preValidated.pageToken,
            preValidated.limit + 1, preValidated.sortingParams)
        val includeUsage = request.includeUsage ?: false
        val includeUsageRaw = request.includeUsageRaw ?: false
        val response = prepareSubtreeAmountsApiResponse(validated, preValidated, aggregates, includeUsage, includeUsageRaw)
        return Result.success(response)
    }

    private suspend fun getProvidersByService(
        request: GetProvidersByServiceRequestDto,
        user: YaUserDetails, locale: Locale): Result<List<ProviderDto>> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val providerIds = getProviderIdsByService(validated.rootService.id)
            .map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }.toList()
        val providers = providersLoader.getProvidersByIdsImmediate(providerIds).awaitSingle()
        val response = providers.stream().map{ m -> ModelDtoConverter.providerDtoFromModel(m, locale) }.toList()
        return Result.success(response)
    }

    private suspend fun getResourceSelectionTree(
        request: GetResourceSelectionDto,
        user: YaUserDetails, locale: Locale): Result<SelectionResourceTreeNodeDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val resourceIds = getResourceIdsByServiceAndProvider(validated.rootService.id, validated.provider.id)
            .map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }.toList()
        val resources: List<ResourceModel> = resourcesLoader.getResourcesByIdsImmediate(resourceIds).awaitSingle()
        val segmentationModels = segmentationsLoader.getResourceSegmentationsByIdsImmediate(
            resources.stream()
                .flatMap{ it.segments.stream() }
                .map(ResourceSegmentSettingsModel::getSegmentationId)
                .distinct()
                .map{ Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }
                .toList()
        ).awaitSingle()
        val segmentationIdsInOrder = segmentationModels
            .sortedWith(compareBy({ it.groupingOrder }, { it.id }))
            .map{ it.id }
            .toTypedArray()
        val selectionResourceTreeNode = resourceTreeUtils.groupToResourceTree(resources, segmentationIdsInOrder, 0)
        val expanded = resourceUtils.expand(
            selectionResourceTreeNode, resources, true, false
        ).awaitSingle()
        return Result.success(toSelectionResourceTreeNodeDto(expanded, locale))
    }

    private suspend fun getResourceSelectionList(
        request: GetResourceSelectionDto,
        user: YaUserDetails,
        locale: Locale
    ): Result<ResourceSelectionListResponseDto> = binding {
        securityManagerService.checkReadPermissions(user, locale).awaitSingle().bind()
        val preValidated = preValidateRequest(request, locale).bind()!!
        val validated = validate(preValidated, locale).bind()!!
        securityManagerService.checkReadPermissions(validated.rootService.id, user, locale).awaitSingle().bind()
        val resourceIds = getResourceIdsByServiceAndProvider(validated.rootService.id, validated.provider.id)
            .map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }.toList()
        val resources = resourcesLoader.getResourcesByIdsImmediate(resourceIds).awaitSingle()
        val expandedResources: ExpandedResources<List<ResourceModel>> = resourceUtils.expand(
            resources, resources, true, false
        ).awaitSingle()
        val response = toResourceSelectionListResponse(expandedResources, locale)
        return Result.success(response)
    }

    private fun toResourceSelectionListResponse(
        expandedResources: ExpandedResources<List<ResourceModel>>,
        locale: Locale
    ): ResourceSelectionListResponseDto {
        val resources: Map<ResourceId, SelectionTreeResourceDto> = expandedResources.resources
            .map { resourceModel ->
                val resourceType = expandedResources.resourceTypes[resourceModel.resourceTypeId]!!
                SelectionTreeResourceDto(
                    ResourceDto(resourceModel, locale),
                    resourceType.nameEn,
                    resourceType.nameRu,
                    resourceType.sortingOrder,
                    resourceType.key,
                    resourceModel.key
                )
            }
            .associateBy { r -> r.id }
        val resourceTypes = expandedResources.resourceTypes.mapValues { entry ->
            resourceTypeDtoFromModel(entry.value, locale, expandedResources.unitsEnsembles)
        }
        val segmentations = prepareResourceSelectionListSegmentations(
            expandedResources.resourceSegmentations, expandedResources.resourceSegments,
            expandedResources.resources, locale
        )
        val unitEnsemblesById = expandedResources.unitsEnsembles.mapValues { entry ->
            FrontUnitsEnsembleDto(entry.value, locale)
        }

        return ResourceSelectionListResponseDto(segmentations, resourceTypes, resources, unitEnsemblesById)
    }

    private fun prepareResourceSelectionListSegmentations(
        segmentationsById: Map<SegmentationId, ResourceSegmentationModel>,
        segmentsById: Map<SegmentId, ResourceSegmentModel>,
        resources: List<ResourceModel>,
        locale: Locale
    ): List<ResourceSelectionListSegmentationDto> {
        val segmentsBySegmentationId = segmentsById.values.groupBy { s -> s.segmentationId }

        return segmentationsById.values
            .sortedBy { segmentationModel ->
                segmentationModel.uiSettings.map { it.choiceOrder }.orElse(segmentationModel.groupingOrder)
            }
            .map { segmentationModel ->
                val segments = segmentsBySegmentationId[segmentationModel.id]
                    ?.map { segmentModel ->
                        val resourceTypeIds: Map<ResourceTypeId, Set<ResourceId>> = resources
                            .filter { r -> r.segments.any { it.segmentId == segmentModel.id } }
                            .groupBy({ r -> r.resourceTypeId }, { r -> r.id })
                            .mapValues { it.value.toSet() }
                        ResourceSelectionListSegmentDto(
                            segmentModel.id,
                            segmentModel.key,
                            Locales.select(segmentModel.nameEn, segmentModel.nameRu, locale),
                            segmentModel.uncommon.orElse(false),
                            resourceTypeIds
                        )
                    }
                    ?.toSet() ?: setOf()

                ResourceSelectionListSegmentationDto(
                    segmentationModel.id,
                    segmentationModel.key,
                    Locales.select(segmentationModel.nameEn, segmentationModel.nameRu, locale),
                    segmentationModel.groupingOrder,
                    segmentationModel.uiSettings.map { SegmentationUISettingsDto(it) }.orElse(null),
                    segments
                )
            }
    }

    private suspend fun prepareSubtreeTotalResponse(request: FindSubtreeTotal,
                                                    aggregate: ServiceAggregateModel?,
                                                    usage: ServiceAggregateUsageModel?,
                                                    locale: Locale): FindSubtreeTotalResponseDto {
        val provider = getProvider(request.resource.providerId)!!
        val resourceType = getResourceType(request.resource.resourceTypeId)!!
        return FindSubtreeTotalResponseDto(listOf(FindSubtreeTotalAggregateDto(
            serviceId = request.rootService.id,
            serviceName = Locales.selectFallbackToRu(request.rootService.nameEn, request.rootService.name, locale),
            serviceSlug = request.rootService.slug,
            providers = listOf(AggregateForProviderDto(
                providerId = provider.id,
                providerName = Locales.select(provider.nameEn, provider.nameRu, locale),
                resourceTypes = listOf(AggregateForResourceTypeDto(
                    resourceTypeId = resourceType.id,
                    resourceTypeName = Locales.select(resourceType.nameEn, resourceType.nameRu, locale),
                    resources = listOf(prepareResourceResponse(aggregate, request.resource, provider, resourceType,
                        usage, request.unitsEnsemble, request.unit, locale))
                ))
            ))
        )))
    }

    private fun prepareSubtreeTotalApiResponse(request: FindSubtreeTotal,
                                               aggregate: ServiceAggregateModel?,
                                               usage: ServiceAggregateUsageModel?,
                                               includeUsageRaw: Boolean): FindSubtreeTotalApiResponseDto {
        val quota = getBundle(aggregate)?.quota ?: BigInteger.ZERO
        val balance = getBundle(aggregate)?.balance ?: BigInteger.ZERO
        val provided = getBundle(aggregate)?.provided ?: BigInteger.ZERO
        val allocated = getBundle(aggregate)?.allocated ?: BigInteger.ZERO
        val unallocated = getBundle(aggregate)?.unallocated ?: BigInteger.ZERO
        val transferable = getBundle(aggregate)?.transferable ?: BigInteger.ZERO
        val convertedQuota = Units.convertToApi(quota, request.resource, request.unitsEnsemble)
        val convertedBalance = Units.convertToApi(balance, request.resource, request.unitsEnsemble)
        val convertedProvided = Units.convertToApi(provided, request.resource, request.unitsEnsemble)
        val convertedAllocated = Units.convertToApi(allocated, request.resource, request.unitsEnsemble)
        val convertedUnallocated = Units.convertToApi(unallocated, request.resource, request.unitsEnsemble)
        val convertedTransferable = Units.convertToApi(transferable, request.resource, request.unitsEnsemble)
        val aggregationSettings = resourceAggregationSettings(request.resource.id, request.provider,
            mapOf(request.resource.id to request.resource), mapOf(request.resourceType.id to request.resourceType))
        return FindSubtreeTotalApiResponseDto(
            aggregate = FindSubtreeTotalAggregateApiDto(
                amounts = AggregateAmountsApiDto(
                    quota = AggregateAmountApiDto(convertedQuota.t1.toPlainString(), convertedQuota.t2.key),
                    balance = AggregateAmountApiDto(convertedBalance.t1.toPlainString(), convertedBalance.t2.key),
                    provided = AggregateAmountApiDto(convertedProvided.t1.toPlainString(), convertedProvided.t2.key),
                    allocated = AggregateAmountApiDto(convertedAllocated.t1.toPlainString(), convertedAllocated.t2.key),
                    unallocated = AggregateAmountApiDto(convertedUnallocated.t1.toPlainString(), convertedAllocated.t2.key),
                    transferable = AggregateAmountApiDto(convertedTransferable.t1.toPlainString(), convertedAllocated.t2.key)
                ),
                lastUpdate = aggregate?.lastUpdate,
                usage = usageService.prepareServiceAggregateUsageApiDto(usageService.getTotalUsageAmount(usage?.exactAmounts),
                    usage?.lastUpdate, getBundle(aggregate), aggregationSettings, request.resource, request.unitsEnsemble, includeUsageRaw)
            )
        )
    }

    private suspend fun prepareResourceResponse(aggregate: ServiceAggregateModel?,
                                                resource: ResourceModel,
                                                provider: ProviderModel,
                                                resourceType: ResourceTypeModel,
                                                usage: ServiceAggregateUsageModel?,
                                                unitsEnsemble: UnitsEnsembleModel,
                                                unit: UnitModel?,
                                                locale: Locale): AggregateForResourceDto {
        val segmentations = getSegmentations(resource.segments.map { it.segmentationId })
        val segments = getSegments(resource.segments.map { it.segmentId })
        return prepareResourceResponse(aggregate, resource, provider, resourceType, segmentations, segments,
            unitsEnsemble, usage, unit, locale)
    }

    private fun prepareResourceResponse(aggregate: ServiceAggregateModel?,
                                        resource: ResourceModel,
                                        provider: ProviderModel,
                                        resourceType: ResourceTypeModel,
                                        segmentations: Map<String, ResourceSegmentationModel>,
                                        segments: Map<String, ResourceSegmentModel>,
                                        unitsEnsemble: UnitsEnsembleModel,
                                        usage: ServiceAggregateUsageModel?,
                                        unit: UnitModel?,
                                        locale: Locale): AggregateForResourceDto {
        val aggregationSettings = resourceAggregationSettings(resource.id, provider, mapOf(resource.id to resource),
            mapOf(resourceType.id to resourceType))
        return AggregateForResourceDto(
            resourceId = resource.id,
            resourceName = Locales.select(resource.nameEn, resource.nameRu, locale),
            segmentations = resource.segments.map {
                val segmentation = segmentations[it.segmentationId]!!
                val segment = segments[it.segmentId]!!
                AggregateResourceSegmentDto(
                    segmentationId = segmentation.id,
                    segmentationName = Locales.select(segmentation.nameEn, segmentation.nameRu, locale),
                    groupingOrder = segmentation.groupingOrder.toLong(),
                    segmentId = segment.id,
                    segmentName = Locales.select(segment.nameEn, segment.nameRu, locale),
                    segmentUncommon = segment.uncommon.orElse(false)
                )
            },
            freeProvisionAggregationMode = toAggregationMode(aggregationSettings.freeProvisionMode),
            usageMode = toUsageMode(aggregationSettings.usageMode ?: UsageMode.UNDEFINED),
            amounts = toAmounts(aggregate, resource, unitsEnsemble, aggregationSettings, unit, locale),
            lastUpdate = aggregate?.lastUpdate,
            usage = usageService.prepareResourceUsageDto(usageService.getTotalUsageAmount(usage?.exactAmounts),
                usage?.lastUpdate, getBundle(aggregate), aggregationSettings, resource, unitsEnsemble, unit, locale)
        )
    }

    private fun prepareServiceResourceResponse(aggregate: ServiceAggregateModel?,
                                               resource: ResourceModel,
                                               provider: ProviderModel,
                                               resourceType: ResourceTypeModel,
                                               segmentations: Map<String, ResourceSegmentationModel>,
                                               segments: Map<String, ResourceSegmentModel>,
                                               unitsEnsemble: UnitsEnsembleModel,
                                               resourceSegmentationVisibility: Map<ResourceId, Map<SegmentationId, Boolean>>,
                                               usage: ServiceAggregateUsageModel?,
                                               locale: Locale): ServiceAggregateForResourceDto {
        val aggregationSettings = resourceAggregationSettings(resource.id, provider, mapOf(resource.id to resource),
            mapOf(resourceType.id to resourceType))
        return ServiceAggregateForResourceDto(
            resourceId = resource.id,
            resourceName = Locales.select(resource.nameEn, resource.nameRu, locale),
            segmentations = resource.segments.map {
                val segmentation = segmentations[it.segmentationId]!!
                val segment = segments[it.segmentId]!!
                ServiceAggregateResourceSegmentDto(
                    segmentationId = segmentation.id,
                    segmentationName = Locales.select(segmentation.nameEn, segmentation.nameRu, locale),
                    groupingOrder = segmentation.groupingOrder.toLong(),
                    segmentId = segment.id,
                    segmentName = Locales.select(segment.nameEn, segment.nameRu, locale),
                    displayInExpandedView = (resourceSegmentationVisibility[resource
                        .id] ?: emptyMap())[segmentation.id] ?: false,
                    segmentUncommon = segment.uncommon.orElse(false)
                )
            },
            freeProvisionAggregationMode = toAggregationMode(aggregationSettings.freeProvisionMode),
            usageMode = toUsageMode(aggregationSettings.usageMode ?: UsageMode.UNDEFINED),
            amounts = toAmounts(aggregate, resource, unitsEnsemble, aggregationSettings, null, locale),
            lastUpdate = aggregate?.lastUpdate,
            usage = usageService.prepareResourceUsageDto(usageService.getTotalUsageAmount(usage?.exactAmounts),
                usage?.lastUpdate, getBundle(aggregate), aggregationSettings, resource, unitsEnsemble, null, locale)
        )
    }

    private suspend fun prepareServiceTotalsResponse(aggregates: List<ServiceAggregateModel>,
                                                     request: FindServiceTotals, locale: Locale): FindServiceTotalsResponseDto {
        if (aggregates.isEmpty() && request.filter != null) {
            return FindServiceTotalsResponseDto(listOf(FindServiceTotalsAggregateDto(
                serviceId = request.rootService.id,
                serviceName = Locales.selectFallbackToRu(request.rootService.nameEn, request.rootService.name, locale),
                serviceSlug = request.rootService.slug,
                providers = listOf(ServiceAggregateForProviderDto(
                    providerId = request.filter.provider.id,
                    providerName = Locales.select(request.filter.provider.nameEn, request.filter.provider.nameRu, locale),
                    resourceTypes = listOf()
                ))
            )))
        }
        val providers = getProviders(aggregates.map { it.key.providerId }.toSet()).filter { !it.isDeleted }
        val eligibleProviderIds = providers.map { it.id }.toSet()
        val providersResources = getProvidersResources(eligibleProviderIds)
        val expandByResourceType = prepareResourceTypesExpansion(providersResources)
        val resourceSegmentationVisibility = prepareResourceSegmentationVisibility(providersResources)
        val resources = getResources(aggregates.map { it.key.resourceId }.toSet()).filter { !it.isDeleted }
        val eligibleResourceIds = resources.map { it.id }.toSet()
        val resourceById = resources.associateBy { it.id }
        val resourceTypesById = getResourceTypes(resources.map { it.resourceTypeId }.toSet()).associateBy { it.id }
        val unitsEnsemblesById = getUnitsEnsembles(resources.map { it.unitsEnsembleId }.toSet()
            + resourceTypesById.values.map { it.unitsEnsembleId }.toSet()).associateBy { it.id }
        val segmentationsById = getSegmentations(resources
            .flatMap { r -> r.segments.map { s -> s.segmentationId } }.toSet())
        val segmentsById = getSegments(resources.flatMap { r -> r.segments.map { s -> s.segmentId } }.toSet())
        val eligibleAggregates = aggregates.filter { eligibleProviderIds.contains(it.key.providerId)
            && eligibleResourceIds.contains(it.key.resourceId) }
        val aggregatesByProvider = eligibleAggregates.groupBy { it.key.providerId }
        val usages = usageService.getServiceUsagesFromAggregates(aggregates) { a -> getBundle(a) }
        val sortedProviders = providers.sortedWith(Comparator
            .comparing({ Locales.select(it.nameEn, it.nameRu, locale) ?: "" }, String.CASE_INSENSITIVE_ORDER))
        return FindServiceTotalsResponseDto(listOf(FindServiceTotalsAggregateDto(
            serviceId = request.rootService.id,
            serviceName = Locales.selectFallbackToRu(request.rootService.nameEn, request.rootService.name, locale),
            serviceSlug = request.rootService.slug,
            providers = sortedProviders.map { provider -> ServiceAggregateForProviderDto(
                providerId = provider.id,
                providerName = Locales.select(provider.nameEn, provider.nameRu, locale),
                resourceTypes = prepareServiceTotalsResourceTypesResponse(
                    aggregatesByProvider[provider.id] ?: emptyList(), provider, resourceById,
                    resourceTypesById, unitsEnsemblesById, segmentationsById, segmentsById, expandByResourceType,
                    resourceSegmentationVisibility, usages, locale)
            ) }
        )))
    }

    private fun prepareResourceTypesExpansion(resources: List<ResourceModel>): Map<ResourceTypeId, Boolean> {
        val result = mutableMapOf<ResourceTypeId, Boolean>()
        val resourcesByType = resources.groupBy { it.resourceTypeId }
        resourcesByType.forEach { (resourceTypeId, resourcesOfType) ->
            result[resourceTypeId] = resourcesOfType.size > 1
        }
        return result
    }

    private fun prepareResourceSegmentationVisibility(resources: List<ResourceModel>): Map<ResourceId, Map<SegmentationId, Boolean>> {
        val result = mutableMapOf<ResourceId, MutableMap<SegmentationId, Boolean>>()
        val resourcesByType = resources.groupBy { it.resourceTypeId }
        resourcesByType.forEach { (_, resourcesOfType) ->
            val segmentsBySegmentation = mutableMapOf<SegmentationId, MutableSet<SegmentId>>()
            resourcesOfType.forEach { resource ->
                resource.segments.forEach { segmentsBySegmentation
                    .computeIfAbsent(it.segmentationId) { mutableSetOf() }.add(it.segmentId) }
            }
            resourcesOfType.forEach { resource ->
                resource.segments.forEach { result.computeIfAbsent(resource.id) { mutableMapOf() }[it
                    .segmentationId] = (segmentsBySegmentation[it.segmentationId] ?: emptySet()).size > 1 }
            }
        }
        return result
    }

    private fun prepareServiceTotalsResourceTypesResponse(
        aggregates: List<ServiceAggregateModel>,
        provider: ProviderModel,
        resourceById: Map<String, ResourceModel>,
        resourceTypesById: Map<String, ResourceTypeModel>,
        unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
        segmentationsById: Map<String, ResourceSegmentationModel>,
        segmentsById: Map<String, ResourceSegmentModel>,
        expandByResourceType: Map<ResourceTypeId, Boolean>,
        resourceSegmentationVisibility: Map<ResourceId, Map<SegmentationId, Boolean>>,
        usages: Map<ServiceAggregateKey, ServiceAggregateUsageModel>,
        locale: Locale): List<ServiceAggregateForResourceTypeDto> {
        val aggregatesByResourceType = aggregates.groupBy { resourceById[it.key.resourceId]!!.resourceTypeId }
        return aggregatesByResourceType.keys.map { resourceTypeId ->
            val resourceType = resourceTypesById[resourceTypeId]!!
            val resourceTypeAggregates = aggregatesByResourceType[resourceTypeId]!!
            val resourceTypeAggregationSettings = resourceTypeAggregationSettings(resourceType, provider)
            val resourceTypeUsageAmount = usageService.aggregateResourceTypeUsage(resourceType, usages, resourceById,
                resourceTypeAggregationSettings, unitsEnsemblesById)
            val resourceTypeQuota = usageService.aggregateResourceTypeQuota(resourceTypeAggregates,
                resourceById, unitsEnsemblesById, resourceType, resourceTypeAggregationSettings) { v -> getBundle(v) }
            val resourceTypeResources = usageService.aggregateResourceTypeResources(resourceTypeAggregates, resourceById)
            ServiceAggregateForResourceTypeDto(
                resourceTypeId = resourceTypeId,
                resourceTypeName = Locales.select(resourceType.nameEn, resourceType.nameRu, locale),
                resources = resourceTypeAggregates.map { aggregate ->
                    val resource = resourceById[aggregate.key.resourceId]!!
                    val unitsEnsemble = unitsEnsemblesById[resource.unitsEnsembleId]!!
                    val usage = usages[ServiceAggregateKey(aggregate.key.tenantId, aggregate.key.serviceId,
                        aggregate.key.providerId, aggregate.key.resourceId)]
                    prepareServiceResourceResponse(aggregate, resource, provider, resourceType, segmentationsById,
                        segmentsById, unitsEnsemble, resourceSegmentationVisibility, usage, locale)
                },
                amounts = aggregateResourceTypeAmounts(resourceTypeAggregates, provider,
                    resourceType, resourceById, unitsEnsemblesById, resourceTypeAggregationSettings, locale),
                expandResources = expandByResourceType[resourceTypeId] ?: false,
                usage = usageService.prepareResourceUsageDtoForResourceType(resourceTypeUsageAmount?.first,
                    resourceTypeUsageAmount?.second, resourceTypeQuota, resourceTypeAggregationSettings,
                    resourceType, unitsEnsemblesById, resourceTypeResources, locale)
            )
        }
    }

    private fun aggregateResourceTypeAmounts(aggregates: List<ServiceAggregateModel>,
                                             provider: ProviderModel,
                                             resourceType: ResourceTypeModel,
                                             resourceById: Map<String, ResourceModel>,
                                             unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
                                             resourceTypeAggregationSettings: AggregationSettings,
                                             locale: Locale): AggregateAmountsDto {
        var quotaValue = BigDecimal.ZERO
        var balanceValue = BigDecimal.ZERO
        var allocatedValue = BigDecimal.ZERO
        var providedValue = BigDecimal.ZERO
        var unallocatedValue = BigDecimal.ZERO
        var hasUnallocated = if (aggregates.isNotEmpty()) {
            false
        } else {
            unallocatedSupported(resourceTypeAggregationSettings)
        }
        var transferableValue = BigDecimal.ZERO
        val resourceTypeUnitsEnsemble = unitsEnsemblesById[resourceType.unitsEnsembleId]!!
        val resourceTypeBaseUnit = if (resourceType.baseUnitId != null) {
            resourceTypeUnitsEnsemble.unitById(resourceType.baseUnitId).orElseThrow()
        } else {
            UnitsComparator.getBaseUnit(resourceTypeUnitsEnsemble)
        }
        val aggregatedResources = mutableSetOf<ResourceModel>()
        aggregates.forEach { aggregate ->
            val resource = resourceById[aggregate.key.resourceId]!!
            aggregatedResources.add(resource)
            val resourceUnitsEnsemble = unitsEnsemblesById[resource.unitsEnsembleId]!!
            val resourceBaseUnit = resourceUnitsEnsemble.unitById(resource.baseUnitId).orElseThrow()
            val aggregationSettings = resourceAggregationSettings(resource.id, provider, mapOf(resource.id to resource),
                mapOf(resourceType.id to resourceType))
            val resourceQuotaValue = getBundle(aggregate)?.quota ?: BigInteger.ZERO
            val resourceBalanceValue = getBundle(aggregate)?.balance ?: BigInteger.ZERO
            val resourceAllocatedValue = getBundle(aggregate)?.allocated ?: BigInteger.ZERO
            val resourceProvidedValue = getBundle(aggregate)?.provided ?: BigInteger.ZERO
            val resourceUnallocatedValue = getBundle(aggregate)?.unallocated ?: BigInteger.ZERO
            val resourceTransferableValue = getBundle(aggregate)?.transferable ?: BigInteger.ZERO
            quotaValue = quotaValue.add(Units.convert(resourceQuotaValue.toBigDecimal(),
                resourceBaseUnit, resourceTypeBaseUnit))
            balanceValue = balanceValue.add(Units.convert(resourceBalanceValue.toBigDecimal(),
                resourceBaseUnit, resourceTypeBaseUnit))
            allocatedValue = allocatedValue.add(Units.convert(resourceAllocatedValue.toBigDecimal(),
                resourceBaseUnit, resourceTypeBaseUnit))
            providedValue = providedValue.add(Units.convert(resourceProvidedValue.toBigDecimal(),
                resourceBaseUnit, resourceTypeBaseUnit))
            unallocatedValue = unallocatedValue.add(Units.convert(resourceUnallocatedValue.toBigDecimal(),
                resourceBaseUnit, resourceTypeBaseUnit))
            hasUnallocated = hasUnallocated || unallocatedSupported(aggregationSettings)
            transferableValue += Units.convert(
                resourceTransferableValue.toBigDecimal(), resourceBaseUnit, resourceTypeBaseUnit
            )
        }
        val quota = toAmountDto(quotaValue, aggregatedResources.toList(), resourceType, resourceTypeUnitsEnsemble, locale)
        val balance = toAmountDto(balanceValue, aggregatedResources.toList(), resourceType, resourceTypeUnitsEnsemble, locale)
        val allocated = toAmountDto(allocatedValue, aggregatedResources.toList(), resourceType, resourceTypeUnitsEnsemble, locale)
        val provided = toAmountDto(providedValue, aggregatedResources.toList(), resourceType, resourceTypeUnitsEnsemble, locale)
        val unallocated = if (hasUnallocated) {
            toAmountDto(unallocatedValue, aggregatedResources.toList(), resourceType, resourceTypeUnitsEnsemble, locale)
        } else {
            null
        }
        val transferable = toAmountDto(transferableValue, aggregatedResources.toList(), resourceType, resourceTypeUnitsEnsemble, locale)
        return AggregateAmountsDto(
            quota = quota,
            balance = balance,
            allocated = allocated,
            provided = provided,
            unallocated = unallocated,
            transferable = transferable
        )
    }

    private fun unallocatedSupported(aggregationSettings: AggregationSettings): Boolean {
        return aggregationSettings.freeProvisionMode == FreeProvisionAggregationMode.UNALLOCATED_TRANSFERABLE
            || aggregationSettings.freeProvisionMode == FreeProvisionAggregationMode
            .UNALLOCATED_TRANSFERABLE_UNUSED_DEALLOCATABLE
    }

    private fun toAmountDto(amount: BigDecimal, resources: List<ResourceModel>, resourceType: ResourceTypeModel,
                            unitsEnsemble: UnitsEnsembleModel, locale: Locale): AmountDto {
        val allowedSortedUnits = if (resources.isNotEmpty()) {
            val allowedUnitIds = resources.flatMap { it.resourceUnits.allowedUnitIds }.toSet()
            unitsEnsemble.units.filter { allowedUnitIds.contains(it.id) }
                .sortedWith(UnitsComparator.INSTANCE).toList()
        } else {
            unitsEnsemble.units.sortedWith(UnitsComparator.INSTANCE).toList()
        }
        if (allowedSortedUnits.isEmpty()) {
            val resourceKeys = resources.map { it.key }
            throw IllegalStateException(
                "No allowed units found for resource type ${resourceType.key} and resources $resourceKeys")
        }
        val baseUnit = if (resources.isNotEmpty()) {
            resources.map { unitsEnsemble.unitById(it.baseUnitId)
                .orElseThrow { IllegalStateException("Base unit not found for resource ${it.key} and resource " +
                    "type ${resourceType.key}") } }.distinct().sortedWith(UnitsComparator.INSTANCE).first()
        } else {
            if (resourceType.baseUnitId != null) {
                unitsEnsemble.unitById(resourceType.baseUnitId).orElseThrow()
            } else {
                UnitsComparator.getBaseUnit(unitsEnsemble)
            }
        }
        val minAllowedUnit = allowedSortedUnits.first()
        return QuotasHelper.getAmountDtoForAggregates(
            amount, allowedSortedUnits, baseUnit, minAllowedUnit, null, locale
        )
    }

    private fun toAmounts(aggregate: ServiceAggregateModel?,
                          resource: ResourceModel,
                          unitsEnsemble: UnitsEnsembleModel,
                          aggregationSettings: AggregationSettings,
                          unit: UnitModel?,
                          locale: Locale): AggregateAmountsDto {
        val quotaValue = getBundle(aggregate)?.quota ?: BigInteger.ZERO
        val balanceValue = getBundle(aggregate)?.balance ?: BigInteger.ZERO
        val allocatedValue = getBundle(aggregate)?.allocated ?: BigInteger.ZERO
        val providedValue = getBundle(aggregate)?.provided ?: BigInteger.ZERO
        val unallocatedValue = getBundle(aggregate)?.unallocated ?: BigInteger.ZERO
        val transferableValue = getBundle(aggregate)?.transferable ?: BigInteger.ZERO
        val quota = QuotasHelper.getAmountDtoForAggregates(quotaValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val balance = QuotasHelper.getAmountDtoForAggregates(balanceValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val allocated = QuotasHelper.getAmountDtoForAggregates(allocatedValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val provided = QuotasHelper.getAmountDtoForAggregates(providedValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val unallocated = if (unallocatedSupported(aggregationSettings)) {
            QuotasHelper.getAmountDtoForAggregates(unallocatedValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        } else {
            null
        }
        val transferable = QuotasHelper.getAmountDtoForAggregates(transferableValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        return AggregateAmountsDto(
            quota = quota,
            balance = balance,
            allocated = allocated,
            provided = provided,
            unallocated = unallocated,
            transferable = transferable
        )
    }

    private fun getBundle(aggregate: ServiceAggregateModel?): AggregateBundle? {
        if (aggregate?.exactAmounts?.total == null) {
            return aggregate?.exactAmounts?.own
        }
        return aggregate.exactAmounts.total
    }

    private fun toAggregationMode(mode: FreeProvisionAggregationMode): FreeProvisionAggregationModeDto {
        return when (mode) {
            FreeProvisionAggregationMode.NONE -> FreeProvisionAggregationModeDto.NONE
            FreeProvisionAggregationMode.UNALLOCATED_TRANSFERABLE ->
                FreeProvisionAggregationModeDto.UNALLOCATED_TRANSFERABLE
            FreeProvisionAggregationMode.UNALLOCATED_TRANSFERABLE_UNUSED_DEALLOCATABLE ->
                FreeProvisionAggregationModeDto.UNALLOCATED_TRANSFERABLE_UNUSED_DEALLOCATABLE
            FreeProvisionAggregationMode.UNDERUTILIZED_TRANSFERABLE ->
                FreeProvisionAggregationModeDto.UNDERUTILIZED_TRANSFERABLE
        }
    }

    private fun toUsageMode(mode: UsageMode): ResourceUsageModeDto {
        return when(mode) {
            UsageMode.UNDEFINED -> ResourceUsageModeDto.UNDEFINED
            UsageMode.TIME_SERIES -> ResourceUsageModeDto.TIME_SERIES
            UsageMode.UNUSED_ESTIMATION_VALUE -> ResourceUsageModeDto.UNUSED_ESTIMATION_VALUE
        }
    }

    private suspend fun prepareSubtreeAmountsResponse(request: RankSubtreeAmounts,
                                                      preValidated: RankSubtreeAmountsRequest,
                                                      aggregates: List<ServiceDenormalizedAggregateModel>,
                                                      locale: Locale): RankSubtreeAmountsResponseDto {
        return RankSubtreeAmountsResponseDto(
            aggregates = aggregatesPage(request, preValidated, aggregates, locale),
            nextPageToken = nextPageToken(preValidated, aggregates)
        )
    }

    private suspend fun prepareSubtreeAmountsApiResponse(
        request: RankSubtreeAmounts, preValidated: RankSubtreeAmountsRequest,
        aggregates: List<ServiceDenormalizedAggregateModel>, includeUsage: Boolean,
        includeUsageRaw: Boolean): RankSubtreeAmountsApiResponseDto
    {
        return RankSubtreeAmountsApiResponseDto(
            aggregates = aggregatesApiPage(request, preValidated, aggregates, includeUsage, includeUsageRaw),
            nextPageToken = nextPageToken(preValidated, aggregates)
        )
    }

    private fun nextPageToken(
        preValidated: RankSubtreeAmountsRequest,
        aggregates: List<ServiceDenormalizedAggregateModel>
    ): String? {
        if (aggregates.size.toLong() == preValidated.limit + 1 && preValidated.limit > 0) {
            val last = aggregates[(preValidated.limit - 1).toInt()]
            val sortingFieldValue = when (preValidated.sortingParams.field) {
                ALLOCATED -> last.exactAmounts.own?.allocated
                BALANCE -> last.exactAmounts.own?.balance
                QUOTA -> last.exactAmounts.own?.quota
                PROVIDED -> last.exactAmounts.own?.provided
                UNALLOCATED -> last.exactAmounts.own?.unallocated
                TRANSFERABLE -> last.transferable
                UNUSED_ESTIMATION -> last.exactAmounts.own?.unusedEst
                UNDERUTILIZED_ESTIMATION -> last.exactAmounts.own?.underutilizedEst
            } ?: return null

            return ContinuationTokens.encode(
                RankSubtreeAmountsPageToken(sortingFieldValue.toString(), last.key.serviceId),
                continuationTokenWriter
            )
        }
        return null
    }

    private suspend fun aggregatesPage(request: RankSubtreeAmounts,
                                       preValidated: RankSubtreeAmountsRequest,
                                       aggregates: List<ServiceDenormalizedAggregateModel>,
                                       locale: Locale): List<RankSubtreeAmountsAggregateDto> {
        if (preValidated.limit <= 0) {
            return emptyList()
        }
        val page = if (aggregates.size.toLong() == preValidated.limit + 1) {
            aggregates.dropLast(1)
        } else {
            aggregates
        }
        if (page.isEmpty()) {
            return listOf()
        }
        val serviceIds = aggregates.map { it.key.serviceId }.distinct()
        val services = getServices(serviceIds)
        val folders = getFoldersByServices(serviceIds)
        val provider = getProvider(request.resource.providerId)!!
        val resourceType = getResourceType(request.resource.resourceTypeId)!!
        val segmentations = getSegmentations(request.resource.segments.map { it.segmentationId })
        val segments = getSegments(request.resource.segments.map { it.segmentId })
        val aggregationSettings = resourceAggregationSettings(request.resource.id, provider,
            mapOf(request.resource.id to request.resource), mapOf(resourceType.id to resourceType))
        val usages = usageService.getServiceUsagesFromDenormalizedAggregates(page)
        return page.map {
            val usage = usages[ServiceAggregateKey(Tenants.DEFAULT_TENANT_ID, it.key.serviceId,
                it.providerId, it.key.resourceId)]
            prepareSubtreeAmountsAggregate(services, folders, provider, resourceType, request.resource, segmentations,
                segments, aggregationSettings, request.unitsEnsemble, it, usage, request.unit, locale)
        }
    }

    private suspend fun aggregatesApiPage(
        request: RankSubtreeAmounts, preValidated: RankSubtreeAmountsRequest,
        aggregates: List<ServiceDenormalizedAggregateModel>,
        includeUsage: Boolean,
        includeUsageRaw: Boolean): List<RankSubtreeAmountsAggregateApiDto>
    {
        if (preValidated.limit <= 0) {
            return emptyList()
        }
        val page = if (aggregates.size.toLong() == preValidated.limit + 1) {
            aggregates.dropLast(1)
        } else {
            aggregates
        }
        if (page.isEmpty()) {
            return listOf()
        }
        val usages = if (includeUsage) {
            usageService.getServiceUsagesFromDenormalizedAggregates(page)
        } else {
            emptyMap()
        }
        return page.map {
            val usage = usages[ServiceAggregateKey(Tenants.DEFAULT_TENANT_ID, it.key.serviceId,
                it.providerId, it.key.resourceId)]
            prepareSubtreeAmountsAggregateApi(request.resource, request.unitsEnsemble, it, usage,
                request.provider, request.resourceType, includeUsageRaw)
        }
    }

    private fun prepareSubtreeAmountsAggregate(services: Map<Long, ServiceMinimalModel>,
                                               folders: Map<Long, List<FolderModel>>,
                                               provider: ProviderModel,
                                               resourceType: ResourceTypeModel,
                                               resource: ResourceModel,
                                               segmentations: Map<String, ResourceSegmentationModel>,
                                               segments: Map<String, ResourceSegmentModel>,
                                               aggregationSettings: AggregationSettings,
                                               unitsEnsemble: UnitsEnsembleModel,
                                               aggregate: ServiceDenormalizedAggregateModel,
                                               usage: ServiceAggregateUsageModel?,
                                               unit: UnitModel?,
                                               locale: Locale): RankSubtreeAmountsAggregateDto {
        val service = services[aggregate.key.serviceId]!!
        val serviceFolders = folders[service.id] ?: emptyList()
        return RankSubtreeAmountsAggregateDto(
            serviceId = service.id,
            serviceName = Locales.selectFallbackToRu(service.nameEn, service.name, locale),
            serviceSlug = service.slug,
            folders = serviceFolders.map { RankSubtreeAmountsAggregateFolderDto(it.id, it.displayName) },
            providers = listOf(AggregateForProviderDto(
                providerId = provider.id,
                providerName = Locales.select(provider.nameEn, provider.nameRu, locale),
                resourceTypes = listOf(AggregateForResourceTypeDto(
                    resourceTypeId = resourceType.id,
                    resourceTypeName = Locales.select(resourceType.nameEn, resourceType.nameRu, locale),
                    resources = listOf(prepareSubtreeResourceResponse(aggregate, resource, segmentations, segments,
                        aggregationSettings, unitsEnsemble, usage, unit, locale))
                ))
            ))
        )
    }

    private fun prepareSubtreeAmountsAggregateApi(
        resource: ResourceModel, unitsEnsemble: UnitsEnsembleModel,
        aggregate: ServiceDenormalizedAggregateModel,
        usage: ServiceAggregateUsageModel?,
        provider: ProviderModel,
        resourceType: ResourceTypeModel,
        includeUsageRaw: Boolean): RankSubtreeAmountsAggregateApiDto
    {
        val quota = aggregate.exactAmounts.own?.quota ?: BigInteger.ZERO
        val balance = aggregate.exactAmounts.own?.balance ?: BigInteger.ZERO
        val provided = aggregate.exactAmounts.own?.provided ?: BigInteger.ZERO
        val allocated = aggregate.exactAmounts.own?.allocated ?: BigInteger.ZERO
        val unallocated = aggregate.exactAmounts.own?.unallocated ?: BigInteger.ZERO
        val transferable = aggregate.exactAmounts.own?.transferable ?: BigInteger.ZERO
        val convertedQuota = Units.convertToApi(quota, resource, unitsEnsemble)
        val convertedBalance = Units.convertToApi(balance, resource, unitsEnsemble)
        val convertedProvided = Units.convertToApi(provided, resource, unitsEnsemble)
        val convertedAllocated = Units.convertToApi(allocated, resource, unitsEnsemble)
        val convertedUnallocated = Units.convertToApi(unallocated, resource, unitsEnsemble)
        val convertedTransferable = Units.convertToApi(transferable, resource, unitsEnsemble)
        val aggregationSettings = resourceAggregationSettings(resource.id, provider,
            mapOf(resource.id to resource), mapOf(resourceType.id to resourceType))
        return RankSubtreeAmountsAggregateApiDto(
            serviceId = aggregate.key.serviceId,
            amounts = AggregateAmountsApiDto(
                quota = AggregateAmountApiDto(convertedQuota.t1.toPlainString(), convertedQuota.t2.key),
                balance = AggregateAmountApiDto(convertedBalance.t1.toPlainString(), convertedBalance.t2.key),
                provided = AggregateAmountApiDto(convertedProvided.t1.toPlainString(), convertedProvided.t2.key),
                allocated = AggregateAmountApiDto(convertedAllocated.t1.toPlainString(), convertedAllocated.t2.key),
                unallocated = AggregateAmountApiDto(convertedUnallocated.t1.toPlainString(), convertedAllocated.t2.key),
                transferable = AggregateAmountApiDto(convertedTransferable.t1.toPlainString(), convertedAllocated.t2.key),
            ),
            lastUpdate = aggregate.lastUpdate,
            usage = usageService.prepareServiceAggregateUsageApiDto(usage?.exactAmounts?.own,
                usage?.lastUpdate, aggregate.exactAmounts.own, aggregationSettings, resource, unitsEnsemble, includeUsageRaw)
        )
    }

    private fun prepareSubtreeResourceResponse(aggregate: ServiceDenormalizedAggregateModel,
                                               resource: ResourceModel,
                                               segmentations: Map<String, ResourceSegmentationModel>,
                                               segments: Map<String, ResourceSegmentModel>,
                                               aggregationSettings: AggregationSettings,
                                               unitsEnsemble: UnitsEnsembleModel,
                                               usage: ServiceAggregateUsageModel?,
                                               unit: UnitModel?,
                                               locale: Locale): AggregateForResourceDto {
        return AggregateForResourceDto(
            resourceId = resource.id,
            resourceName = Locales.select(resource.nameEn, resource.nameRu, locale),
            segmentations = resource.segments.map {
                val segmentation = segmentations[it.segmentationId]!!
                val segment = segments[it.segmentId]!!
                AggregateResourceSegmentDto(
                    segmentationId = segmentation.id,
                    segmentationName = Locales.select(segmentation.nameEn, segmentation.nameRu, locale),
                    groupingOrder = segmentation.groupingOrder.toLong(),
                    segmentId = segment.id,
                    segmentName = Locales.select(segment.nameEn, segment.nameRu, locale),
                    segmentUncommon = segment.uncommon.orElse(false)
                )
            },
            freeProvisionAggregationMode = toAggregationMode(aggregationSettings.freeProvisionMode),
            usageMode = toUsageMode(aggregationSettings.usageMode ?: UsageMode.UNDEFINED),
            amounts = toSubtreeAmounts(aggregate, resource, unitsEnsemble, aggregationSettings, unit, locale),
            lastUpdate = aggregate.lastUpdate,
            usage = usageService.prepareResourceUsageDto(usage?.exactAmounts?.own, usage?.lastUpdate,
                aggregate.exactAmounts.own, aggregationSettings, resource, unitsEnsemble, unit, locale)
        )
    }

    private fun toSubtreeAmounts(aggregate: ServiceDenormalizedAggregateModel,
                                 resource: ResourceModel,
                                 unitsEnsemble: UnitsEnsembleModel,
                                 aggregationSettings: AggregationSettings,
                                 unit: UnitModel?,
                                 locale: Locale): AggregateAmountsDto {
        val quotaValue = aggregate.exactAmounts.own?.quota ?: BigInteger.ZERO
        val balanceValue = aggregate.exactAmounts.own?.balance ?: BigInteger.ZERO
        val allocatedValue = aggregate.exactAmounts.own?.allocated ?: BigInteger.ZERO
        val providedValue = aggregate.exactAmounts.own?.provided ?: BigInteger.ZERO
        val unallocatedValue = aggregate.exactAmounts.own?.unallocated ?: BigInteger.ZERO
        val transferableValue = aggregate.exactAmounts.own?.transferable ?: BigInteger.ZERO
        val quota = QuotasHelper.getAmountDtoForAggregates(quotaValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val balance = QuotasHelper.getAmountDtoForAggregates(balanceValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val allocated = QuotasHelper.getAmountDtoForAggregates(allocatedValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val provided = QuotasHelper.getAmountDtoForAggregates(providedValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        val unallocated = if (unallocatedSupported(aggregationSettings)) {
            QuotasHelper.getAmountDtoForAggregates(unallocatedValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        } else {
            null
        }
        val transferable = QuotasHelper.getAmountDtoForAggregates(transferableValue.toBigDecimal(), resource, unitsEnsemble, unit, locale)
        return AggregateAmountsDto(
            quota = quota,
            balance = balance,
            allocated = allocated,
            provided = provided,
            unallocated = unallocated,
            transferable = transferable
        )
    }

    private fun preValidateRequest(request: FindSubtreeTotalRequestDto, locale: Locale): Result<FindSubtreeTotalRequest> {
        val errors = ErrorCollection.builder()
        if (request.rootServiceId == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (request.resourceId == null) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            if (!Uuids.isValidUuid(request.resourceId)) {
                errors.addError("resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)))
            }
        }
        if (request.unitId != null && !Uuids.isValidUuid(request.unitId)) {
            errors.addError("unitId", TypedError.invalid(messages
                .getMessage("errors.unit.not.found", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(FindSubtreeTotalRequest(request.rootServiceId!!, request.resourceId!!, null, request.unitId))
    }

    private fun preValidateApiRequest(request: FindSubtreeTotalApiRequestDto, locale: Locale): Result<FindSubtreeTotalRequest> {
        val errors = ErrorCollection.builder()
        if (request.rootServiceId == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (request.resourceId == null) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            if (!Uuids.isValidUuid(request.resourceId)) {
                errors.addError("resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)))
            }
        }
        if (request.providerId == null) {
            errors.addError("providerId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            if (!Uuids.isValidUuid(request.providerId)) {
                errors.addError("providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.not.found", null, locale)))
            }
        }
        if (request.includeUsageRaw == true && request.includeUsage != true) {
            errors.addError("includeUsageRaw", TypedError.invalid(messages
                .getMessage("errors.invalid.flag", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(FindSubtreeTotalRequest(request.rootServiceId!!, request.resourceId!!, request.providerId!!, null))
    }

    private fun preValidateRequest(request: RankSubtreeAmountsRequestDto,
                                   locale: Locale): Result<RankSubtreeAmountsRequest> {
        val errors = ErrorCollection.builder()
        if (request.rootServiceId == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (request.resourceId == null) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            if (!Uuids.isValidUuid(request.resourceId)) {
                errors.addError("resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)))
            }
        }
        if (request.unitId != null && !Uuids.isValidUuid(request.unitId)) {
            errors.addError("unitId", TypedError.invalid(messages
                .getMessage("errors.unit.not.found", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val pageRequest = PageRequest(request.from, request.limit)
        val validatedPageRequest: Result<PageRequest.Validated<RankSubtreeAmountsPageToken>>
            = pageRequest.validate(continuationTokenReader, messages, locale)
        val validatedPageToken: Result<PageRequest.Validated<PageToken>> = validatedPageRequest
            .andThen { validatePageToken(it, locale) }
        val sortingParams = request.sortingParams ?: RankSubtreeSortingParamsDto.default()
        return validatedPageToken.apply { r ->
            RankSubtreeAmountsRequest(
                request.rootServiceId!!, request.resourceId!!,
                r.continuationToken.orElse(null), r.limit.toLong(), null, sortingParams, request.unitId
            )
        }
    }

    private fun preValidateApiRequest(request: RankSubtreeAmountsApiRequestDto,
                                      locale: Locale): Result<RankSubtreeAmountsRequest> {
        val errors = ErrorCollection.builder()
        if (request.rootServiceId == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (request.resourceId == null) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            if (!Uuids.isValidUuid(request.resourceId)) {
                errors.addError("resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)))
            }
        }
        if (request.providerId == null) {
            errors.addError("providerId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            if (!Uuids.isValidUuid(request.providerId)) {
                errors.addError("providerId", TypedError.invalid(messages
                    .getMessage("errors.provider.not.found", null, locale)))
            }
        }
        if (request.includeUsageRaw == true && request.includeUsage != true) {
            errors.addError("includeUsageRaw", TypedError.invalid(messages
                .getMessage("errors.invalid.flag", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val pageRequest = PageRequest(request.from, request.limit)
        val validatedPageRequest: Result<PageRequest.Validated<RankSubtreeAmountsPageToken>>
            = pageRequest.validate(continuationTokenReader, messages, locale)
        val validatedPageToken: Result<PageRequest.Validated<PageToken>> = validatedPageRequest
            .andThen { validatePageToken(it, locale) }
        val sortingParams = request.sortingParams ?: RankSubtreeSortingParamsDto.default()
        return validatedPageToken.apply { r ->
            RankSubtreeAmountsRequest(
                request.rootServiceId!!, request.resourceId!!,
                r.continuationToken.orElse(null), r.limit.toLong(), request.providerId!!, sortingParams, null
            )
        }
    }

    private fun preValidateRequest(request: FindServiceTotalsRequestDto,
                                   locale: Locale): Result<FindServiceTotalsRequest> {
        val errors = ErrorCollection.builder()
        if (request.rootServiceId == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (request.filter != null) {
            if (request.expandedFilter != null) {
                errors.addError("filter", TypedError.invalid(messages
                    .getMessage("errors.aggregation.both.filters.not.allowed", null, locale)))
            }
            if (request.filter.providerId == null) {
                errors.addError("filter.providerId", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)))
            } else {
                if (!Uuids.isValidUuid(request.filter.providerId)) {
                    errors.addError("filter.providerId", TypedError.invalid(messages
                        .getMessage("errors.provider.not.found", null, locale)))
                }
            }
            if (request.filter.segments != null) {
                request.filter.segments.forEachIndexed { i, s ->
                    if (s == null) {
                        errors.addError("filter.segments[$i]", TypedError.invalid(messages
                            .getMessage("errors.field.is.required", null, locale)))
                    } else {
                        if (s.segmentationId == null) {
                            errors.addError("filter.segments[$i].segmentationId", TypedError.invalid(messages
                                .getMessage("errors.field.is.required", null, locale)))
                        } else {
                            if (!Uuids.isValidUuid(s.segmentationId)) {
                                errors.addError("filter.segments[$i].segmentationId", TypedError.invalid(messages
                                    .getMessage("errors.resource.segmentation.not.found", null, locale)))
                            }
                        }
                        if (s.segmentId == null) {
                            errors.addError("filter.segments[$i].segmentId", TypedError.invalid(messages
                                .getMessage("errors.field.is.required", null, locale)))
                        } else {
                            if (!Uuids.isValidUuid(s.segmentId)) {
                                errors.addError("filter.segments[$i].segmentId", TypedError.invalid(messages
                                    .getMessage("errors.resource.segment.not.found", null, locale)))
                            }
                        }
                    }
                }
            }
        }
        if (request.expandedFilter != null) {
            if (request.expandedFilter.filters != null) {
                if (request.expandedFilter.filters.isEmpty()) {
                    errors.addError("expandedFilter.singleFilter", TypedError.invalid(messages
                        .getMessage("errors.field.is.required", null, locale)))
                }
                request.expandedFilter.filters.forEach{ singleFilter ->
                    preValidateSingleExpandedFilter(singleFilter, errors, locale)
                }
            }
        }

        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }

        val filter = if (request.filter != null) {
            val segments = if (request.filter.segments != null) {
                request.filter.segments.map { FindServiceTotalsRequestFilterSegment(it!!.segmentationId!!, it.segmentId!!) }
            } else {
                null
            }
            FindServiceTotalsRequestFilter(request.filter.providerId!!, segments)
        } else {
            null
        }

        val expandedFilters = if (request.expandedFilter != null) {
            val filters = request.expandedFilter.filters?.map { singleFilter ->
                val segments = if (singleFilter!!.segments != null) {
                    singleFilter.segments!!.map {
                        @Suppress("UNCHECKED_CAST") // for 'as Set<SegmentId>' (validated they`re not null)
                        SegmentationAndSegmentsIds(
                            it!!.segmentationId!!,
                            it.segmentIds!! as Set<SegmentId>
                        )
                    }.toSet()
                } else {
                    setOf()
                }
                val resourceTypeIds = if (singleFilter.resourceTypeIds != null) {
                    singleFilter.resourceTypeIds.map { it!! }.toSet()
                } else {
                    setOf()
                }

                FindServiceTotalsSingleExpandedFilter(singleFilter.providerId!!, segments, resourceTypeIds)
            }?.toSet() ?: setOf()

            FindServiceTotalsExpandedFilter(filters)
        } else {
            null
        }

        return Result.success(FindServiceTotalsRequest(request.rootServiceId!!, filter, expandedFilters))
    }

    private fun preValidateSingleExpandedFilter(
        singleFilter: FindServiceTotalsSingleExpandedFilterDto?,
        errors: ErrorCollection.Builder,
        locale: Locale
    ) {
        if (singleFilter == null) {
            errors.addError(
                "expandedFilter.singleFilter", TypedError.invalid(
                    messages.getMessage("errors.field.is.required", null, locale)
                )
            )
        } else {
            if (singleFilter.providerId == null) {
                errors.addError(
                    "expandedFilter.providerId", TypedError.invalid(
                        messages.getMessage("errors.field.is.required", null, locale)
                    )
                )
            } else {
                if (!Uuids.isValidUuid(singleFilter.providerId)) {
                    errors.addError(
                        "expandedFilter.providerId", TypedError.invalid(
                            messages.getMessage("errors.provider.not.found", null, locale)
                        )
                    )
                }
            }
            if (singleFilter.segments != null) {
                for (segmentsDto in singleFilter.segments) {
                    if (segmentsDto == null) {
                        errors.addError(
                            "expandedFilter.segments", TypedError.invalid(
                                messages.getMessage("errors.field.is.required", null, locale)
                            )
                        )
                    } else {
                        if (segmentsDto.segmentationId == null) {
                            errors.addError(
                                "expandedFilter.segments.segmentationId", TypedError.invalid(
                                    messages.getMessage("errors.field.is.required", null, locale)
                                )
                            )
                        } else {
                            if (!Uuids.isValidUuid(segmentsDto.segmentationId)) {
                                errors.addError(
                                    "expandedFilter.segments.segmentationId", TypedError.invalid(
                                        messages.getMessage(
                                            "errors.resource.segmentation.not.found",
                                            null,
                                            locale
                                        )
                                    )
                                )
                            }
                        }

                        if (segmentsDto.segmentIds == null) {
                            errors.addError(
                                "expandedFilter.segments.segmentIds", TypedError.invalid(
                                    messages.getMessage("errors.field.is.required", null, locale)
                                )
                            )
                        } else {
                            for (segmentId in segmentsDto.segmentIds) {
                                if (segmentId == null) {
                                    errors.addError(
                                        "expandedFilter.segments.segmentId", TypedError.invalid(
                                            messages.getMessage("errors.field.is.required", null, locale)
                                        )
                                    )
                                } else {
                                    if (!Uuids.isValidUuid(segmentId)) {
                                        errors.addError(
                                            "expandedFilter.segments.segmentId", TypedError.invalid(
                                                messages.getMessage(
                                                    "errors.resource.segment.not.found", null, locale
                                                )
                                            )
                                        )
                                    }
                                }
                            }
                        }
                    }
                }
            }

            if (singleFilter.resourceTypeIds != null) {
                for (resourceTypeId in singleFilter.resourceTypeIds) {
                    if (resourceTypeId == null) {
                        errors.addError(
                            "expandedFilter.resourceTypeId", TypedError.invalid(
                                messages.getMessage("errors.field.is.required", null, locale)
                            )
                        )
                    } else if (!Uuids.isValidUuid(resourceTypeId)) {
                        errors.addError(
                            "expandedFilter.resourceTypeId", TypedError.invalid(
                                messages.getMessage(
                                    "errors.resource.segment.not.found", null, locale
                                )
                            )
                        )
                    }
                }
            }
        }
    }

    private fun preValidateRequest(
        request: GetProvidersByServiceRequestDto, locale: Locale): Result<GetProvidersByServiceRequest> {
        val errors = ErrorCollection.builder()
        if (request.rootServiceId == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(GetProvidersByServiceRequest(request.rootServiceId!!))
    }

    private fun preValidateRequest(
        request: GetResourceSelectionDto, locale: Locale): Result<GetResourceSelectionTreeRequest> {
        val errors = ErrorCollection.builder()
        if (request.rootServiceId == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (request.providerId == null) {
            errors.addError("providerId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(GetResourceSelectionTreeRequest(request.rootServiceId!!, request.providerId!!))
    }

    private fun validatePageToken(
        pageToken: PageRequest.Validated<RankSubtreeAmountsPageToken>, locale: Locale
    ): Result<PageRequest.Validated<PageToken>> {
        if (pageToken.continuationToken.isEmpty) {
            return Result.success(PageRequest.Validated(null, pageToken.limit))
        }
        if (pageToken.continuationToken.get().serviceId == null
            || pageToken.continuationToken.get().sortingField == null) {
            return Result.failure(ErrorCollection.builder().addError("from", TypedError.invalid(messages
                .getMessage("errors.resource.not.found", null, locale))).build())
        }
        return Result.success(PageRequest.Validated(PageToken(pageToken.continuationToken.get().sortingField!!,
            pageToken.continuationToken.get().serviceId!!), pageToken.limit))
    }

    private suspend fun validate(request: FindSubtreeTotalRequest, locale: Locale): Result<FindSubtreeTotal> {
        val errors = ErrorCollection.builder()
        val service = getService(request.rootServiceId)
        if (service == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.service.not.found", null, locale)))
        }
        val resource = getResource(request.resourceId)
        if (resource == null || resource.isDeleted) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.resource.not.found", null, locale)))
        }
        if (resource != null && request.providerId != null && resource.providerId != request.providerId) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.resource.not.found", null, locale)))
        }
        val unitsEnsemble = if (resource != null) {
            getUnitsEnsemble(resource.unitsEnsembleId)!!
        } else {
            null
        }
        val unit = if (unitsEnsemble != null && request.unitId != null) {
            val unitO = unitsEnsemble.unitById(request.unitId)
            if (unitO.isEmpty || unitO.get().isDeleted) {
                errors.addError("unitId", TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)))
                null
            } else {
                unitO.get()
            }
        } else {
            null
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        val provider = providersLoader.getProviderByIdImmediate(resource!!.providerId, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElseThrow()
        val resourceType = resourceTypesLoader.getResourceTypeByIdImmediate(resource.resourceTypeId,
            Tenants.DEFAULT_TENANT_ID).awaitSingle().orElseThrow()
        return Result.success(FindSubtreeTotal(service!!, resource, provider, resourceType, unitsEnsemble!!, unit))
    }

    private suspend fun validate(request: RankSubtreeAmountsRequest, locale: Locale): Result<RankSubtreeAmounts> {
        val errors = ErrorCollection.builder()
        val service = getService(request.rootServiceId)
        if (service == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.service.not.found", null, locale)))
        }
        val resource = getResource(request.resourceId)
        if (resource == null || resource.isDeleted) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.resource.not.found", null, locale)))
        }
        if (resource != null && request.providerId != null && resource.providerId != request.providerId) {
            errors.addError("resourceId", TypedError.invalid(messages
                .getMessage("errors.resource.not.found", null, locale)))
        }
        val provider = if (resource != null) {
            providersLoader.getProviderByIdImmediate(resource.providerId, Tenants.DEFAULT_TENANT_ID)
                .awaitSingle().orElseThrow()
        } else {
            null
        }
        val resourceType = if (resource != null) {
            resourceTypesLoader.getResourceTypeByIdImmediate(resource.resourceTypeId,
                Tenants.DEFAULT_TENANT_ID).awaitSingle().orElseThrow()
        } else {
            null
        }
        val aggregationSettings = if (resource != null && provider != null && resourceType != null) {
            resourceAggregationSettings(resource.id, provider, mapOf(resource.id to resource),
                mapOf(resourceType.id to resourceType))
        } else {
            null
        }
        if (aggregationSettings != null) {
            val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
            if (usageMode != UsageMode.UNUSED_ESTIMATION_VALUE && request.sortingParams.field == UNUSED_ESTIMATION) {
                errors.addError("sortingParams.field", TypedError.invalid(messages
                    .getMessage("errors.unsupported.sort.mode", null, locale)))
            }
            if (usageMode != UsageMode.TIME_SERIES && request.sortingParams.field == UNDERUTILIZED_ESTIMATION) {
                errors.addError("sortingParams.field", TypedError.invalid(messages
                    .getMessage("errors.unsupported.sort.mode", null, locale)))
            }
        }
        val unitsEnsemble = if (resource != null) {
            getUnitsEnsemble(resource.unitsEnsembleId)!!
        } else {
            null
        }
        val unit = if (unitsEnsemble != null && request.unitId != null) {
            val unitO = unitsEnsemble.unitById(request.unitId)
            if (unitO.isEmpty || unitO.get().isDeleted) {
                errors.addError("unitId", TypedError.invalid(messages
                    .getMessage("errors.unit.not.found", null, locale)))
                null
            } else {
                unitO.get()
            }
        } else {
            null
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(RankSubtreeAmounts(service!!, resource!!, provider!!, resourceType!!, unitsEnsemble!!, unit))
    }

    private suspend fun validate(request: FindServiceTotalsRequest, locale: Locale): Result<FindServiceTotals> {
        val errors = ErrorCollection.builder()
        val service = getService(request.rootServiceId)
        if (service == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.service.not.found", null, locale)))
        }
        val filter = validateFilter(request, locale, errors)
        val expandedFilters = validateExpandedFilters(request, locale, errors)
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(FindServiceTotals(service!!, filter, expandedFilters))
    }

    private suspend fun validateFilter(
        request: FindServiceTotalsRequest,
        locale: Locale,
        errors: ErrorCollection.Builder
    ) = if (request.filter != null) {
        var noProviderErrors = true
        val provider = getProvider(request.filter.providerId)
        if (provider == null || provider.isDeleted) {
            errors.addError("filter.providerId", TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale)))
            noProviderErrors = false
        }
        var noSegmentsErrors = true
        val filterSegments = if (request.filter.segments != null) {
            val segmentationIds = request.filter.segments.map { it.segmentationId }.distinct()
            val segmentIds = request.filter.segments.map { it.segmentId }.distinct()
            val segmentationsById = getSegmentations(segmentationIds)
            val segmentsById = getSegments(segmentIds)
            val segments = mutableListOf<FindServiceTotalsFilterSegment>()
            val processedSegments = mutableSetOf<FindServiceTotalsRequestFilterSegment>()
            request.filter.segments.forEachIndexed { i, s ->
                val segmentation = segmentationsById[s.segmentationId]
                val segment = segmentsById[s.segmentId]
                var noSegmentErrors = true
                if (segmentation == null || segmentation.isDeleted
                    || (provider != null && segmentation.providerId != provider.id)) {
                    errors.addError("filter.segments[$i].segmentationId", TypedError.invalid(messages
                        .getMessage("errors.resource.segmentation.not.found", null, locale)))
                    noSegmentErrors = false
                }
                if (segment == null || segment.isDeleted
                    || (segmentation != null && segment.segmentationId != segmentation.id)) {
                    errors.addError("filter.segments[$i].segmentId", TypedError.invalid(messages
                        .getMessage("errors.resource.segment.not.found", null, locale)))
                    noSegmentErrors = false
                }
                if (noSegmentErrors) {
                    if (!processedSegments.contains(s)) {
                        processedSegments.add(s)
                        segments.add(FindServiceTotalsFilterSegment(segmentation!!, segment!!))
                    }
                } else {
                    noSegmentsErrors = false
                }
            }
            segments.toList()
        } else {
            null
        }
        if (noProviderErrors && noSegmentsErrors) {
            FindServiceTotalsFilter(provider!!, filterSegments)
        } else {
            null
        }
    } else {
        null
    }

    private suspend fun validateExpandedFilters(
        request: FindServiceTotalsRequest,
        locale: Locale,
        errors: ErrorCollection.Builder
    ) = if (request.expandedFilter != null) {

        val segmentationIds = request.expandedFilter.filters.flatMap { f -> f.segments.map { it.segmentationId } }.distinct()
        val segmentIds = request.expandedFilter.filters.flatMap{ f -> f.segments.flatMap { it.segmentIds } }.distinct()
        val resourceTypeIds = request.expandedFilter.filters.flatMap { it.resourceTypeIds }.distinct()

        val segmentationsByIds = getSegmentations(segmentationIds)
        val segmentsByIds = getSegments(segmentIds)
        val resourceTypesByIds = getResourceTypes(resourceTypeIds).associateBy { it.id }

        request.expandedFilter.filters.forEach { f ->
            val provider = getProvider(f.providerId)
            if (provider == null || provider.isDeleted) {
                errors.addError(
                    "expandedFilter.providerId", TypedError.invalid(
                        messages.getMessage("errors.provider.not.found", null, locale)
                    )
                )
            }

            f.segments.forEach { s ->
                val segmentation = segmentationsByIds[s.segmentationId]
                if (segmentation == null || segmentation.isDeleted ||
                    (provider != null && segmentation.providerId != provider.id)
                ) {
                    errors.addError(
                        "expandedFilter.segments.segmentationId", TypedError.invalid(
                            messages.getMessage("errors.resource.segmentation.not.found", null, locale)
                        )
                    )
                }
                s.segmentIds.forEach { segmentId ->
                    val segment = segmentsByIds[segmentId]
                    if (segment == null || segment.isDeleted ||
                        (segmentation != null && segment.segmentationId != segmentation.id)
                    ) {
                        errors.addError(
                            "expandedFilter.segments.segmentId", TypedError.invalid(
                                messages.getMessage("errors.resource.segment.not.found", null, locale)
                            )
                        )
                    }
                }
            }

            f.resourceTypeIds.forEach { resourceTypeId ->
                val resourceType = resourceTypesByIds[resourceTypeId]
                if (resourceType == null || resourceType.isDeleted ||
                    (provider != null && resourceType.providerId != provider.id)
                ) {
                    errors.addError(
                        "expandedFilter.resourceTypeIds", TypedError.invalid(
                            messages.getMessage("errors.resource.type.not.found", null, locale)
                        )
                    )
                }
            }
        }

        if (errors.hasAnyErrors()) {
            null
        } else {
            request.expandedFilter
        }
    } else {
        null
    }

    private suspend fun validate(request: GetProvidersByServiceRequest, locale: Locale): Result<GetProvidersByService> {
        val errors = ErrorCollection.builder()
        val service = getService(request.rootServiceId)
        if (service == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.service.not.found", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(GetProvidersByService(service!!))
    }

    private suspend fun validate(request: GetResourceSelectionTreeRequest, locale: Locale): Result<GetResourceSelectionTree> {
        val errors = ErrorCollection.builder()
        val service = getService(request.rootServiceId)
        if (service == null) {
            errors.addError("rootServiceId", TypedError.invalid(messages
                .getMessage("errors.service.not.found", null, locale)))
        }
        val provider = getProvider(request.providerId)
        if (provider == null || provider.isDeleted) {
            errors.addError("filter.providerId", TypedError.invalid(messages
                .getMessage("errors.provider.not.found", null, locale)))
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(GetResourceSelectionTree(service!!, provider!!))
    }

    private suspend fun getService(serviceId: Long): ServiceMinimalModel? {
        return dbSessionRetryable(tableClient) {
            servicesDao.getByIdMinimal(roStaleSingleRetryableCommit(), serviceId)
                .awaitSingle().get().orElse(null)
        }
    }

    private suspend fun getServices(serviceIds: Collection<Long>): Map<Long, ServiceMinimalModel> {
        if (serviceIds.isEmpty()) {
            return emptyMap()
        }
        return dbSessionRetryable(tableClient) {
            serviceIds.chunked(1000).map { ids -> servicesDao.getByIdsMinimal(roStaleSingleRetryableCommit(),
                ids.toList()).awaitSingle() }.flatten().associateBy { it.id }
        }!!
    }

    private suspend fun getFoldersByServices(serviceIds: Collection<Long>): Map<Long, List<FolderModel>> {
        if (serviceIds.isEmpty()) {
            return emptyMap()
        }
        return dbSessionRetryable(tableClient) {
            folderDao.getAllFoldersByServiceIds(roStaleSingleRetryableCommit(),
                serviceIds.map { WithTenant(Tenants.DEFAULT_TENANT_ID, it) }.toSet()).awaitSingle().get()
                .groupBy { it.serviceId }
        }!!
    }

    private suspend fun getResource(resourceId: String): ResourceModel? {
        return resourcesLoader.getResourceByIdImmediate(resourceId, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElse(null)
    }

    private suspend fun getResources(resourceIds: Collection<String>): List<ResourceModel> {
        return resourceIds.chunked(1000).map { ids -> resourcesLoader
            .getResourcesByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }.flatten()
    }

    private suspend fun getProvidersResources(providerIds: Collection<String>): List<ResourceModel> {
        return dbSessionRetryable(tableClient) {
            resourcesDao.getAllByProviders(roStaleSingleRetryableCommit(), providerIds,
                Tenants.DEFAULT_TENANT_ID, false).awaitSingle()
        }!!
    }

    private suspend fun getProvider(providerId: String): ProviderModel? {
        return providersLoader.getProviderByIdImmediate(providerId, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElse(null)
    }

    private suspend fun getProviders(providerIds: Collection<String>): List<ProviderModel> {
        return providerIds.chunked(1000).map { ids -> providersLoader
            .getProvidersByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.flatten()
    }

    private suspend fun getResourceType(resourceTypeId: String): ResourceTypeModel? {
        return resourceTypesLoader.getResourceTypeByIdImmediate(resourceTypeId, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElse(null)
    }

    private suspend fun getResourceTypes(resourceTypeIds: Collection<String>): List<ResourceTypeModel> {
        return resourceTypeIds.chunked(1000).map { ids -> resourceTypesLoader
            .getResourceTypesByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.flatten()
    }

    private suspend fun getSegmentations(segmentationIds: Collection<String>): Map<SegmentationId, ResourceSegmentationModel> {
        return segmentationIds.chunked(1000).map { ids -> segmentationsLoader
            .getResourceSegmentationsByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }.toList())
            .awaitSingle() }.flatten().associateBy { it.id }
    }

    private suspend fun getSegments(segmentIds: Collection<String>): Map<SegmentId, ResourceSegmentModel> {
        return segmentIds.chunked(1000).map { ids -> segmentsLoader.getResourceSegmentsByIdsImmediate(ids
            .map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) }.toList()).awaitSingle() }.flatten().associateBy { it.id }
    }

    private suspend fun getUnitsEnsemble(unitsEnsembleId: String): UnitsEnsembleModel? {
        return unitsEnsemblesLoader.getUnitsEnsembleByIdImmediate(unitsEnsembleId, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle().orElse(null)
    }

    private suspend fun getUnitsEnsembles(unitsEnsembleIds: Collection<String>): List<UnitsEnsembleModel> {
        return unitsEnsembleIds.chunked(1000).map { ids -> unitsEnsemblesLoader
            .getUnitsEnsemblesByIdsImmediate(ids.map { Tuples.of(it, Tenants.DEFAULT_TENANT_ID) })
            .awaitSingle() }.flatten()
    }

    private suspend fun getServiceAggregate(service: ServiceMinimalModel,
                                            resource: ResourceModel): ServiceAggregateModel? {
        return dbSessionRetryable(tableClient) {
            serviceAggregatesDao.getById(roStaleSingleRetryableCommit(), ServiceAggregateKey(resource.tenantId,
                service.id, resource.providerId, resource.id))
        }
    }

    private suspend fun getProviderIdsByService(serviceId: Long): List<String> {
        return dbSessionRetryable(tableClient) {
            serviceAggregatesDao.getProviderIdsByService(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId)
        }!!
    }

    private suspend fun getResourceIdsByServiceAndProvider(serviceId: Long, providerId: String): List<String> {
        return dbSessionRetryable(tableClient) {
            serviceAggregatesDao.getResourceIdsByServiceAndProvider(
                roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, serviceId, providerId)
        }!!
    }

    private suspend fun getDenormalizedAggregates(
        service: ServiceMinimalModel,
        resource: ResourceModel,
        pageToken: PageToken?,
        limit: Long,
        sortingParams: RankSubtreeSortingParamsDto
    ) = when (sortingParams.field) {
        TRANSFERABLE -> getSubtreeTransferable(service, resource, pageToken, limit, sortingParams)

        ALLOCATED, BALANCE, QUOTA, UNALLOCATED, PROVIDED, UNUSED_ESTIMATION, UNDERUTILIZED_ESTIMATION -> getSubtreeCustomSort(
            service, resource, pageToken, limit, sortingParams
        )

        else -> throw IllegalArgumentException("Unsupported sorting field")
    }

    private suspend fun getSubtreeTransferable(
        service: ServiceMinimalModel,
        resource: ResourceModel,
        pageToken: PageToken?,
        limit: Long,
        sortingParams: RankSubtreeSortingParamsDto
    ): List<ServiceDenormalizedAggregateModel> {
        return dbSessionRetryable(tableClient) {
            if (pageToken != null) {
                if (sortingParams.order == RankSubtreeSortingOrder.DESC) {
                    serviceDenormalizedAggregatesDao
                        .getSubtreeDescendingTransferableNextPage(
                            roStaleSingleRetryableCommit(), resource.tenantId,
                            service.id, resource.id, pageToken.fromSortingField.toLong(), pageToken.fromServiceId, limit
                        )
                } else {
                    serviceDenormalizedAggregatesDao
                        .getSubtreeAscendingTransferableNextPage(
                            roStaleSingleRetryableCommit(), resource.tenantId,
                            service.id, resource.id, pageToken.fromSortingField.toLong(), pageToken.fromServiceId, limit
                        )
                }
            } else {
                if (sortingParams.order == RankSubtreeSortingOrder.DESC) {
                    serviceDenormalizedAggregatesDao
                        .getSubtreeDescendingTransferableFirstPage(
                            roStaleSingleRetryableCommit(), resource.tenantId,
                            service.id, resource.id, limit
                        )
                } else {
                    serviceDenormalizedAggregatesDao
                        .getSubtreeAscendingTransferableFirstPage(
                            roStaleSingleRetryableCommit(), resource.tenantId,
                            service.id, resource.id, limit
                        )
                }
            }
        }!!
    }

    private suspend fun getSubtreeCustomSort(
        service: ServiceMinimalModel,
        resource: ResourceModel,
        pageToken: PageToken?,
        limit: Long,
        sortingParams: RankSubtreeSortingParamsDto
    ): List<ServiceDenormalizedAggregateModel> {
        return dbSessionRetryable(tableClient) {
            if (pageToken != null) {
                serviceDenormalizedAggregatesDao
                    .getSubtreeWithCustomSortNextPage(
                        roStaleSingleRetryableCommit(), resource.tenantId,
                        service.id, resource.id, pageToken.fromSortingField,
                        pageToken.fromServiceId, limit, sortingParams
                    )
            } else {
                serviceDenormalizedAggregatesDao
                    .getSubtreeWithCustomSortFirstPage(
                        roStaleSingleRetryableCommit(), resource.tenantId,
                        service.id, resource.id, limit, sortingParams
                    )
            }
        }!!
    }

    private suspend fun getServiceAggregates(request: FindServiceTotals): List<ServiceAggregateModel> {
        if (request.filter == null && request.expandedFilter == null) {
            return dbSessionRetryable(tableClient) {
                serviceAggregatesDao.getByService(roStaleSingleRetryableCommit(),
                    Tenants.DEFAULT_TENANT_ID, request.rootService.id)
            }!!
        } else if (request.filter != null) {
            if (request.filter.segments == null || request.filter.segments.isEmpty()) {
                return dbSessionRetryable(tableClient) {
                    serviceAggregatesDao.getByServiceAndProvider(roStaleSingleRetryableCommit(),
                        Tenants.DEFAULT_TENANT_ID, request.rootService.id, request.filter.provider.id)
                }!!
            } else {
                val providerResources = getProviderResources(request.filter.provider.id)
                val eligibleResources = filterResources(providerResources, request.filter.segments)
                if (eligibleResources.isEmpty()) {
                    return emptyList()
                }
                val allProviderAggregates = dbSessionRetryable(tableClient) {
                    serviceAggregatesDao.getByServiceAndProvider(roStaleSingleRetryableCommit(),
                        Tenants.DEFAULT_TENANT_ID, request.rootService.id, request.filter.provider.id)
                }!!
                val eligibleResourceIds = eligibleResources.map { it.id }.toSet()
                return allProviderAggregates.filter { eligibleResourceIds.contains(it.key.resourceId) }
            }
        } else {
            return getAggregatesByExpandedFilter(request)
        }
    }

    private suspend fun getAggregatesByExpandedFilter(
        request: FindServiceTotals
    ) : List<ServiceAggregateModel> {
        val resourcesByProviderId = mutableMapOf<ProviderId, List<ResourceModel>>()
        val eligibleResourcesIds = mutableSetOf<ResourceId>()
        val aggregates = mutableMapOf<ProviderId, List<ServiceAggregateModel>>()
        request.expandedFilter!!.filters.forEach { singleFilter ->
            if (!resourcesByProviderId.containsKey(singleFilter.providerId)) {
                resourcesByProviderId[singleFilter.providerId] = getProviderResources(singleFilter.providerId)
            }
            val providerResources = resourcesByProviderId[singleFilter.providerId]!!
            val resourceTypeIds = singleFilter.resourceTypeIds
            eligibleResourcesIds.addAll(
                providerResources
                    .filter { resource ->
                        resourceTypeIds.isEmpty() ||
                            resourceTypeIds.contains(resource.resourceTypeId)
                    }
                    .filter { resource ->
                        singleFilter.segments.isEmpty() ||
                            singleFilter.segments.all { s ->
                                s.segmentIds
                                    .contains(resource.segments
                                        .firstOrNull { ss -> ss.segmentationId == s.segmentationId }
                                        ?.segmentId
                                    )
                            }
                    }
                    .map { it.id }
            )
            if (!aggregates.containsKey(singleFilter.providerId)) {
                val allProviderAggregates = dbSessionRetryable(tableClient) {
                    serviceAggregatesDao.getByServiceAndProvider(roStaleSingleRetryableCommit(),
                        Tenants.DEFAULT_TENANT_ID, request.rootService.id, singleFilter.providerId)
                }!!
                aggregates[singleFilter.providerId] = allProviderAggregates
            }
        }
        return aggregates.values.flatten().filter { eligibleResourcesIds.contains(it.key.resourceId) }
    }

    private suspend fun getProviderResources(providerId: String): List<ResourceModel> {
        return dbSessionRetryable(tableClient) {
            resourcesDao.getAllByProvider(roStaleSingleRetryableCommit(), providerId, Tenants.DEFAULT_TENANT_ID,
                false).awaitSingle()
        }!!
    }

    private fun filterResources(resources: List<ResourceModel>,
                                segments: List<FindServiceTotalsFilterSegment>): List<ResourceModel> {
        val segmentsSet = segments
            .map { ResourceSegmentSettingsModel(it.segmentation.id, it.segment.id) }.toSet()
        return resources.filter { r -> segmentsSet.all { s -> r.segments.contains(s) } }
    }

    private data class FindSubtreeTotalRequest(
        val rootServiceId: Long,
        val resourceId: String,
        val providerId: String?,
        val unitId: String?
    )

    private data class RankSubtreeAmountsRequest(
        val rootServiceId: Long,
        val resourceId: String,
        val pageToken: PageToken?,
        val limit: Long,
        val providerId: String?,
        val sortingParams: RankSubtreeSortingParamsDto,
        val unitId: String?
    )

    private data class PageToken(
        val fromSortingField: String,
        val fromServiceId: Long
    )

    private data class FindSubtreeTotal(
        val rootService: ServiceMinimalModel,
        val resource: ResourceModel,
        val provider: ProviderModel,
        val resourceType: ResourceTypeModel,
        val unitsEnsemble: UnitsEnsembleModel,
        val unit: UnitModel?
    )

    private data class RankSubtreeAmounts(
        val rootService: ServiceMinimalModel,
        val resource: ResourceModel,
        val provider: ProviderModel,
        val resourceType: ResourceTypeModel,
        val unitsEnsemble: UnitsEnsembleModel,
        val unit: UnitModel?
    )

    private data class FindServiceTotalsRequest(
        val rootServiceId: Long,
        val filter: FindServiceTotalsRequestFilter?,
        val expandedFilter: FindServiceTotalsExpandedFilter?
    )

    private data class FindServiceTotalsRequestFilter(
        val providerId: String,
        val segments: List<FindServiceTotalsRequestFilterSegment>?
    )

    private data class FindServiceTotalsExpandedFilter(
        val filters: Set<FindServiceTotalsSingleExpandedFilter>
    )

    private data class FindServiceTotalsSingleExpandedFilter(
        val providerId: ProviderId,
        val segments: Set<SegmentationAndSegmentsIds>,
        val resourceTypeIds: Set<ResourceTypeId>
    )

    private data class SegmentationAndSegmentsIds(
        val segmentationId: SegmentationId,
        val segmentIds: Set<SegmentId>
    )

    private data class FindServiceTotalsRequestFilterSegment(
        val segmentationId: String,
        val segmentId: String
    )

    private data class FindServiceTotals(
        val rootService: ServiceMinimalModel,
        val filter: FindServiceTotalsFilter?,
        val expandedFilter: FindServiceTotalsExpandedFilter?
    )

    private data class FindServiceTotalsFilter(
        val provider: ProviderModel,
        val segments: List<FindServiceTotalsFilterSegment>?
    )

    private data class FindServiceTotalsFilterSegment(
        val segmentation: ResourceSegmentationModel,
        val segment: ResourceSegmentModel
    )

    private data class GetProvidersByServiceRequest(
        val rootServiceId: Long
    )

    private data class  GetProvidersByService(
        val rootService: ServiceMinimalModel
    )

    private data class GetResourceSelectionTreeRequest(
        val rootServiceId: Long,
        val providerId: String
    )

    private data class  GetResourceSelectionTree(
        val rootService: ServiceMinimalModel,
        val provider: ProviderModel
    )

}
