package ru.yandex.intranet.d.services.aggregates

import org.springframework.stereotype.Component
import ru.yandex.intranet.d.dao.aggregates.ServiceAggregateUsageDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.EpochSeconds
import ru.yandex.intranet.d.model.aggregates.AggregateBundle
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateKey
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateModel
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateUsageModel
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateModel
import ru.yandex.intranet.d.model.providers.AggregationSettings
import ru.yandex.intranet.d.model.providers.UsageMode
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.model.units.UnitModel
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel
import ru.yandex.intranet.d.model.usage.HistogramBin
import ru.yandex.intranet.d.model.usage.ServiceUsageAmounts
import ru.yandex.intranet.d.model.usage.UsageAmount
import ru.yandex.intranet.d.model.usage.UsagePoint
import ru.yandex.intranet.d.services.quotas.QuotasHelper
import ru.yandex.intranet.d.services.units.UnitsComparator
import ru.yandex.intranet.d.services.usage.accumulate
import ru.yandex.intranet.d.services.usage.histogram
import ru.yandex.intranet.d.services.usage.mean
import ru.yandex.intranet.d.services.usage.minMedianMax
import ru.yandex.intranet.d.services.usage.relativeUsage
import ru.yandex.intranet.d.services.usage.roundToIntegerHalfUp
import ru.yandex.intranet.d.services.usage.standardDeviation
import ru.yandex.intranet.d.services.usage.sumTimeSeries
import ru.yandex.intranet.d.services.usage.underutilized
import ru.yandex.intranet.d.services.usage.variance
import ru.yandex.intranet.d.services.usage.variationCoefficient
import ru.yandex.intranet.d.util.units.Units
import ru.yandex.intranet.d.web.model.AmountDto
import ru.yandex.intranet.d.web.model.aggregation.AggregatedResourceUsageDto
import ru.yandex.intranet.d.web.model.aggregation.ResourceUsageModeDto
import ru.yandex.intranet.d.web.model.aggregation.api.AggregateAmountApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.AggregateUsageApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.HistogramBinApiDto
import ru.yandex.intranet.d.web.model.aggregation.api.TimeSeriesPointApiDto
import java.math.BigDecimal
import java.math.BigInteger
import java.math.RoundingMode
import java.time.Instant
import java.util.*

/**
 * Query aggregates usage service.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class QueryAggregatesUsageService(
    private val tableClient: YdbTableClient,
    private val serviceAggregateUsageDao: ServiceAggregateUsageDao
) {

    suspend fun getServiceUsage(service: ServiceMinimalModel,
                                resource: ResourceModel,
                                aggregate: ServiceAggregateModel?,
                                bundleProducer: (aggregate: ServiceAggregateModel?) -> AggregateBundle?
    ): ServiceAggregateUsageModel? {
        if (aggregate == null || bundleProducer(aggregate)?.extUsage != true) {
            return null
        }
        return dbSessionRetryable(tableClient) {
            serviceAggregateUsageDao.getById(roStaleSingleRetryableCommit(), ServiceAggregateKey(resource.tenantId,
                service.id, resource.providerId, resource.id))
        }
    }

    suspend fun getServiceUsagesFromDenormalizedAggregates(
        page: List<ServiceDenormalizedAggregateModel>): Map<ServiceAggregateKey, ServiceAggregateUsageModel> {
        val usageIds = page.flatMap {
            if (it.exactAmounts.own?.extUsage == true) {
                listOf(ServiceAggregateKey(it.key.tenantId, it.key.serviceId, it.providerId, it.key.resourceId))
            } else {
                emptyList()
            }
        }
        if (usageIds.isEmpty()) {
            return emptyMap()
        }
        return dbSessionRetryable(tableClient) {
            usageIds.chunked(250).flatMap { chunk -> serviceAggregateUsageDao
                .getByIds(roStaleSingleRetryableCommit(), chunk) }.associateBy { it.key }
        }!!
    }

    suspend fun getServiceUsagesFromAggregates(
        aggregates: List<ServiceAggregateModel>,
        bundleProducer: (aggregate: ServiceAggregateModel?) -> AggregateBundle?
    ): Map<ServiceAggregateKey, ServiceAggregateUsageModel> {
        val usageIds = aggregates.flatMap {
            if (bundleProducer(it)?.extUsage == true) {
                listOf(ServiceAggregateKey(it.key.tenantId, it.key.serviceId, it.key.providerId, it.key.resourceId))
            } else {
                emptyList()
            }
        }
        if (usageIds.isEmpty()) {
            return emptyMap()
        }
        return dbSessionRetryable(tableClient) {
            usageIds.chunked(250).flatMap { chunk -> serviceAggregateUsageDao
                .getByIds(roStaleSingleRetryableCommit(), chunk) }.associateBy { it.key }
        }!!
    }

    fun getTotalUsageAmount(amounts: ServiceUsageAmounts?): UsageAmount? {
        if (amounts?.total == null) {
            return amounts?.own
        }
        return amounts.total
    }

    fun aggregateResourceTypeQuota(
        aggregates: List<ServiceAggregateModel>,
        resourceById: Map<String, ResourceModel>,
        unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
        resourceType: ResourceTypeModel,
        aggregationSettings: AggregationSettings,
        bundleProducer: (aggregate: ServiceAggregateModel?) -> AggregateBundle?
    ): ResourceTypeAggregate? {
        val resourceTypeUnitsEnsemble = unitsEnsemblesById[resourceType.unitsEnsembleId]!!
        val resourceTypeBaseUnit = if (resourceType.baseUnitId != null) {
            resourceTypeUnitsEnsemble.unitById(resourceType.baseUnitId).orElseThrow()
        } else {
            UnitsComparator.getBaseUnit(resourceTypeUnitsEnsemble)
        }
        val matchingBaseUnits = aggregates.all {
            val resource = resourceById[it.key.resourceId]!!
            val resourceUnitsEnsemble = unitsEnsemblesById[resource.unitsEnsembleId]!!
            val resourceBaseUnit = resourceUnitsEnsemble.unitById(resource.baseUnitId).orElseThrow()
            resourceTypeBaseUnit == resourceBaseUnit
        }
        if (!matchingBaseUnits) {
            return null
        }
        var provision = BigInteger.ZERO
        aggregates.forEach {
            provision = provision.add(bundleProducer(it)?.provided ?: BigInteger.ZERO)
        }
        val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
        val unusedEstimation = if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
            var sum = BigInteger.ZERO
            aggregates.forEach {
                sum = sum.add(bundleProducer(it)?.unusedEst ?: BigInteger.ZERO)
            }
            sum
        } else {
            null
        }
        return ResourceTypeAggregate(provision, unusedEstimation)
    }

    fun aggregateResourceTypeResources(aggregates: List<ServiceAggregateModel>,
                                       resourceById: Map<String, ResourceModel>): List<ResourceModel> {
        return aggregates.map { it.key.resourceId }.distinct().map { resourceById[it]!! }.toList()
    }

    fun aggregateResourceTypeUsage(
        resourceType: ResourceTypeModel,
        usages: Map<ServiceAggregateKey, ServiceAggregateUsageModel>,
        resourceById: Map<String, ResourceModel>,
        resourceTypeAggregationSettings: AggregationSettings,
        unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
    ): Pair<UsageAmount, Instant>? {
        val resourceIds = resourceById.filterValues { it.resourceTypeId == resourceType.id }.values.map { it.id }.toSet()
        val resourceTypeUnitsEnsemble = unitsEnsemblesById[resourceType.unitsEnsembleId]!!
        val resourceTypeBaseUnit = if (resourceType.baseUnitId != null) {
            resourceTypeUnitsEnsemble.unitById(resourceType.baseUnitId).orElseThrow()
        } else {
            UnitsComparator.getBaseUnit(resourceTypeUnitsEnsemble)
        }
        val matchingBaseUnits = resourceIds.all {
            val resource = resourceById[it]!!
            val resourceUnitsEnsemble = unitsEnsemblesById[resource.unitsEnsembleId]!!
            val resourceBaseUnit = resourceUnitsEnsemble.unitById(resource.baseUnitId).orElseThrow()
            resourceTypeBaseUnit == resourceBaseUnit
        }
        if (!matchingBaseUnits) {
            return null
        }
        val resourceTypeUsages = usages.filterKeys { resourceIds.contains(it.resourceId) }
        val usageMode = resourceTypeAggregationSettings.usageMode ?: UsageMode.UNDEFINED
        if (resourceTypeUsages.isEmpty() || usageMode == UsageMode.UNDEFINED) {
            return null
        }
        val lastUpdate = resourceTypeUsages.values.minOf { it.lastUpdate }
        if (usageMode == UsageMode.TIME_SERIES) {
            val timeSeriesCollection = resourceTypeUsages.values
                .mapNotNull { v ->
                    val totalAmount = getTotalUsageAmount(v.exactAmounts)
                    toTimeSeries(totalAmount?.values, totalAmount?.valuesX, totalAmount?.valuesY)
                }
            if (timeSeriesCollection.isEmpty()) {
                return null
            }
            val timeSeries = sumTimeSeries(timeSeriesCollection)
            if (timeSeries.isEmpty() || timeSeries.values.all { it.compareTo(BigInteger.ZERO) == 0 }) {
                return null
            }
            val gridSpacing = resourceTypeAggregationSettings.timeSeriesGridSpacingSeconds
            val average = mean(timeSeries.values)
            val minMedianMax = minMedianMax(timeSeries.values)
            val variance = variance(timeSeries.values, average)
            val accumulated = if (gridSpacing != null) {
                accumulate(timeSeries, gridSpacing)
            } else {
                null
            }
            val histogram = histogram(timeSeries.values, minMedianMax.first, minMedianMax.third)
            val sortedEntries = timeSeries.entries.sortedBy { it.key }
            val valuesX = sortedEntries.map { it.key }
            val valuesY = sortedEntries.map { it.value }
            return Pair(UsageAmount(
                value = null,
                average = roundToIntegerHalfUp(average),
                min = minMedianMax.first,
                max = minMedianMax.third,
                median = roundToIntegerHalfUp(minMedianMax.second),
                variance = roundToIntegerHalfUp(variance),
                accumulated = if (accumulated != null) { roundToIntegerHalfUp(accumulated.first) } else { null },
                accumulatedDuration = accumulated?.second,
                histogram = histogram,
                values = null,
                valuesX = valuesX,
                valuesY = valuesY,
                unused = null
            ), lastUpdate)
        } else if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
            val unused = resourceTypeUsages.values
                .mapNotNull { getTotalUsageAmount(it.exactAmounts)?.unused }.sumOf { it }
            return Pair(UsageAmount(
                value = null,
                average = null,
                min = null,
                max = null,
                median = null,
                variance = null,
                accumulated = null,
                accumulatedDuration = null,
                histogram = null,
                values = null,
                valuesX = null,
                valuesY = null,
                unused = unused
            ), lastUpdate)
        }
        return null
    }

    fun prepareResourceUsageDto(usageAmount: UsageAmount?,
                                usageLastUpdate: Instant?,
                                aggregateBundle: AggregateBundle?,
                                aggregationSettings: AggregationSettings,
                                resource: ResourceModel,
                                unitsEnsemble: UnitsEnsembleModel,
                                unit: UnitModel?,
                                locale: Locale): AggregatedResourceUsageDto? {
        val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
        if (usageMode == UsageMode.UNDEFINED) {
            return null
        }
        if (usageMode == UsageMode.TIME_SERIES) {
            return prepareTimeSeriesResourceUsageDto(usageAmount, usageLastUpdate,
                aggregateBundle, resource, unitsEnsemble, unit, locale)
        }
        if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
            return prepareUnusedEstimationResourceUsageDto(usageLastUpdate,
                aggregateBundle, resource, unitsEnsemble, unit, locale)
        }
        return null
    }

    fun prepareResourceUsageDtoForResourceType(usageAmount: UsageAmount?,
                                               usageLastUpdate: Instant?,
                                               resourceTypeQuota: ResourceTypeAggregate?,
                                               aggregationSettings: AggregationSettings,
                                               resourceType: ResourceTypeModel,
                                               unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
                                               resources: List<ResourceModel>,
                                               locale: Locale): AggregatedResourceUsageDto? {
        val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
        if (usageMode == UsageMode.UNDEFINED) {
            return null
        }
        if (usageMode == UsageMode.TIME_SERIES) {
            return prepareTimeSeriesResourceUsageDtoForResourceType(usageAmount, usageLastUpdate,
                resourceTypeQuota?.provision, resourceType, unitsEnsemblesById, resources, locale)
        }
        if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
            return prepareUnusedEstimationResourceUsageDtoForResourceType(resourceTypeQuota?.unusedEst,
                usageLastUpdate, resourceType, unitsEnsemblesById, resources, locale)
        }
        return null
    }

    fun prepareServiceAggregateUsageApiDto(usage: UsageAmount?,
                                           lastUpdate: Instant?,
                                           aggregate: AggregateBundle?,
                                           aggregationSettings: AggregationSettings,
                                           resource: ResourceModel,
                                           unitsEnsemble: UnitsEnsembleModel,
                                           includeUsageRaw: Boolean): AggregateUsageApiDto? {
        val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
        if (usageMode == UsageMode.UNDEFINED) {
            return null
        }
        if (usageMode == UsageMode.TIME_SERIES) {
            return prepareTimeSeriesAggregateUsageApiDto(usage, lastUpdate, aggregate,
                resource, unitsEnsemble, includeUsageRaw)
        }
        if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
            return prepareUnusedEstimationAggregateUsageApiDto(lastUpdate, aggregate, resource, unitsEnsemble)
        }
        return null
    }

    private fun prepareTimeSeriesAggregateUsageApiDto(usage: UsageAmount?,
                                                      lastUpdated: Instant?,
                                                      aggregate: AggregateBundle?,
                                                      resource: ResourceModel,
                                                      unitsEnsemble: UnitsEnsembleModel,
                                                      includeUsageRaw: Boolean): AggregateUsageApiDto {
        val mean = usage?.average ?: BigInteger.ZERO
        val standardDeviation = standardDeviation(usage?.variance ?: BigInteger.ZERO)
        val provided = aggregate?.provided ?: BigInteger.ZERO
        val accumulated = usage?.accumulated ?: BigInteger.ZERO
        val accumulatedDuration = usage?.accumulatedDuration ?: 0L
        val underutilizedEstimation = aggregate?.underutilizedEst ?: BigInteger.ZERO
        val convertedAverage = Units.convertToApi(mean, resource, unitsEnsemble)
        val convertedMin = Units.convertToApi(usage?.min ?: BigInteger.ZERO, resource, unitsEnsemble)
        val convertedMax = Units.convertToApi(usage?.max ?: BigInteger.ZERO, resource, unitsEnsemble)
        val convertedMedian = Units.convertToApi(usage?.median ?: BigInteger.ZERO, resource, unitsEnsemble)
        val convertedStdev = Units.convertToApi(standardDeviation, resource, unitsEnsemble)
        val convertedUnderutilizedEstimation = Units.convertToApi(underutilizedEstimation, resource, unitsEnsemble)
        return AggregateUsageApiDto(
            value = null,
            average = AggregateAmountApiDto(convertedAverage.t1.toPlainString(), convertedAverage.t2.key),
            min = AggregateAmountApiDto(convertedMin.t1.toPlainString(), convertedMin.t2.key),
            max = AggregateAmountApiDto(convertedMax.t1.toPlainString(), convertedMax.t2.key),
            median = AggregateAmountApiDto(convertedMedian.t1.toPlainString(), convertedMedian.t2.key),
            stdev = AggregateAmountApiDto(convertedStdev.t1.toPlainString(), convertedStdev.t2.key),
            relativeUsage = relativeUsage(accumulated, provided, accumulatedDuration),
            kv = variationCoefficient(standardDeviation, mean),
            histogram = prepareHistogramApi(usage?.histogram, resource, unitsEnsemble),
            timeSeries = prepareTimeSeriesApi(usage?.values, usage?.valuesX, usage?.valuesY, resource, unitsEnsemble, includeUsageRaw),
            lastUpdate = lastUpdated,
            unusedEstimation = null,
            underutilizedEstimation = AggregateAmountApiDto(convertedUnderutilizedEstimation.t1.toPlainString(),
                convertedUnderutilizedEstimation.t2.key)
        )
    }

    private fun prepareUnusedEstimationAggregateUsageApiDto(lastUpdated: Instant?,
                                                            aggregate: AggregateBundle?,
                                                            resource: ResourceModel,
                                                            unitsEnsemble: UnitsEnsembleModel): AggregateUsageApiDto {
        val convertedToApi = Units.convertToApi(aggregate?.unusedEst ?: BigInteger.ZERO, resource, unitsEnsemble)
        val convertedUnusedEstimation = AggregateAmountApiDto(convertedToApi.t1.toPlainString(), convertedToApi.t2.key)
        return AggregateUsageApiDto(
            value = null,
            average = null,
            min = null,
            max = null,
            median = null,
            stdev = null,
            relativeUsage = null,
            kv = null,
            histogram = null,
            timeSeries = null,
            lastUpdate = lastUpdated,
            unusedEstimation = convertedUnusedEstimation,
            underutilizedEstimation = null
        )
    }

    private fun prepareHistogramApi(histogram: List<HistogramBin>?,
                                    resource: ResourceModel,
                                    unitsEnsemble: UnitsEnsembleModel
    ): List<HistogramBinApiDto>? {
        if (histogram == null) {
            return null
        }
        val total = histogram.sumOf { it.amount }.toBigDecimal()
        return histogram.map { bin ->
            val convertedFrom = Units.convertToApi(bin.from, resource, unitsEnsemble)
            val convertedTo = Units.convertToApi(bin.to, resource, unitsEnsemble)
            HistogramBinApiDto(
                from = AggregateAmountApiDto(convertedFrom.t1.toPlainString(), convertedFrom.t2.key),
                to = AggregateAmountApiDto(convertedTo.t1.toPlainString(), convertedTo.t2.key),
                value = if (total.compareTo(BigDecimal.ZERO) == 0) {
                    0.0
                } else {
                    bin.amount.toBigDecimal().divide(total, 2, RoundingMode.HALF_UP).toDouble()
                }
            )
        }
    }

    private fun prepareTimeSeriesApi(timeSeries: List<UsagePoint>?,
                                     pointsX: List<Long>?,
                                     pointsY: List<BigInteger>?,
                                     resource: ResourceModel,
                                     unitsEnsemble: UnitsEnsembleModel,
                                     includeUsageRaw: Boolean
    ): List<TimeSeriesPointApiDto>? {
        if (!includeUsageRaw) {
            return null;
        }
        if (timeSeries == null && (pointsX == null || pointsY == null)) {
            return null
        }
        if (pointsX != null && pointsY != null) {
            return pointsX.mapIndexed { index, x ->
                val y = if (index < pointsY.size) { pointsY[index] } else { BigInteger.ZERO }
                val convertedY = Units.convertToApi(y, resource, unitsEnsemble)
                TimeSeriesPointApiDto(
                    x = Instant.ofEpochSecond(x),
                    y = AggregateAmountApiDto(convertedY.t1.toPlainString(), convertedY.t2.key)
                )
            }
        } else if (timeSeries != null) {
            return timeSeries.map { point ->
                val convertedY = Units.convertToApi(point.y, resource, unitsEnsemble)
                TimeSeriesPointApiDto(
                    x = Instant.ofEpochSecond(point.x),
                    y = AggregateAmountApiDto(convertedY.t1.toPlainString(), convertedY.t2.key)
                )
            }
        } else {
            return null
        }
    }

    private fun prepareTimeSeriesResourceUsageDto(usageAmount: UsageAmount?,
                                                  usageLastUpdate: Instant?,
                                                  aggregateBundle: AggregateBundle?,
                                                  resource: ResourceModel,
                                                  unitsEnsemble: UnitsEnsembleModel,
                                                  unit: UnitModel?,
                                                  locale: Locale): AggregatedResourceUsageDto {
        val mean = usageAmount?.average ?: BigInteger.ZERO
        val standardDeviation = standardDeviation(usageAmount?.variance ?: BigInteger.ZERO)
        val provided = aggregateBundle?.provided ?: BigInteger.ZERO
        val accumulated = usageAmount?.accumulated ?: BigInteger.ZERO
        val accumulatedDuration = usageAmount?.accumulatedDuration ?: 0L
        val underutilizedEstimation = aggregateBundle?.underutilizedEst ?: BigInteger.ZERO
        return AggregatedResourceUsageDto(
            mode = ResourceUsageModeDto.TIME_SERIES,
            value = null,
            average = QuotasHelper.getAmountDtoForAggregates(mean.toBigDecimal(), resource, unitsEnsemble, unit, locale),
            min = QuotasHelper.getAmountDtoForAggregates((usageAmount?.min ?: BigInteger.ZERO).toBigDecimal(),
                resource, unitsEnsemble, unit, locale),
            max = QuotasHelper.getAmountDtoForAggregates((usageAmount?.max ?: BigInteger.ZERO).toBigDecimal(),
                resource, unitsEnsemble, unit, locale),
            median = QuotasHelper.getAmountDtoForAggregates((usageAmount?.median ?: BigInteger.ZERO).toBigDecimal(),
                resource, unitsEnsemble, unit, locale),
            stdev = QuotasHelper.getAmountDtoForAggregates(standardDeviation.toBigDecimal(), resource, unitsEnsemble, unit, locale),
            relativeUsage = relativeUsage(accumulated, provided, accumulatedDuration),
            kv = variationCoefficient(standardDeviation, mean),
            lastUpdate = usageLastUpdate,
            unusedEstimation = null,
            underutilizedEstimation = QuotasHelper.getAmountDtoForAggregates(underutilizedEstimation.toBigDecimal(),
                resource, unitsEnsemble, unit, locale)
        )
    }

    private fun prepareUnusedEstimationResourceUsageDto(usageLastUpdate: Instant?,
                                                        aggregateBundle: AggregateBundle?,
                                                        resource: ResourceModel,
                                                        unitsEnsemble: UnitsEnsembleModel,
                                                        unit: UnitModel?,
                                                        locale: Locale): AggregatedResourceUsageDto {
        val convertedUnusedEstimation = QuotasHelper.getAmountDtoForAggregates(aggregateBundle?.unusedEst?.toBigDecimal() ?: BigDecimal.ZERO,
                resource, unitsEnsemble, unit, locale)
        return AggregatedResourceUsageDto(
            mode = ResourceUsageModeDto.UNUSED_ESTIMATION_VALUE,
            value = null,
            average = null,
            min = null,
            max = null,
            median = null,
            stdev = null,
            relativeUsage = null,
            kv = null,
            lastUpdate = usageLastUpdate,
            unusedEstimation = convertedUnusedEstimation,
            underutilizedEstimation = null
        )
    }

    private fun prepareTimeSeriesResourceUsageDtoForResourceType(usageAmount: UsageAmount?,
                                                                 usageLastUpdate: Instant?,
                                                                 resourceTypeProvision: BigInteger?,
                                                                 resourceType: ResourceTypeModel,
                                                                 unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
                                                                 resources: List<ResourceModel>,
                                                                 locale: Locale): AggregatedResourceUsageDto {
        val resourceTypeUnitsEnsemble = unitsEnsemblesById[resourceType.unitsEnsembleId]!!
        val mean = usageAmount?.average ?: BigInteger.ZERO
        val standardDeviation = standardDeviation(usageAmount?.variance ?: BigInteger.ZERO)
        val accumulated = usageAmount?.accumulated ?: BigInteger.ZERO
        val accumulatedDuration = usageAmount?.accumulatedDuration ?: 0L
        val underutilizedEstimation = underutilized(accumulated,
            resourceTypeProvision ?: BigInteger.ZERO, accumulatedDuration)
        return AggregatedResourceUsageDto(
            mode = ResourceUsageModeDto.TIME_SERIES,
            value = null,
            average = toAmountDto(mean.toBigDecimal(), resources, resourceType, resourceTypeUnitsEnsemble, locale),
            min = toAmountDto((usageAmount?.min ?: BigInteger.ZERO).toBigDecimal(), resources, resourceType,
                resourceTypeUnitsEnsemble, locale),
            max = toAmountDto((usageAmount?.max ?: BigInteger.ZERO).toBigDecimal(), resources, resourceType,
                resourceTypeUnitsEnsemble, locale),
            median = toAmountDto((usageAmount?.median ?: BigInteger.ZERO).toBigDecimal(), resources, resourceType,
                resourceTypeUnitsEnsemble, locale),
            stdev = toAmountDto(standardDeviation.toBigDecimal(), resources, resourceType,
                resourceTypeUnitsEnsemble, locale),
            relativeUsage = relativeUsage(accumulated, resourceTypeProvision ?: BigInteger.ZERO, accumulatedDuration),
            kv = variationCoefficient(standardDeviation, mean),
            lastUpdate = usageLastUpdate,
            unusedEstimation = null,
            underutilizedEstimation = toAmountDto(underutilizedEstimation.toBigDecimal(), resources, resourceType,
                resourceTypeUnitsEnsemble, locale)
        )
    }

    private fun prepareUnusedEstimationResourceUsageDtoForResourceType(unusedEstimation: BigInteger?,
                                                                       usageLastUpdate: Instant?,
                                                                       resourceType: ResourceTypeModel,
                                                                       unitsEnsemblesById: Map<String, UnitsEnsembleModel>,
                                                                       resources: List<ResourceModel>,
                                                                       locale: Locale): AggregatedResourceUsageDto {
        val resourceTypeUnitsEnsemble = unitsEnsemblesById[resourceType.unitsEnsembleId]!!
        val convertedUnusedEstimation = toAmountDto(unusedEstimation?.toBigDecimal() ?: BigDecimal.ZERO,
            resources, resourceType, resourceTypeUnitsEnsemble, locale)
        return AggregatedResourceUsageDto(
            mode = ResourceUsageModeDto.UNUSED_ESTIMATION_VALUE,
            value = null,
            average = null,
            min = null,
            max = null,
            median = null,
            stdev = null,
            relativeUsage = null,
            kv = null,
            lastUpdate = usageLastUpdate,
            unusedEstimation = convertedUnusedEstimation,
            underutilizedEstimation = null
        )
    }

    private fun toTimeSeries(points: List<UsagePoint>?, pointsX: List<Long>?, pointsY: List<BigInteger>?): Map<EpochSeconds, BigInteger>? {
        if (points == null && (pointsX == null || pointsY == null)) {
            return null
        }
        val result = mutableMapOf<EpochSeconds, BigInteger>()
        if (pointsX != null && pointsY != null) {
            pointsX.forEachIndexed { index, x ->
                result[x] = if (index < pointsY.size) { pointsY[index] } else { BigInteger.ZERO }
            }
        } else if (points != null) {
            points.forEach {
                result[it.x] = it.y
            }
        }
        return result
    }

    private fun toAmountDto(amount: BigDecimal, resources: List<ResourceModel>, resourceType: ResourceTypeModel,
                            unitsEnsemble: UnitsEnsembleModel, locale: Locale): AmountDto {
        val allowedSortedUnits = if (resources.isNotEmpty()) {
            val allowedUnitIds = resources.flatMap { it.resourceUnits.allowedUnitIds }.toSet()
            unitsEnsemble.units.filter { allowedUnitIds.contains(it.id) }
                .sortedWith(UnitsComparator.INSTANCE).toList()
        } else {
            unitsEnsemble.units.sortedWith(UnitsComparator.INSTANCE).toList()
        }
        if (allowedSortedUnits.isEmpty()) {
            val resourceKeys = resources.map { it.key }
            throw IllegalStateException(
                "No allowed units found for resource type ${resourceType.key} and resources $resourceKeys")
        }
        val baseUnit = if (resources.isNotEmpty()) {
            resources.map { unitsEnsemble.unitById(it.baseUnitId)
                .orElseThrow { IllegalStateException("Base unit not found for resource ${it.key} and resource " +
                    "type ${resourceType.key}") } }.distinct().sortedWith(UnitsComparator.INSTANCE).first()
        } else {
            if (resourceType.baseUnitId != null) {
                unitsEnsemble.unitById(resourceType.baseUnitId).orElseThrow()
            } else {
                UnitsComparator.getBaseUnit(unitsEnsemble)
            }
        }
        val minAllowedUnit = allowedSortedUnits.first()
        return QuotasHelper.getAmountDtoForAggregates(
            amount, allowedSortedUnits, baseUnit, minAllowedUnit, null, locale
        )
    }

}
