package ru.yandex.intranet.d.services.aggregates

import ru.yandex.intranet.d.kotlin.EpochSeconds
import ru.yandex.intranet.d.kotlin.ResourceId
import ru.yandex.intranet.d.kotlin.ResourceTypeId
import ru.yandex.intranet.d.kotlin.ServiceId
import ru.yandex.intranet.d.model.aggregates.AggregateBundle
import ru.yandex.intranet.d.model.aggregates.AggregateEpochModel
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateAmounts
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateKey
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateModel
import ru.yandex.intranet.d.model.aggregates.ServiceAggregateUsageModel
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateAmounts
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateKey
import ru.yandex.intranet.d.model.aggregates.ServiceDenormalizedAggregateModel
import ru.yandex.intranet.d.model.providers.AggregationSettings
import ru.yandex.intranet.d.model.providers.FreeProvisionAggregationMode
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.providers.UsageMode
import ru.yandex.intranet.d.model.quotas.QuotaAggregationModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.usage.ServiceUsageAmounts
import ru.yandex.intranet.d.model.usage.ServiceUsageModel
import ru.yandex.intranet.d.model.usage.UsageAmount
import ru.yandex.intranet.d.model.usage.UsagePoint
import ru.yandex.intranet.d.services.usage.accumulate
import ru.yandex.intranet.d.services.usage.histogram
import ru.yandex.intranet.d.services.usage.mean
import ru.yandex.intranet.d.services.usage.minMedianMax
import ru.yandex.intranet.d.services.usage.roundToIntegerHalfUp
import ru.yandex.intranet.d.services.usage.sumTimeSeries
import ru.yandex.intranet.d.services.usage.underutilized
import ru.yandex.intranet.d.services.usage.variance
import java.math.BigDecimal
import java.math.BigInteger
import java.math.RoundingMode
import java.time.Instant
import java.util.*

fun prepareAggregates(provider: ProviderModel,
                      resources: Map<ResourceId, ResourceModel>,
                      resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
                      epochs: Map<ResourceId, AggregateEpochModel>,
                      quotas: QuotasIndex,
                      hierarchy: ServiceHierarchy,
                      timestamp: Instant,
                      usage: UsageIndex
): AggregatedQuotas {
    // Prepare aggregates using post-order depth first traversal
    val aggregatesList = mutableListOf<ServiceAggregateModel>()
    val usageAggregatesList = mutableListOf<ServiceAggregateUsageModel>()
    val denormalizedAggregatesList = mutableListOf<ServiceDenormalizedAggregateModel>()
    // More than one root may exist
    hierarchy.roots.forEach { root ->
        val aggregates = mutableMapOf<ServiceId, Map<ResourceId, ServiceAggregateModel>>()
        val usageAggregates = mutableMapOf<ServiceId, Map<ResourceId, ServiceAggregateUsageModel>>()
        // First traversal to accumulate aggregates
        traverseForAggregates(provider, resources, resourceTypes, epochs, quotas, hierarchy, timestamp,
            root, aggregates, usageAggregates, usage)
        // Another traversal to add denormalized aggregates
        traverseForDenormalizedAggregates(provider, epochs, hierarchy, timestamp, root, aggregates,
            denormalizedAggregatesList)
        // Flatten prepared aggregates
        aggregates.forEach{ (_, byResource) -> byResource.forEach { (_, agg) -> aggregatesList.add(agg)}}
        usageAggregates.forEach { (_, byResource) -> byResource.forEach { (_, agg) -> usageAggregatesList.add(agg) } }
    }
    return AggregatedQuotas(aggregatesList, usageAggregatesList, denormalizedAggregatesList)
}

fun resourceAggregationSettings(resourceId: String,
                                provider: ProviderModel,
                                resources: Map<ResourceId, ResourceModel>,
                                resourceTypes: Map<ResourceTypeId, ResourceTypeModel>
): AggregationSettings {
    val resource = resources[resourceId]!!
    val resourceType = resourceTypes[resource.resourceTypeId]!!
    // From the most specific to the least specific
    return ((resource.aggregationSettings.orElse(null) ?: resourceType.aggregationSettings.orElse(null))
        ?: provider.aggregationSettings.orElse(null)) ?: defaultAggregationSettings()
}

fun resourceTypeAggregationSettings(resourceType: ResourceTypeModel,
                                    provider: ProviderModel
): AggregationSettings {
    // From the most specific to the least specific
    return (resourceType.aggregationSettings.orElse(null)
        ?: provider.aggregationSettings.orElse(null)) ?: defaultAggregationSettings()
}

// Default aggregation settings
fun defaultAggregationSettings() = DEFAULT_AGGREGATION_SETTINGS

private val DEFAULT_AGGREGATION_SETTINGS = AggregationSettings(
    freeProvisionMode = FreeProvisionAggregationMode.NONE,
    usageMode = UsageMode.UNDEFINED,
    timeSeriesGridSpacingSeconds = null
)

// Convert to [Long.MIN_VALUE; Long.MAX_VALUE] range
fun normalizeFraction(numerator: BigInteger, denominator: BigInteger): Long {
    if (denominator.compareTo(BigInteger.ZERO) == 0 || numerator.compareTo(BigInteger.ZERO) == 0) {
        return 0L
    }
    if (numerator.compareTo(denominator) == 0) {
        return Long.MAX_VALUE
    }
    if (numerator.abs() > denominator.abs()) {
        return if (numerator.signum() == denominator.signum()) { Long.MAX_VALUE } else { Long.MIN_VALUE }
    }
    return if (numerator.signum() == denominator.signum()) {
        numerator.abs().toBigDecimal().divide(denominator.abs().toBigDecimal(), 20, RoundingMode.HALF_UP)
            .multiply(BigDecimal.valueOf(Long.MAX_VALUE)).setScale(0, RoundingMode.HALF_UP).toLong()
    } else {
        numerator.abs().toBigDecimal().divide(denominator.abs().toBigDecimal(), 20, RoundingMode.HALF_UP)
            .multiply(BigDecimal.valueOf(Long.MIN_VALUE)).setScale(0, RoundingMode.HALF_UP).toLong()
    }
}

private fun traverseForAggregates(provider: ProviderModel,
                                  resources: Map<ResourceId, ResourceModel>,
                                  resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
                                  epochs: Map<ResourceId, AggregateEpochModel>,
                                  quotas: QuotasIndex,
                                  hierarchy: ServiceHierarchy,
                                  timestamp: Instant,
                                  root: Long,
                                  aggregates: MutableMap<ServiceId, Map<ResourceId, ServiceAggregateModel>>,
                                  usageAggregates: MutableMap<ServiceId, Map<ResourceId, ServiceAggregateUsageModel>>,
                                  usage: UsageIndex) {
    val stack = ArrayDeque<TraversalStackFrame>()
    stack.push(
        TraversalStackFrame(
            root, (hierarchy.childrenByParent[root] ?: setOf()).toMutableSet(), mutableMapOf(), mutableMapOf(), null
        )
    )
    var counter = 0
    val failsafeLimit = hierarchy.nodesCount.toLong() * 3
    while (!stack.isEmpty()) {
        val currentStackFrame = stack.peek()
        if (currentStackFrame.unvisitedChildren.isEmpty()) {
            // Visit this node
            val nodeQuotas = quotas.quotas[currentStackFrame.id] ?: mapOf()
            val nodeUsage = usage.usage[currentStackFrame.id] ?: mapOf()
            val nodeAggregates = if (currentStackFrame.visitedChildren.isEmpty()) {
                // Leaf node
                aggregateLeafServiceResources(nodeQuotas, provider, resources, resourceTypes, epochs,
                    currentStackFrame.id, timestamp, nodeUsage)
            } else {
                // Non-leaf node
                aggregateServiceResources(nodeQuotas, currentStackFrame.visitedChildren.values,
                    currentStackFrame.visitedChildrenUsage.values, provider, resources,
                    resourceTypes, epochs, currentStackFrame.id, timestamp, nodeUsage)
            }
            if (currentStackFrame.parentFrame != null) {
                // Non-root node
                currentStackFrame.parentFrame.unvisitedChildren.remove(currentStackFrame.id)
                currentStackFrame.parentFrame.visitedChildren[currentStackFrame.id] = nodeAggregates.first
                currentStackFrame.parentFrame.visitedChildrenUsage[currentStackFrame.id] = nodeAggregates.second
            }
            if (nodeAggregates.first.isNotEmpty()) {
                aggregates[currentStackFrame.id] = nodeAggregates.first
            }
            if (nodeAggregates.second.isNotEmpty()) {
                usageAggregates[currentStackFrame.id] = nodeAggregates.second
            }
            stack.pop()
        } else {
            // Go deeper to next child
            val nextChildNode = currentStackFrame.unvisitedChildren.first()
            stack.push(
                TraversalStackFrame(nextChildNode,
                    (hierarchy.childrenByParent[nextChildNode] ?: setOf()).toMutableSet(), mutableMapOf(), mutableMapOf(),
                    currentStackFrame
                )
            )
        }
        counter++
        if (counter > failsafeLimit) {
            throw IllegalStateException("Too many tree traversal iterations")
        }
    }
}

private fun traverseForDenormalizedAggregates(provider: ProviderModel,
                                              epochs: Map<ResourceId, AggregateEpochModel>,
                                              hierarchy: ServiceHierarchy,
                                              timestamp: Instant,
                                              root: Long,
                                              aggregates: Map<ServiceId, Map<ResourceId, ServiceAggregateModel>>,
                                              denormalizedAggregatesList: MutableList<ServiceDenormalizedAggregateModel>) {
    val pathStack = ArrayDeque<PathTraversalStackFrame>()
    pathStack.push(
        PathTraversalStackFrame(
            root, listOf(root),
            (hierarchy.childrenByParent[root] ?: setOf()).toMutableSet(), null
        )
    )
    var counter = 0
    val failsafeLimit = hierarchy.nodesCount.toLong() * 3
    while (!pathStack.isEmpty()) {
        val currentPathStackFrame = pathStack.peek()
        if (currentPathStackFrame.unvisitedChildren.isEmpty()) {
            // Visit this node
            val nodeAggregate = aggregates[currentPathStackFrame.id]
            if (nodeAggregate != null && nodeAggregate.isNotEmpty()) {
                // At least one aggregate is present for this node
                nodeAggregate.forEach { (resourceId, aggregate) ->
                    // Traverse path back to root
                    currentPathStackFrame.pathToRoot.forEach { pathNodeId ->
                        addDenormalizedAggregatesForNode(aggregates, pathNodeId, resourceId, aggregate, provider,
                            currentPathStackFrame.id, timestamp, epochs, denormalizedAggregatesList)
                    }
                }
            }
            if (currentPathStackFrame.parentFrame != null) {
                // Non-root node
                currentPathStackFrame.parentFrame.unvisitedChildren.remove(currentPathStackFrame.id)
            }
            pathStack.pop()
        } else {
            // Go deeper to next child
            val nextChildNode = currentPathStackFrame.unvisitedChildren.first()
            pathStack.push(
                PathTraversalStackFrame(nextChildNode, currentPathStackFrame.pathToRoot + nextChildNode,
                    (hierarchy.childrenByParent[nextChildNode] ?: setOf()).toMutableSet(), currentPathStackFrame
                )
            )
        }
        counter++
        if (counter > failsafeLimit) {
            throw IllegalStateException("Too many tree traversal iterations")
        }
    }
}

private fun aggregateServiceResources(serviceQuotas: Map<ResourceId, Set<QuotaAggregationModel>>,
                                      childAggregates: Collection<Map<ResourceId, ServiceAggregateModel>>,
                                      childUsageAggregates: Collection<Map<ResourceId, ServiceAggregateUsageModel>>,
                                      provider: ProviderModel,
                                      resources: Map<ResourceId, ResourceModel>,
                                      resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
                                      epochs: Map<ResourceId, AggregateEpochModel>,
                                      serviceId: Long,
                                      timestamp: Instant,
                                      serviceUsage: Map<ResourceId, ServiceUsageModel>
): Pair<Map<ResourceId, ServiceAggregateModel>, Map<ResourceId, ServiceAggregateUsageModel>> {
    if (serviceQuotas.isEmpty() && childAggregates.isEmpty() && serviceUsage.isEmpty()) {
        return Pair(mapOf(), mapOf())
    }
    val ownAccumulators = mutableMapOf<ResourceId, AggregateAccumulator>()
    val settingsByResource = mutableMapOf<ResourceId, AggregationSettings>()
    serviceQuotas.forEach { (resourceId, quotas) ->
        val accumulator = ownAccumulators.computeIfAbsent(resourceId) { AggregateAccumulator() }
        quotas.forEach { quota -> addQuotaToAccumulator(quota, accumulator) }
    }
    val subtreeAccumulators = mutableMapOf<ResourceId, AggregateAccumulator>()
    childAggregates.forEach { aggregatesByResource -> aggregatesByResource.forEach { (resourceId, aggregate) ->
        val accumulator = subtreeAccumulators.computeIfAbsent(resourceId) { AggregateAccumulator() }
        addAggregateToAccumulator(aggregate, accumulator)
    }}
    val childUsageAggregatesByResource = mutableMapOf<ResourceId, MutableList<ServiceAggregateUsageModel>>()
    childUsageAggregates.forEach { aggregatesByResource -> aggregatesByResource.forEach { (resourceId, aggregate) ->
        childUsageAggregatesByResource.computeIfAbsent(resourceId) { mutableListOf() }.add(aggregate)
    }}
    childUsageAggregatesByResource.forEach { (resourceId, aggregatesForResource) ->
        val settings = settingsByResource.computeIfAbsent(resourceId) { resourceAggregationSettings(resourceId,
            provider, resources, resourceTypes) }
        val accumulator = subtreeAccumulators.computeIfAbsent(resourceId) { AggregateAccumulator() }
        addAggregatesToAccumulator(aggregatesForResource, accumulator, settings)
    }
    ownAccumulators.forEach { (resourceId, accumulator) ->
        val settings = settingsByResource.computeIfAbsent(resourceId) { resourceAggregationSettings(resourceId,
            provider, resources, resourceTypes) }
        updateDerivedAccumulatorFields(accumulator, settings)
    }
    subtreeAccumulators.forEach { (resourceId, accumulator) ->
        val settings = settingsByResource.computeIfAbsent(resourceId) { resourceAggregationSettings(resourceId,
            provider, resources, resourceTypes) }
        updateDerivedAccumulatorFields(accumulator, settings)
    }
    serviceUsage.forEach { (resourceId, usage) ->
        val settings = settingsByResource.computeIfAbsent(resourceId) { resourceAggregationSettings(resourceId,
            provider, resources, resourceTypes) }
        val ownAccumulator = ownAccumulators.computeIfAbsent(resourceId) { AggregateAccumulator() }
        val subtreeAccumulator = subtreeAccumulators.computeIfAbsent(resourceId) { AggregateAccumulator() }
        addLeafUsageToAccumulator(usage, settings, ownAccumulator)
        addLeafUsageAmountToAccumulator(usage, settings, ownAccumulator)
        addSubtreeUsageToAccumulator(usage, settings, subtreeAccumulator)
    }
    val resourceIds = ownAccumulators.keys + subtreeAccumulators.keys
    val result = mutableMapOf<ResourceId, ServiceAggregateModel>()
    val usageResult = mutableMapOf<ResourceId, ServiceAggregateUsageModel>()
    resourceIds.forEach { resourceId ->
        val settings = settingsByResource.computeIfAbsent(resourceId) { resourceAggregationSettings(resourceId,
            provider, resources, resourceTypes) }
        val ownAccumulator = ownAccumulators[resourceId]
        val subtreeAccumulator = subtreeAccumulators[resourceId]
        val (ownBundle, ownUsageAmount) = if (ownAccumulator != null && !isEmptyAccumulator(ownAccumulator)) {
            Pair(toAggregateBundle(ownAccumulator, settings), ownAccumulator.usageAmount)
        } else {
            Pair(null, null)
        }
        val (subtreeBundle, subtreeUsageAmount) = if (subtreeAccumulator != null && !isEmptyAccumulator(subtreeAccumulator)) {
            Pair(toAggregateBundle(subtreeAccumulator, settings), subtreeAccumulator.usageAmount)
        } else {
            Pair(null, null)
        }
        if (ownBundle != null || subtreeBundle != null) {
            val totalAccumulator = if (subtreeBundle != null) {
                val combinedAccumulator = combineAccumulators(subtreeAccumulator, ownAccumulator, settings)
                addTotalUsageToAccumulator(serviceUsage[resourceId], settings, combinedAccumulator)
            } else {
                null
            }
            val (totalBundle, totalUsageAmount) = if (totalAccumulator != null && !isEmptyAccumulator(totalAccumulator)) {
                Pair(toAggregateBundle(totalAccumulator, settings), totalAccumulator.usageAmount)
            } else {
                Pair(null, null)
            }
            result[resourceId] = ServiceAggregateModel(
                key = ServiceAggregateKey(
                    tenantId = provider.tenantId,
                    serviceId = serviceId,
                    providerId = provider.id,
                    resourceId = resourceId
                ),
                lastUpdate = timestamp,
                epoch = epochs[resourceId]!!.epoch,
                exactAmounts = ServiceAggregateAmounts(
                    own = ownBundle,
                    subtree = subtreeBundle,
                    total = totalBundle
                )
            )
            if (ownUsageAmount != null || subtreeUsageAmount != null || totalUsageAmount != null) {
                usageResult[resourceId] = ServiceAggregateUsageModel(
                    key = ServiceAggregateKey(
                        tenantId = provider.tenantId,
                        serviceId = serviceId,
                        providerId = provider.id,
                        resourceId = resourceId
                    ),
                    lastUpdate = earliestTimestamp(ownAccumulator?.lastUsageUpdate,
                        subtreeAccumulator?.lastUsageUpdate, totalAccumulator?.lastUsageUpdate) ?: timestamp,
                    epoch = epochs[resourceId]!!.epoch,
                    exactAmounts = ServiceUsageAmounts(
                        own = ownUsageAmount,
                        subtree = subtreeUsageAmount,
                        total = totalUsageAmount
                    )
                )
            }
        }
    }
    return Pair(result, usageResult)
}

private fun earliestTimestamp(vararg timestamps: Instant?): Instant? {
    return timestamps.filterNotNull().minOrNull()
}

private fun earliestTimestamp(timestamps: Collection<Instant?>): Instant? {
    return timestamps.filterNotNull().minOrNull()
}

private fun aggregateLeafServiceResources(serviceQuotas: Map<ResourceId, Set<QuotaAggregationModel>>,
                                          provider: ProviderModel,
                                          resources: Map<ResourceId, ResourceModel>,
                                          resourceTypes: Map<ResourceTypeId, ResourceTypeModel>,
                                          epochs: Map<ResourceId, AggregateEpochModel>,
                                          serviceId: Long,
                                          timestamp: Instant,
                                          serviceUsage: Map<ResourceId, ServiceUsageModel>
): Pair<Map<ResourceId, ServiceAggregateModel>, Map<ResourceId, ServiceAggregateUsageModel>> {
    if (serviceQuotas.isEmpty() && serviceUsage.isEmpty()) {
        return Pair(mapOf(), mapOf())
    }
    val accumulators = mutableMapOf<ResourceId, AggregateAccumulator>()
    val settingsByResource = mutableMapOf<ResourceId, AggregationSettings>()
    serviceQuotas.forEach { (resourceId, quotas) ->
        val settings = settingsByResource.computeIfAbsent(resourceId)
            { resourceAggregationSettings(resourceId, provider, resources, resourceTypes) }
        val accumulator = accumulators.computeIfAbsent(resourceId) { AggregateAccumulator() }
        quotas.forEach { quota -> addQuotaToAccumulator(quota, accumulator) }
        updateDerivedAccumulatorFields(accumulator, settings)
    }
    serviceUsage.forEach { (resourceId, usage) ->
        val settings = settingsByResource.computeIfAbsent(resourceId)
            { resourceAggregationSettings(resourceId, provider, resources, resourceTypes) }
        val accumulator = accumulators.computeIfAbsent(resourceId) { AggregateAccumulator() }
        addLeafUsageToAccumulator(usage, settings, accumulator)
        addLeafUsageAmountToAccumulator(usage, settings, accumulator)
    }
    val result = mutableMapOf<ResourceId, ServiceAggregateModel>()
    val usageResult = mutableMapOf<ResourceId, ServiceAggregateUsageModel>()
    accumulators.forEach { (resourceId, accumulator) ->
        if (!isEmptyAccumulator(accumulator)) {
            val settings = settingsByResource.computeIfAbsent(resourceId)
                { resourceAggregationSettings(resourceId, provider, resources, resourceTypes) }
            result[resourceId] = ServiceAggregateModel(
                key = ServiceAggregateKey(
                    tenantId = provider.tenantId,
                    serviceId = serviceId,
                    providerId = provider.id,
                    resourceId = resourceId
                ),
                lastUpdate = timestamp,
                epoch = epochs[resourceId]!!.epoch,
                exactAmounts = ServiceAggregateAmounts(
                    own = toAggregateBundle(accumulator, settings),
                    subtree = null,
                    total = null
                )
            )
            if (accumulator.usageAmount != null) {
                usageResult[resourceId] = ServiceAggregateUsageModel(
                    key = ServiceAggregateKey(
                        tenantId = provider.tenantId,
                        serviceId = serviceId,
                        providerId = provider.id,
                        resourceId = resourceId
                    ),
                    lastUpdate = accumulator.lastUsageUpdate ?: timestamp,
                    epoch = epochs[resourceId]!!.epoch,
                    exactAmounts = ServiceUsageAmounts(
                        own = accumulator.usageAmount,
                        subtree = null,
                        total = null
                    )
                )
            }
        }
    }
    return Pair(result, usageResult)
}

private fun addLeafUsageAmountToAccumulator(usage: ServiceUsageModel,
                                            settings: AggregationSettings,
                                            accumulator: AggregateAccumulator) {
    accumulator.usageAmount = combineUsageAmount(accumulator.usageAmount, usage.usage.own, settings)
    accumulator.lastUsageUpdate = earliestTimestamp(accumulator.lastUsageUpdate, usage.lastUpdate)
}

private fun addLeafUsageToAccumulator(usage: ServiceUsageModel,
                                      settings: AggregationSettings,
                                      accumulator: AggregateAccumulator) {
    val usageMode = settings.usageMode ?: UsageMode.UNDEFINED
    if (usageMode == UsageMode.UNDEFINED) {
        return
    }
    if (usageMode == UsageMode.TIME_SERIES && usage.usage.own != null && usage.usage.own.accumulated != null
        && usage.usage.own.accumulated.compareTo(BigInteger.ZERO) != 0) {
         accumulator.extUsage = true
    }
}

private fun addSubtreeUsageToAccumulator(usage: ServiceUsageModel,
                                         settings: AggregationSettings,
                                         accumulator: AggregateAccumulator) {
    val usageMode = settings.usageMode ?: UsageMode.UNDEFINED
    if (usageMode == UsageMode.UNDEFINED) {
        return
    }
    if (usageMode == UsageMode.TIME_SERIES && usage.usage.subtree != null && usage.usage.subtree.accumulated != null
        && usage.usage.subtree.accumulated.compareTo(BigInteger.ZERO) != 0) {
        accumulator.extUsage = true
    }
}

private fun addTotalUsageToAccumulator(usage: ServiceUsageModel?,
                                       settings: AggregationSettings,
                                       accumulator: AggregateAccumulator?): AggregateAccumulator? {
    if (usage == null) {
        return accumulator
    }
    val usageMode = settings.usageMode ?: UsageMode.UNDEFINED
    if (usageMode == UsageMode.UNDEFINED) {
        return accumulator
    }
    if (usageMode == UsageMode.TIME_SERIES && usage.usage.total != null && usage.usage.total.accumulated != null
        && usage.usage.total.accumulated.compareTo(BigInteger.ZERO) != 0) {
        return if (accumulator != null) {
            accumulator.extUsage = true
            accumulator
        } else {
            val emptyAccumulator = AggregateAccumulator()
            emptyAccumulator.extUsage = true
            emptyAccumulator
        }
    }
    return accumulator
}

private fun addQuotaToAccumulator(
    quota: QuotaAggregationModel,
    accumulator: AggregateAccumulator
) {
    if (quota.quota != null) {
        accumulator.quota = accumulator.quota.add(BigInteger.valueOf(quota.quota))
    }
    if (quota.balance != null) {
        accumulator.balance = accumulator.balance.add(BigInteger.valueOf(quota.balance))
    }
    if (quota.provided != null) {
        accumulator.provided = accumulator.provided.add(BigInteger.valueOf(quota.provided))
    }
    if (quota.allocated != null) {
        accumulator.allocated = accumulator.allocated.add(BigInteger.valueOf(quota.allocated))
    }
    // TODO Usage is not implemented yet
}

private fun addAggregateToAccumulator(
    aggregate: ServiceAggregateModel,
    accumulator: AggregateAccumulator
) {
    if (aggregate.exactAmounts.total != null) {
        // Aggregate is for a non-leaf node
        accumulator.quota = accumulator.quota.add(aggregate.exactAmounts.total.quota)
        accumulator.balance = accumulator.balance.add(aggregate.exactAmounts.total.balance)
        accumulator.provided = accumulator.provided.add(aggregate.exactAmounts.total.provided)
        accumulator.allocated = accumulator.allocated.add(aggregate.exactAmounts.total.allocated)
        // TODO Usage is not implemented yet
    } else if (aggregate.exactAmounts.own != null) {
        // Aggregate is for a leaf node
        accumulator.quota = accumulator.quota.add(aggregate.exactAmounts.own.quota)
        accumulator.balance = accumulator.balance.add(aggregate.exactAmounts.own.balance)
        accumulator.provided = accumulator.provided.add(aggregate.exactAmounts.own.provided)
        accumulator.allocated = accumulator.allocated.add(aggregate.exactAmounts.own.allocated)
        // TODO Usage is not implemented yet
    }
}

private fun addAggregatesToAccumulator(
    aggregates: List<ServiceAggregateUsageModel>,
    accumulator: AggregateAccumulator,
    aggregationSettings: AggregationSettings
) {
    val amounts = aggregates.mapNotNull { aggregate -> aggregate.exactAmounts.total ?: aggregate.exactAmounts.own }
    val amountsToCombine = if (accumulator.usageAmount != null) {
        amounts + accumulator.usageAmount!!
    } else {
        amounts
    }
    accumulator.usageAmount = combineUsageAmounts(amountsToCombine, aggregationSettings)
    accumulator.lastUsageUpdate = earliestTimestamp(aggregates.map { it.lastUpdate } + accumulator.lastUsageUpdate)
}

private fun combineAccumulators(left: AggregateAccumulator?, right: AggregateAccumulator?,
                                aggregationSettings: AggregationSettings): AggregateAccumulator? {
    if (left == null && right == null) {
        return null
    } else if (left == null) {
        return right
    } else if (right == null) {
        return left
    } else {
        return AggregateAccumulator(
            quota = left.quota.add(right.quota),
            balance = left.balance.add(right.balance),
            provided = left.provided.add(right.provided),
            allocated = left.allocated.add(right.allocated),
            usage = left.usage.add(right.usage),
            unallocated = left.unallocated.add(right.unallocated),
            unused = left.unused.add(right.unused),
            underutilized = left.underutilized.add(right.underutilized),
            transferable = left.transferable.add(right.transferable),
            deallocatable = left.deallocatable.add(right.deallocatable),
            usageAmount = combineUsageAmount(left.usageAmount, right.usageAmount, aggregationSettings),
            lastUsageUpdate = earliestTimestamp(left.lastUsageUpdate, right.lastUsageUpdate)
        )
    }
}

private fun combineUsageAmount(left: UsageAmount?, right: UsageAmount?,
                               aggregationSettings: AggregationSettings): UsageAmount? {
    if (left == null && right == null) {
        return null
    }
    if (left == null) {
        return right
    }
    if (right == null) {
        return left
    }
    val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
    if (usageMode == UsageMode.UNDEFINED) {
        return null
    }
    if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
        return UsageAmount(
            value = null,
            average = null,
            min = null,
            max = null,
            median = null,
            variance = null,
            accumulated = null,
            accumulatedDuration = null,
            histogram = null,
            values = null,
            valuesX = null,
            valuesY = null,
            unused = (left.unused ?: BigInteger.ZERO).add(right.unused ?: BigInteger.ZERO)
        )
    }
    if (usageMode == UsageMode.TIME_SERIES) {
        val timeSeries = sumTimeSeries(toTimeSeries(left.values, left.valuesX, left.valuesY),
            toTimeSeries(right.values, right.valuesX, right.valuesY))
        val gridSpacing = aggregationSettings.timeSeriesGridSpacingSeconds
        val average = mean(timeSeries.values)
        val minMedianMax = minMedianMax(timeSeries.values)
        val variance = variance(timeSeries.values, average)
        val accumulated = if (gridSpacing != null) {
            accumulate(timeSeries, gridSpacing)
        } else {
            Pair(BigDecimal.ZERO, 0L)
        }
        val histogram = histogram(timeSeries.values, minMedianMax.first, minMedianMax.third)
        val sortedEntries = timeSeries.entries.sortedBy { it.key }
        val valuesX = sortedEntries.map { it.key }
        val valuesY = sortedEntries.map { it.value }
        return UsageAmount(
            value = null,
            average = roundToIntegerHalfUp(average),
            min = minMedianMax.first,
            max = minMedianMax.third,
            median = roundToIntegerHalfUp(minMedianMax.second),
            variance = roundToIntegerHalfUp(variance),
            accumulated = roundToIntegerHalfUp(accumulated.first),
            accumulatedDuration = accumulated.second,
            histogram = histogram,
            values = null,
            valuesX = valuesX,
            valuesY = valuesY,
            unused = null
        )
    }
    return null
}

private fun combineUsageAmounts(amounts: List<UsageAmount>, aggregationSettings: AggregationSettings): UsageAmount? {
    if (amounts.isEmpty()) {
        return null
    }
    val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
    if (usageMode == UsageMode.UNDEFINED) {
        return null
    }
    if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
        if (amounts.size == 1) {
            return amounts.first()
        }
        return UsageAmount(
            value = null,
            average = null,
            min = null,
            max = null,
            median = null,
            variance = null,
            accumulated = null,
            accumulatedDuration = null,
            histogram = null,
            values = null,
            valuesX = null,
            valuesY = null,
            unused = amounts.sumOf { it.unused ?: BigInteger.ZERO }
        )
    }
    if (usageMode == UsageMode.TIME_SERIES) {
        if (amounts.size == 1) {
            return amounts.first()
        }
        val timeSeries = sumTimeSeries(amounts.map { toTimeSeries(it.values, it.valuesX, it.valuesY) })
        val gridSpacing = aggregationSettings.timeSeriesGridSpacingSeconds
        val average = mean(timeSeries.values)
        val minMedianMax = minMedianMax(timeSeries.values)
        val variance = variance(timeSeries.values, average)
        val accumulated = if (gridSpacing != null) {
            accumulate(timeSeries, gridSpacing)
        } else {
            Pair(BigDecimal.ZERO, 0L)
        }
        val histogram = histogram(timeSeries.values, minMedianMax.first, minMedianMax.third)
        val sortedEntries = timeSeries.entries.sortedBy { it.key }
        val valuesX = sortedEntries.map { it.key }
        val valuesY = sortedEntries.map { it.value }
        return UsageAmount(
            value = null,
            average = roundToIntegerHalfUp(average),
            min = minMedianMax.first,
            max = minMedianMax.third,
            median = roundToIntegerHalfUp(minMedianMax.second),
            variance = roundToIntegerHalfUp(variance),
            accumulated = roundToIntegerHalfUp(accumulated.first),
            accumulatedDuration = accumulated.second,
            histogram = histogram,
            values = null,
            valuesX = valuesX,
            valuesY = valuesY,
            unused = null
        )
    }
    return null
}

private fun toTimeSeries(points: List<UsagePoint>?, pointsX: List<Long>?, pointsY: List<BigInteger>?): Map<EpochSeconds, BigInteger> {
    if (points == null && (pointsX == null || pointsY == null)) {
        return emptyMap()
    }
    val result = mutableMapOf<EpochSeconds, BigInteger>()
    if (pointsX != null && pointsY != null) {
        pointsX.forEachIndexed { index, x ->
            result[x] = if (index < pointsY.size) { pointsY[index] } else { BigInteger.ZERO }
        }
    } else if (points != null) {
        points.forEach {
            result[it.x] = it.y
        }
    }
    return result
}

private fun updateDerivedAccumulatorFields(
    accumulator: AggregateAccumulator,
    settings: AggregationSettings
) {
    accumulator.unallocated = accumulator.provided.subtract(accumulator.allocated)
    accumulator.transferable = accumulator.transferable.add(accumulator.balance)
    if (settings.freeProvisionMode == FreeProvisionAggregationMode.UNALLOCATED_TRANSFERABLE
        || settings.freeProvisionMode == FreeProvisionAggregationMode.UNALLOCATED_TRANSFERABLE_UNUSED_DEALLOCATABLE
    ) {
        accumulator.transferable = accumulator.transferable.add(accumulator.unallocated)
    }
    // TODO Usage is not implemented yet
}

private fun isEmptyAccumulator(accumulator: AggregateAccumulator): Boolean {
    return accumulator.quota.compareTo(BigInteger.ZERO) == 0 && accumulator.balance.compareTo(BigInteger.ZERO) == 0
        && accumulator.provided.compareTo(BigInteger.ZERO) == 0 && accumulator.allocated.compareTo(BigInteger.ZERO) == 0
        && accumulator.usage.compareTo(BigInteger.ZERO) == 0 && accumulator.unallocated.compareTo(BigInteger.ZERO) == 0
        && accumulator.unused.compareTo(BigInteger.ZERO) == 0 && accumulator.underutilized.compareTo(BigInteger.ZERO) == 0
        && accumulator.transferable.compareTo(BigInteger.ZERO) == 0 && accumulator.deallocatable.compareTo(BigInteger.ZERO) == 0
        && accumulator.extUsage == null && accumulator.usageAmount == null
}

private fun toAggregateBundle(accumulator: AggregateAccumulator, aggregationSettings: AggregationSettings) = AggregateBundle(
        quota = accumulator.quota,
        balance = accumulator.balance,
        provided = accumulator.provided,
        allocated = accumulator.allocated,
        usage = accumulator.usage,
        unallocated = accumulator.unallocated,
        unused = accumulator.unused,
        underutilized = accumulator.underutilized,
        transferable = accumulator.transferable,
        deallocatable = accumulator.deallocatable,
        extUsage = accumulator.extUsage,
        unusedEst = toUnused(accumulator.usageAmount, aggregationSettings),
        underutilizedEst = toUnderutilized(accumulator.usageAmount, accumulator.provided, aggregationSettings)
    )

private fun toUnused(usageAmount: UsageAmount?, aggregationSettings: AggregationSettings): BigInteger? {
    val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
    return if (usageMode == UsageMode.UNUSED_ESTIMATION_VALUE) {
        usageAmount?.unused ?: BigInteger.ZERO
    } else {
        null
    }
}

private fun toUnderutilized(usageAmount: UsageAmount?, provided: BigInteger,
                            aggregationSettings: AggregationSettings): BigInteger? {
    val usageMode = aggregationSettings.usageMode ?: UsageMode.UNDEFINED
    if (usageMode == UsageMode.TIME_SERIES) {
        if (usageAmount == null) {
            return provided
        }
        return underutilized(usageAmount.accumulated ?: BigInteger.ZERO, provided, usageAmount.accumulatedDuration ?: 0L)
    } else {
        return null
    }
}

private fun addDenormalizedAggregatesForNode(
    aggregates: Map<ServiceId, Map<ResourceId, ServiceAggregateModel>>,
    superTreeServiceId: ServiceId,
    resourceId: ResourceId,
    aggregate: ServiceAggregateModel,
    provider: ProviderModel,
    serviceId: ServiceId,
    timestamp: Instant,
    epochs: Map<ResourceId, AggregateEpochModel>,
    denormalizedAggregatesList: MutableList<ServiceDenormalizedAggregateModel>
) {
    val pathNodeAggregates = aggregates[superTreeServiceId]
    val pathNodeAggregate = if (pathNodeAggregates != null) {
        pathNodeAggregates[resourceId]
    } else {
        null
    }
    if (superTreeServiceId == serviceId && aggregate.exactAmounts.own != null) {
        // Add denormalized aggregate for node itself when available
        val denormalizedAggregate = toDenormalizedAggregate(aggregate.exactAmounts.own,
            aggregate.exactAmounts.total, provider, superTreeServiceId, resourceId,
            serviceId, timestamp, epochs, true
        )
        denormalizedAggregatesList.add(denormalizedAggregate)
    } else if (pathNodeAggregate != null && aggregate.exactAmounts.own != null
        && pathNodeAggregate.exactAmounts.total != null
    ) {
        // Add denormalized aggregate when available
        val denormalizedAggregate = toDenormalizedAggregate(aggregate.exactAmounts.own,
            pathNodeAggregate.exactAmounts.total, provider, superTreeServiceId, resourceId,
            serviceId, timestamp, epochs, false
        )
        denormalizedAggregatesList.add(denormalizedAggregate)
    }
}

private fun toDenormalizedAggregate(
    own: AggregateBundle,
    total: AggregateBundle?,
    provider: ProviderModel,
    superTreeServiceId: ServiceId,
    resourceId: ResourceId,
    serviceId: ServiceId,
    timestamp: Instant,
    epochs: Map<ResourceId, AggregateEpochModel>,
    selfAggregate: Boolean
): ServiceDenormalizedAggregateModel {
    val transferable = if (total != null) {
        normalizeFraction(own.transferable, total.transferable)
    } else if (selfAggregate) {
        normalizeFraction(own.transferable, own.transferable)
    } else {
        0L
    }
    val deallocatable = if (total != null) {
        normalizeFraction(own.deallocatable, total.deallocatable)
    } else if (selfAggregate) {
        normalizeFraction(own.deallocatable, own.deallocatable)
    } else {
        0L
    }
    return ServiceDenormalizedAggregateModel(
        key = ServiceDenormalizedAggregateKey(
            tenantId = provider.tenantId,
            superTreeServiceId = superTreeServiceId,
            resourceId = resourceId,
            serviceId = serviceId
        ),
        lastUpdate = timestamp,
        epoch = epochs[resourceId]!!.epoch,
        providerId = provider.id,
        transferable = transferable,
        deallocatable = deallocatable,
        exactAmounts = ServiceDenormalizedAggregateAmounts(
            own = own
        )
    )
}

private data class TraversalStackFrame(
    val id: ServiceId,
    val unvisitedChildren: MutableSet<ServiceId>,
    val visitedChildren: MutableMap<ServiceId, Map<ResourceId, ServiceAggregateModel>>,
    val visitedChildrenUsage: MutableMap<ServiceId, Map<ResourceId, ServiceAggregateUsageModel>>,
    val parentFrame: TraversalStackFrame?
)

private data class PathTraversalStackFrame(
    val id: ServiceId,
    val pathToRoot: List<ServiceId>,
    val unvisitedChildren: MutableSet<ServiceId>,
    val parentFrame: PathTraversalStackFrame?
)

private data class AggregateAccumulator(
    var quota: BigInteger = BigInteger.ZERO,
    var balance: BigInteger = BigInteger.ZERO, // quota - provided
    var provided: BigInteger = BigInteger.ZERO,
    var allocated: BigInteger = BigInteger.ZERO,
    var usage: BigInteger = BigInteger.ZERO,
    var unallocated: BigInteger = BigInteger.ZERO, // provided - allocated
    var unused: BigInteger = BigInteger.ZERO, // allocated - usage
    var underutilized: BigInteger = BigInteger.ZERO, // provided - usage
    var transferable: BigInteger = BigInteger.ZERO, // (balance OR 0) + (unallocated OR underutilized OR 0)
    var deallocatable: BigInteger = BigInteger.ZERO, // (unused OR 0)
    var extUsage: Boolean? = null, // Is non-empty usage present in separate table? Null means false.
    var usageAmount: UsageAmount? = null, // Usage amount
    var lastUsageUpdate: Instant? = null // The earliest of the aggregated usage last update timestamps
)
