package ru.yandex.intranet.d.services.aggregates

import com.google.common.util.concurrent.ThreadFactoryBuilder
import kotlinx.coroutines.*
import kotlinx.coroutines.reactor.awaitSingle
import kotlinx.coroutines.reactor.awaitSingleOrNull
import mu.KotlinLogging
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.scheduling.annotation.Scheduled
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsQuotasDao
import ru.yandex.intranet.d.dao.accounts.ProviderReserveAccountsDao
import ru.yandex.intranet.d.dao.aggregates.ServiceAggregateUsageDao
import ru.yandex.intranet.d.dao.aggregates.ServiceAggregatesDao
import ru.yandex.intranet.d.dao.aggregates.ServiceAggregatesEpochsDao
import ru.yandex.intranet.d.dao.aggregates.ServiceDenormalizedAggregatesDao
import ru.yandex.intranet.d.dao.providers.ProvidersDao
import ru.yandex.intranet.d.dao.quotas.QuotasDao
import ru.yandex.intranet.d.dao.resources.ResourcesDao
import ru.yandex.intranet.d.dao.resources.types.ResourceTypesDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.dao.usage.ServiceUsageDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.kotlin.*
import ru.yandex.intranet.d.model.aggregates.AggregateEpochKey
import ru.yandex.intranet.d.model.aggregates.AggregateEpochModel
import ru.yandex.intranet.d.model.folders.FolderType
import ru.yandex.intranet.d.model.providers.AggregationQuotaQueryType
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.providers.UsageMode
import ru.yandex.intranet.d.model.quotas.QuotaAggregationModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.resources.types.ResourceTypeModel
import ru.yandex.intranet.d.model.services.ServiceNode
import ru.yandex.intranet.d.model.usage.ServiceUsageModel
import ru.yandex.intranet.d.util.AggregationAlgorithmHelper
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import ru.yandex.monlib.metrics.labels.Labels
import ru.yandex.monlib.metrics.primitives.GaugeInt64
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64
import ru.yandex.monlib.metrics.registry.MetricRegistry
import java.math.BigInteger
import java.time.Clock
import java.time.Duration
import java.time.Instant
import java.util.*
import java.util.concurrent.ConcurrentHashMap
import java.util.concurrent.ScheduledExecutorService
import java.util.concurrent.ScheduledThreadPoolExecutor
import java.util.concurrent.TimeUnit
import java.util.concurrent.atomic.AtomicLong
import javax.annotation.PreDestroy
import kotlin.math.ceil
import kotlin.math.max

private val logger = KotlinLogging.logger {}

/**
 * Service aggregates refresh implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class RefreshServiceAggregatesService(private val tableClient: YdbTableClient,
                                      private val providersDao: ProvidersDao,
                                      private val resourcesDao: ResourcesDao,
                                      private val resourceTypesDao: ResourceTypesDao,
                                      private val servicesDao: ServicesDao,
                                      private val quotasDao: QuotasDao,
                                      private val accountsQuotasDao: AccountsQuotasDao,
                                      private val serviceAggregatesDao: ServiceAggregatesDao,
                                      private val serviceAggregateUsageDao: ServiceAggregateUsageDao,
                                      private val serviceDenormalizedAggregatesDao: ServiceDenormalizedAggregatesDao,
                                      private val serviceAggregatesEpochsDao: ServiceAggregatesEpochsDao,
                                      private val serviceUsageDao: ServiceUsageDao,
                                      private val providerReserveAccountsDao: ProviderReserveAccountsDao,
                                      @Qualifier("messageSource") private val messages: MessageSource
) {

    private val metrics: ConcurrentHashMap<String, ProviderMetrics> = ConcurrentHashMap()
    private val lastSuccess: ConcurrentHashMap<String, Instant> = ConcurrentHashMap()
    private val scheduler: ScheduledExecutorService
    private val computeDispatcher: ExecutorCoroutineDispatcher

    init {
        val threadFactory = ThreadFactoryBuilder()
            .setDaemon(true)
            .setNameFormat("aggregation-compute-pool-%d")
            .setUncaughtExceptionHandler { t: Thread, e: Throwable? ->
                logger.error(e) { "Uncaught exception in aggregation compute pool thread $t" }
            }
            .build()
        val scheduledThreadPoolExecutor = ScheduledThreadPoolExecutor(
            (Runtime.getRuntime().availableProcessors() / 2).coerceAtLeast(1), threadFactory)
        scheduledThreadPoolExecutor.removeOnCancelPolicy = true
        scheduler = scheduledThreadPoolExecutor
        computeDispatcher = scheduledThreadPoolExecutor.asCoroutineDispatcher()
    }

    fun refreshProviderWithValidationMono(providerId: String,
                                          user: YaUserDetails, locale: Locale, clock: Clock): Mono<Result<Unit>> {
        return mono {
            refreshProviderWithValidation(providerId, user, locale, clock)
        }
    }

    suspend fun refreshProvider(providerId: String, clock: Clock) {
        val provider = meter({ getProvider(providerId) }, "Provider aggregation, load provider")
        refreshProvider(provider, clock)
    }

    @Scheduled(fixedDelayString = "\${metrics.aggregationMetricsRefreshDelayMs}",
        initialDelayString = "\${metrics.aggregationMetricsRefreshInitialDelayMs}")
    fun refreshMetrics() {
        try {
            runBlocking {
                val providers = dbSessionRetryable(tableClient) {
                    providersDao.getAllByTenant(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, false)
                        .awaitSingle().get()
                }!!
                providers.forEach {
                    val providerKey = it.key
                    lastSuccess.computeIfAbsent(providerKey) { Instant.now() }
                    metrics.computeIfAbsent(providerKey) { ProviderMetrics(
                        timeSinceLastSuccess = newTimeSinceLastSuccessMetric(providerKey),
                        aggregatesCount = aggregationsCountMetric(providerKey),
                        maxAggregatesCountByResource = maxAggregationsCountByResourceMetric(providerKey)
                    ) }
                }
            }
        } catch (e: Exception) {
            logger.error(e) { "Failed to refresh aggregation metrics" }
        }
    }

    @PreDestroy
    fun preDestroy() {
        logger.info("Stopping aggregation compute dispatcher...")
        computeDispatcher.close()
        try {
            scheduler.awaitTermination(1, TimeUnit.SECONDS)
        } catch (e: InterruptedException) {
            Thread.currentThread().interrupt()
        }
        scheduler.shutdownNow()
        logger.info("Stopped aggregation compute dispatcher")
    }

    private suspend fun refreshProviderWithValidation(providerId: String, user: YaUserDetails,
                                                      locale: Locale, clock: Clock): Result<Unit> {
        if (user.user.isEmpty || !user.user.get().dAdmin) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                .getMessage("errors.access.denied", null, locale))).build())
        }
        val provider = tryGetProvider(providerId)
            ?: return Result.failure(ErrorCollection.builder().addError(TypedError.forbidden(messages
                .getMessage("errors.provider.not.found", null, locale))).build())
        refreshProvider(provider, clock)
        return Result.success(Unit)
    }

    private suspend fun refreshProvider(provider: ProviderModel, clock: Clock) = coroutineScope {
        meter({
            logger.info { "Aggregating provider ${provider.key}..." }
            val aggregationParams = prepareParams(provider)
            if (!aggregationParams.disabled) {
                val resources = meter({ getResources(provider.id) },
                    "Provider ${provider.key} aggregation, load resources")
                val resourceTypes = meter({ getResourceTypes(resources.map { it.resourceTypeId }.toSet()) },
                    "Provider ${provider.key} aggregation, load resource types (${resources.size} resources)")
                val serviceNodes = meter({ getAllServiceNodes() },
                    "Provider ${provider.key} aggregation, load services")
                val hierarchy = meter({ withContext(computeDispatcher) {
                    prepareHierarchy(serviceNodes)
                } }, "Provider ${provider.key} aggregation, prepare service tree")
                val aggregatedQuotasStats = aggregateProvider(provider, resources, resourceTypes, hierarchy, clock,
                    aggregationParams)
                metrics[provider.key]?.apply {
                    aggregatesCount.set(aggregatedQuotasStats.denormalizedAggregatesCount.toLong())
                    maxAggregatesCountByResource.set(aggregatedQuotasStats.maxPerResourceDenormalizedAggregatesCount
                        .toLong())
                }
            }
        }, "Provider ${provider.key} aggregation")
        lastSuccess[provider.key] = Instant.now()
    }

    private fun newTimeSinceLastSuccessMetric(providerKey: String) = MetricRegistry.root().lazyGaugeInt64(
        "cron.jobs.duration_since_last_success_millis",
        Labels.of("job", "RefreshServiceAggregatesJob", "provider", providerKey))
    { Duration.between(lastSuccess.getOrDefault(providerKey, Instant.now()), Instant.now()).toMillis() }

    private fun aggregationsCountMetric(providerKey: String) = MetricRegistry.root().gaugeInt64(
        "cron.jobs.aggregations_count",
        Labels.of("job", "RefreshServiceAggregatesJob", "provider", providerKey)
    )

    private fun maxAggregationsCountByResourceMetric(providerKey: String) = MetricRegistry.root().gaugeInt64(
        "cron.jobs.aggregations_count.max_by_resource",
        Labels.of("job", "RefreshServiceAggregatesJob", "provider", providerKey)
    )

    private suspend fun getProvider(providerId: String): ProviderModel {
        return dbSessionRetryable(tableClient) {
            providersDao.getById(roStaleSingleRetryableCommit(), providerId, Tenants.DEFAULT_TENANT_ID)
                .awaitSingle().orElseThrow()
        }!!
    }

    private suspend fun tryGetProvider(providerId: String): ProviderModel? {
        return dbSessionRetryable(tableClient) {
            providersDao.getById(roStaleSingleRetryableCommit(), providerId, Tenants.DEFAULT_TENANT_ID)
                .awaitSingle().orElse(null)
        }
    }

    private suspend fun getResources(providerId: String): List<ResourceModel> {
        return dbSessionRetryable(tableClient) {
            resourcesDao.getAllByProvider(roStaleSingleRetryableCommit(), providerId, Tenants.DEFAULT_TENANT_ID,
                false).awaitSingle()
        }!!
    }

    private suspend fun getResourceTypes(resourceTypeIds: Collection<String>): Map<String, ResourceTypeModel> {
        return dbSessionRetryable(tableClient) {
            resourceTypeIds.chunked(1000).map { resourceTypesDao.getByIds(roStaleSingleRetryableCommit(),
                it.map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }
                .flatten().associateBy { it.id }
        }!!
    }

    private suspend fun getAllServiceNodes(): List<ServiceNode> {
        return dbSessionRetryable(tableClient) {
            servicesDao.getAllServiceNodes(session).awaitSingle()
        }!!
    }

    private fun prepareHierarchy(nodes: List<ServiceNode>): ServiceHierarchy {
        val roots = mutableListOf<Long>()
        val childrenByParent = mutableMapOf<Long, MutableSet<Long>>()
        nodes.forEach {
            if (it.parentId == null) {
                roots.add(it.id)
            } else {
                childrenByParent.computeIfAbsent(it.parentId) { mutableSetOf() }.add(it.id)
            }
        }
        return ServiceHierarchy(roots, childrenByParent, nodes.size)
    }

    private suspend fun aggregateProvider(provider: ProviderModel,
                                          resources: List<ResourceModel>,
                                          resourceTypes: Map<String, ResourceTypeModel>,
                                          hierarchy: ServiceHierarchy,
                                          clock: Clock,
                                          aggregationParams: AlgorithmParams) = coroutineScope {
        if (resources.isEmpty() || hierarchy.roots.isEmpty()) {
            return@coroutineScope AggregatedQuotasStats.empty()
        }
        val resourcesCount = resources.size
        if (aggregationParams.concurrentAggregation) {
            val chunkSize = ceil((resources.size.toDouble() / 3.0)).toInt().coerceAtLeast(1)
            val jobs = resources.chunked(chunkSize).map {
                async {
                    return@async aggregateResourcesChunk(provider, it, resourceTypes, hierarchy, clock,
                        aggregationParams, resourcesCount)
                }
            }
            return@coroutineScope jobs.awaitAll().reduce { x, y -> x + y }
        } else {
            return@coroutineScope aggregateResourcesChunk(provider, resources, resourceTypes, hierarchy, clock,
                aggregationParams, resourcesCount)
        }
    }

    private suspend fun aggregateResourcesChunk(provider: ProviderModel,
                                                resources: List<ResourceModel>,
                                                resourceTypes: Map<String, ResourceTypeModel>,
                                                hierarchy: ServiceHierarchy,
                                                clock: Clock,
                                                aggregationParams: AlgorithmParams,
                                                resourcesCount: Int): AggregatedQuotasStats {
        meter({
            if (aggregationParams.paginateResources) {
                return resources.chunked(aggregationParams.resourcesPageSize.toInt()).map {
                    aggregateResourcesPage(provider, it, resourceTypes, hierarchy, clock, aggregationParams,
                        resourcesCount)
                }.reduce { x, y -> x + y }
            } else {
                return aggregateResourcesPage(provider, resources, resourceTypes, hierarchy, clock,
                    aggregationParams, resourcesCount)
            }
        }, "Provider ${provider.key} aggregation, resource chunk aggregation")
    }

    private suspend fun aggregateResourcesPage(provider: ProviderModel,
                                               resources: List<ResourceModel>,
                                               resourceTypes: Map<String, ResourceTypeModel>,
                                               hierarchy: ServiceHierarchy,
                                               clock: Clock,
                                               aggregationParams: AlgorithmParams,
                                               resourcesCount: Int): AggregatedQuotasStats {
        meter({
            val timestamp = Instant.now(clock)
            val resourceIds = resources.map { it.id }
            val loadAllProviderQuotas = resources.size == resourcesCount
            val quotas = meter( { getAggregationQuotas(provider.id, resourceIds, aggregationParams,
                loadAllProviderQuotas) }, "Provider ${provider.key} aggregation, reading quotas")
            val quotasIndex = meter( { withContext(computeDispatcher) { prepareQuotasIndex(quotas, resourceIds.toSet()) } },
                "Provider ${provider.key} aggregation, grouping quotas (${quotas.size} total)")
            val resourcesWithUsage = findResourcesWithUsage(provider, resources, resourceTypes).map { it.id }.toSet()
            val usage = if (resourcesWithUsage.isNotEmpty()) {
                meter( { getAggregationUsage(provider.id, resourcesWithUsage, aggregationParams,
                    loadAllProviderQuotas) }, "Provider ${provider.key} aggregation, reading usage")
            } else {
                listOf()
            }
            val usageIndex = if (usage.isNotEmpty()) {
                meter( { withContext(computeDispatcher) { prepareUsageIndex(usage, resourcesWithUsage) } },
                    "Provider ${provider.key} aggregation, grouping usage (${usage.size} total)")
            } else {
                UsageIndex(mapOf())
            }
            return aggregateQuotas(provider, resources, resourceTypes, quotasIndex, hierarchy, timestamp,
                aggregationParams, usageIndex)
        }, "Provider ${provider.key} aggregation, resource page aggregation")
    }

    private fun findResourcesWithUsage(provider: ProviderModel,
                                       resources: List<ResourceModel>,
                                       resourceTypes: Map<String, ResourceTypeModel>): List<ResourceModel> {
        val resourceById = resources.filter { !it.isDeleted }.associateBy { it.id }
        val resourceIdsWithUsage = resourceById.mapValues {
            val settings = resourceAggregationSettings(it.key, provider, resourceById, resourceTypes)
            val usageMode = settings.usageMode ?: UsageMode.UNDEFINED
            usageMode != UsageMode.UNDEFINED
        }.filterValues { it }.keys
        return resourceById.filterKeys { resourceIdsWithUsage.contains(it) }.values.toList()
    }

    private suspend fun aggregateQuotas(provider: ProviderModel,
                                        resources: List<ResourceModel>,
                                        resourceTypes: Map<String, ResourceTypeModel>,
                                        quotas: QuotasIndex,
                                        hierarchy: ServiceHierarchy,
                                        timestamp: Instant,
                                        aggregationParams: AlgorithmParams,
                                        usage: UsageIndex): AggregatedQuotasStats {
        val epochs = meter({ incrementAndGetEpochs(provider.id, resources.map { it.id }) },
            "Provider ${provider.key} aggregation, incrementing epochs (${resources.size} resources)")
        val aggregated = meter( { withContext(computeDispatcher) {
            prepareAggregates(provider, resources.associateBy { it.id }, resourceTypes, epochs, quotas, hierarchy,
                timestamp, usage) } }, "Provider ${provider.key} aggregation, preparing aggregates " +
            "(${hierarchy.nodesCount} services, ${hierarchy.roots.size} service roots)")
        meter( { upsertAggregates(aggregated, aggregationParams) }, "Provider ${provider.key} aggregation, " +
            "upserting aggregates (${aggregated.aggregates.size} aggregates, " +
            "${aggregated.denormalizedAggregates.size} denormalized aggregates)")
        val cleanupStats = meter( { cleanupPreviousEpochs(epochs, aggregationParams) },
            "Provider ${provider.key} aggregation, cleaning up old epochs aggregates")
        logger.info { "Provider ${provider.key} aggregation, cleaned up ${cleanupStats.staleAggregates} aggregates, " +
            "${cleanupStats.staleAggregateUsage} usage aggregates " +
            "and ${cleanupStats.staleDenormalizedAggregates} denormalized aggregates" }
        return AggregatedQuotasStats.from(aggregated)
    }

    private suspend fun cleanupPreviousEpochs(epochs: Map<String, AggregateEpochModel>,
                                              aggregationParams: AlgorithmParams): CleanupStats = coroutineScope {
        val staleAggregatesCounter = AtomicLong(0L)
        val staleAggregateUsageCounter = AtomicLong(0L)
        val staleDenormalizedAggregatesCounter = AtomicLong(0L)
        if (aggregationParams.concurrentCleanup) {
            epochs.values.chunked(3).forEach { page ->
                val jobs = page.map { async { cleanupPreviousEpochsPage(listOf(it), staleAggregatesCounter,
                    staleAggregateUsageCounter, staleDenormalizedAggregatesCounter) } }
                jobs.awaitAll()
            }
        } else {
            cleanupPreviousEpochsPage(epochs.values, staleAggregatesCounter, staleAggregateUsageCounter,
                staleDenormalizedAggregatesCounter)
        }
        CleanupStats(staleAggregatesCounter.get(), staleAggregateUsageCounter.get(),
            staleDenormalizedAggregatesCounter.get())
    }

    private suspend fun cleanupPreviousEpochsPage(epochs: Collection<AggregateEpochModel>,
                                                  staleAggregatesCounter: AtomicLong,
                                                  staleAggregateUsageCounter: AtomicLong,
                                                  staleDenormalizedAggregatesCounter: AtomicLong) {
        epochs.forEach { epoch ->
            var nextAggregate = dbSessionRetryable(tableClient) {
                rwTxRetryableOptimized(
                    { serviceAggregatesDao.getKeysForOlderEpochsFirstPage(txSession, epoch.key.tenantId,
                            epoch.key.providerId, epoch.key.resourceId, epoch.epoch) },
                    { page -> page },
                    { page ->
                        if (page.keys.isEmpty()) {
                            txSession.commitTransaction().awaitSingleOrNull()
                        } else {
                            staleAggregatesCounter.addAndGet(page.keys.size.toLong())
                            serviceAggregatesDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                        }
                        page.nextFrom
                    })
            }
            while (nextAggregate != null) {
                nextAggregate = dbSessionRetryable(tableClient) {
                    rwTxRetryableOptimized(
                        { serviceAggregatesDao.getKeysForOlderEpochsNextPage(txSession, nextAggregate!!) },
                        { page -> page },
                        { page ->
                            if (page.keys.isEmpty()) {
                                txSession.commitTransaction().awaitSingleOrNull()
                            } else {
                                staleAggregatesCounter.addAndGet(page.keys.size.toLong())
                                serviceAggregatesDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                            }
                            page.nextFrom
                        })
                }
            }
            var nextAggregateUsage = dbSessionRetryable(tableClient) {
                rwTxRetryableOptimized(
                    { serviceAggregateUsageDao.getKeysForOlderEpochsFirstPage(txSession, epoch.key.tenantId,
                        epoch.key.providerId, epoch.key.resourceId, epoch.epoch, 250) },
                    { page -> page },
                    { page ->
                        if (page.keys.isEmpty()) {
                            txSession.commitTransaction().awaitSingleOrNull()
                        } else {
                            staleAggregateUsageCounter.addAndGet(page.keys.size.toLong())
                            serviceAggregateUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                        }
                        page.nextFrom
                    })
            }
            while (nextAggregateUsage != null) {
                nextAggregateUsage = dbSessionRetryable(tableClient) {
                    rwTxRetryableOptimized(
                        { serviceAggregateUsageDao.getKeysForOlderEpochsNextPage(txSession, nextAggregateUsage!!, 250) },
                        { page -> page },
                        { page ->
                            if (page.keys.isEmpty()) {
                                txSession.commitTransaction().awaitSingleOrNull()
                            } else {
                                staleAggregateUsageCounter.addAndGet(page.keys.size.toLong())
                                serviceAggregateUsageDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                            }
                            page.nextFrom
                        })
                }
            }
            var nextDenormalizedAggregate = dbSessionRetryable(tableClient) {
                rwTxRetryableOptimized(
                    { serviceDenormalizedAggregatesDao.getKeysForOlderEpochsFirstPage(txSession, epoch.key.tenantId,
                            epoch.key.providerId, epoch.key.resourceId, epoch.epoch) },
                    { page -> page },
                    { page ->
                        if (page.keys.isEmpty()) {
                            txSession.commitTransaction().awaitSingleOrNull()
                        } else {
                            staleDenormalizedAggregatesCounter.addAndGet(page.keys.size.toLong())
                            serviceDenormalizedAggregatesDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                        }
                        page.nextFrom
                    })
            }
            while (nextDenormalizedAggregate != null) {
                nextDenormalizedAggregate = dbSessionRetryable(tableClient) {
                    rwTxRetryableOptimized(
                        { serviceDenormalizedAggregatesDao.getKeysForOlderEpochsNextPage(txSession,
                            nextDenormalizedAggregate!!) },
                        { page -> page },
                        { page ->
                            if (page.keys.isEmpty()) {
                                txSession.commitTransaction().awaitSingleOrNull()
                            } else {
                                staleDenormalizedAggregatesCounter.addAndGet(page.keys.size.toLong())
                                serviceDenormalizedAggregatesDao.deleteByIdsRetryable(txSession, page.keys.map { it.key })
                            }
                            page.nextFrom
                        })
                }
            }
        }
    }

    private suspend fun getAggregationQuotas(providerId: String,
                                             resourceIds: Collection<String>,
                                             aggregationParams: AlgorithmParams,
                                             loadAllProviderQuotas: Boolean): List<QuotaAggregationModel> {
        return when(aggregationParams.quotaQueryType) {
            AggregationQuotaQueryType.SCAN_SNAPSHOT -> dbSessionRetryable(tableClient) {
                val reserveAccounts = providerReserveAccountsDao.getAllByProvider(roStaleSingleRetryableCommit(),
                    Tenants.DEFAULT_TENANT_ID, providerId).map { it.key.accountId }.toSet()
                val quotasAndProvisions = quotasDao.scanQuotasAggregationSubset(session, Tenants.DEFAULT_TENANT_ID,
                    providerId, resourceIds, Duration.ofMinutes(5L)).awaitSingle()
                val quotas = quotasAndProvisions.filter { it.quota != null }
                val provisions = quotasAndProvisions.filter { it.provided != null }
                getQuotaAndProvisionsMinusReserveProvision(quotas, provisions, reserveAccounts)
            }!!
            AggregationQuotaQueryType.PAGINATE -> dbSessionRetryable(tableClient) {
                val reserveAccounts = providerReserveAccountsDao.getAllByProvider(roStaleSingleRetryableCommit(),
                    Tenants.DEFAULT_TENANT_ID, providerId).map { it.key.accountId }.toSet()
                if (loadAllProviderQuotas) {
                    val quotas = quotasDao.getForAggregationByProvider(roStaleSingleRetryableCommit(),
                        Tenants.DEFAULT_TENANT_ID, providerId).awaitSingle()
                    val provisions = accountsQuotasDao.getForAggregationByProvider(roStaleSingleRetryableCommit(),
                        Tenants.DEFAULT_TENANT_ID, providerId).awaitSingle()
                    getQuotaAndProvisionsMinusReserveProvision(quotas, provisions, reserveAccounts)
                } else {
                    val quotas = quotasDao.getForAggregationByProviderAndResources(roStaleSingleRetryableCommit(),
                        Tenants.DEFAULT_TENANT_ID, providerId, resourceIds).awaitSingle()
                    val provisions = accountsQuotasDao.getForAggregationByProviderAndResources(
                        roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID, providerId, resourceIds)
                        .awaitSingle()
                    getQuotaAndProvisionsMinusReserveProvision(quotas, provisions, reserveAccounts)
                }
            }!!
        }.filter {
                quotaAggregation -> quotaAggregation.folderType != FolderType.PROVIDER_RESERVE
        }
    }

    private fun getQuotaAndProvisionsMinusReserveProvision(quotas: List<QuotaAggregationModel>,
                                                           provisions: List<QuotaAggregationModel>,
                                                           reserveAccounts: Set<AccountId>): List<QuotaAggregationModel> {
        val reserveProvisions = HashMap<Pair<FolderId, ResourceId>, BigInteger>()
        for (provision in provisions) {
            if (reserveAccounts.contains(provision.accountId)) {
                reserveProvisions[Pair(provision.folderId, provision.resourceId)] =
                    reserveProvisions.getOrDefault(Pair(provision.folderId, provision.resourceId), BigInteger.ZERO)
                        .plus(BigInteger.valueOf(provision.provided!!))
            }
        }

        return quotas.map {
            QuotaAggregationModel(
                it.resourceId,
                BigInteger.valueOf(it.quota!!)
                    .minus(reserveProvisions.getOrDefault(Pair(it.folderId, it.resourceId), BigInteger.ZERO)).toLong(),
                it.balance,
                it.provided,
                it.allocated,
                it.serviceId,
                it.folderId,
                it.accountId,
                it.folderType
            )
        } + provisions.filter {
            !reserveAccounts.contains(it.accountId)
        }
    }

    private suspend fun getAggregationUsage(providerId: String,
                                            resourceIds: Collection<String>,
                                            aggregationParams: AlgorithmParams,
                                            loadAllProviderQuotas: Boolean): List<ServiceUsageModel> {
        return when(aggregationParams.quotaQueryType) {
            AggregationQuotaQueryType.SCAN_SNAPSHOT -> dbSessionRetryable(tableClient) {
                if (loadAllProviderQuotas) {
                    serviceUsageDao.scanByProvider(session, Tenants.DEFAULT_TENANT_ID, providerId,
                        Duration.ofMinutes(5L))
                } else {
                    serviceUsageDao.scanByProviderResources(session,  Tenants.DEFAULT_TENANT_ID, providerId,
                        resourceIds, Duration.ofMinutes(5L))
                }
            }!!
            AggregationQuotaQueryType.PAGINATE -> dbSessionRetryable(tableClient) {
                if (loadAllProviderQuotas) {
                    serviceUsageDao.getAllByProvider(roStaleSingleRetryableCommit(),
                        Tenants.DEFAULT_TENANT_ID, providerId, 250)
                } else {
                    serviceUsageDao.getAllByProviderResources(roStaleSingleRetryableCommit(), Tenants.DEFAULT_TENANT_ID,
                        providerId, resourceIds, 250)
                }
            }!!
        }
    }

    private suspend fun incrementAndGetEpochs(providerId: String,
                                              resourceIds: Collection<String>): Map<String, AggregateEpochModel> {
        return dbSessionRetryable(tableClient) {
            rwTxRetryable {
                val keys = resourceIds.map { AggregateEpochKey(Tenants.DEFAULT_TENANT_ID, providerId, it) }
                val epochs = serviceAggregatesEpochsDao.getByIds(txSession, keys).associateBy { it.key }
                val incrementedEpochs = keys.map {
                    val currentEpoch = epochs.getOrDefault(it, AggregateEpochModel(it, -1))
                    AggregateEpochModel(it, currentEpoch.epoch + 1)
                }
                serviceAggregatesEpochsDao.upsertManyRetryable(txSession, incrementedEpochs)
                incrementedEpochs.associateBy { it.key.resourceId }
            }!!
        }!!
    }

    private suspend fun upsertAggregates(aggregates: AggregatedQuotas,
                                         aggregationParams: AlgorithmParams) = coroutineScope {
        if (aggregationParams.concurrentAggregatesUpsert) {
            val aggregatesJob = async {
                dbSessionRetryable(tableClient) {
                    aggregates.aggregates.chunked(aggregationParams.upsertPageSize.toInt())
                        .forEach { serviceAggregatesDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
                }
            }
            val aggregateUsageJob = async {
                dbSessionRetryable(tableClient) {
                    aggregates.usageAggregates.chunked(aggregationParams.upsertPageSize.toInt())
                        .forEach { serviceAggregateUsageDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
                }
            }
            val denormalizedAggregatesJob = async {
                dbSessionRetryable(tableClient) {
                    aggregates.denormalizedAggregates.chunked(aggregationParams.upsertPageSize.toInt())
                        .forEach { serviceDenormalizedAggregatesDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
                }
            }
            listOf(aggregatesJob, aggregateUsageJob, denormalizedAggregatesJob).awaitAll()
        } else {
            dbSessionRetryable(tableClient) {
                aggregates.aggregates.chunked(aggregationParams.upsertPageSize.toInt())
                    .forEach { serviceAggregatesDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
                aggregates.usageAggregates.chunked(aggregationParams.upsertPageSize.toInt())
                    .forEach { serviceAggregateUsageDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
                aggregates.denormalizedAggregates.chunked(aggregationParams.upsertPageSize.toInt())
                    .forEach { serviceDenormalizedAggregatesDao.upsertManyRetryable(rwSingleRetryableCommit(), it) }
            }
        }
    }

    private fun prepareQuotasIndex(quotas: List<QuotaAggregationModel>, resourceIds: Set<ResourceId>): QuotasIndex {
        val index = mutableMapOf<ServiceId, MutableMap<ResourceId, MutableSet<QuotaAggregationModel>>>()
        quotas.forEach {
            if (resourceIds.contains(it.resourceId)) {
                ((index.computeIfAbsent(it.serviceId) { mutableMapOf() })
                    .computeIfAbsent(it.resourceId) { mutableSetOf() }).add(it)
            }
        }
        return QuotasIndex(index)
    }

    private fun prepareUsageIndex(usage: List<ServiceUsageModel>, resourceIds: Set<ResourceId>): UsageIndex {
        val index = mutableMapOf<ServiceId, MutableMap<ResourceId, ServiceUsageModel>>()
        usage.forEach {
            if (resourceIds.contains(it.key.resourceId)) {
                (index.computeIfAbsent(it.key.serviceId) { mutableMapOf() })[it.key.resourceId] = it
            }
        }
        return UsageIndex(index)
    }

    private inline fun <T> meter(statement: () -> T,
                                 label: String): T {
        return elapsed(statement)
            { millis, success -> logger.info { "$label: duration = $millis ms, success = $success" } }
    }

    private fun prepareParams(provider: ProviderModel): AlgorithmParams {
        if (provider.aggregationAlgorithm.isEmpty) {
            return AlgorithmParams(
                disabled = AggregationAlgorithmHelper.DEFAULT_DISABLED_FLAG,
                upsertPageSize = AggregationAlgorithmHelper.DEFAULT_UPSERT_PAGE_SIZE,
                quotaQueryType = AggregationAlgorithmHelper.DEFAULT_QUOTA_QUERY_TYPE,
                concurrentAggregatesUpsert = AggregationAlgorithmHelper.DEFAULT_CONCURRENT_AGGREGATES_UPSERT,
                concurrentAggregation = AggregationAlgorithmHelper.DEFAULT_CONCURRENT_AGGREGATION,
                resourcesPageSize = AggregationAlgorithmHelper.DEFAULT_RESOURCES_PAGE_SIZE,
                paginateResources = AggregationAlgorithmHelper.DEFAULT_PAGINATE_RESOURCES,
                concurrentCleanup = AggregationAlgorithmHelper.DEFAULT_CONCURRENT_CLEANUP
            )
        }
        val algorithm = provider.aggregationAlgorithm.get()
        return AlgorithmParams(
            disabled = algorithm.disabled ?: AggregationAlgorithmHelper.DEFAULT_DISABLED_FLAG,
            upsertPageSize = algorithm.upsertPageSize ?: AggregationAlgorithmHelper.DEFAULT_UPSERT_PAGE_SIZE,
            quotaQueryType = algorithm.quotaQueryType ?: AggregationAlgorithmHelper.DEFAULT_QUOTA_QUERY_TYPE,
            concurrentAggregatesUpsert = algorithm.concurrentAggregatesUpsert
                ?: AggregationAlgorithmHelper.DEFAULT_CONCURRENT_AGGREGATES_UPSERT,
            concurrentAggregation = algorithm.concurrentAggregation
                ?: AggregationAlgorithmHelper.DEFAULT_CONCURRENT_AGGREGATION,
            resourcesPageSize = algorithm.resourcesPageSize ?: AggregationAlgorithmHelper.DEFAULT_RESOURCES_PAGE_SIZE,
            paginateResources = algorithm.paginateResources ?: AggregationAlgorithmHelper.DEFAULT_PAGINATE_RESOURCES,
            concurrentCleanup = algorithm.concurrentCleanup ?: AggregationAlgorithmHelper.DEFAULT_CONCURRENT_CLEANUP
        )
    }

    private data class AggregatedQuotasStats(
        val aggregatesCount: Int,
        val denormalizedAggregatesCount: Int,
        val maxPerResourceDenormalizedAggregatesCount: Int
    ) {
        companion object {
            fun from(aggregates: AggregatedQuotas): AggregatedQuotasStats {
                val maxCountByResource = aggregates.denormalizedAggregates.groupingBy { it.key.resourceId }
                    .eachCount()
                    .maxOfOrNull { it.value } ?: 0
                return AggregatedQuotasStats(aggregates.aggregates.size, aggregates.denormalizedAggregates.size,
                    maxCountByResource)
            }

            fun empty() = AggregatedQuotasStats(0, 0, 0)
        }

        operator fun plus(stats: AggregatedQuotasStats): AggregatedQuotasStats {
            return AggregatedQuotasStats(
                aggregatesCount = this.aggregatesCount + stats.aggregatesCount,
                denormalizedAggregatesCount = this.denormalizedAggregatesCount + stats.denormalizedAggregatesCount,
                maxPerResourceDenormalizedAggregatesCount = max(this.maxPerResourceDenormalizedAggregatesCount,
                    stats.maxPerResourceDenormalizedAggregatesCount)
            )
        }
    }

    private data class ProviderMetrics(
        val timeSinceLastSuccess: LazyGaugeInt64,
        val aggregatesCount: GaugeInt64,
        val maxAggregatesCountByResource: GaugeInt64,
    )

    private data class CleanupStats(
        val staleAggregates: Long,
        val staleAggregateUsage: Long,
        val staleDenormalizedAggregates: Long
    )

    private data class AlgorithmParams(
        val disabled: Boolean,
        val upsertPageSize: Long,
        val quotaQueryType: AggregationQuotaQueryType,
        val concurrentAggregatesUpsert: Boolean,
        val concurrentAggregation: Boolean,
        val resourcesPageSize: Long,
        val paginateResources: Boolean,
        val concurrentCleanup: Boolean
    )

}
