package ru.yandex.intranet.d.services.delivery

import kotlinx.coroutines.reactor.awaitSingle
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import reactor.core.publisher.Mono
import reactor.util.function.Tuples
import ru.yandex.intranet.d.dao.Tenants
import ru.yandex.intranet.d.dao.accounts.AccountsDao
import ru.yandex.intranet.d.dao.accounts.AccountsSpacesDao
import ru.yandex.intranet.d.dao.accounts.FolderProviderAccountsSpace
import ru.yandex.intranet.d.dao.folders.FolderDao
import ru.yandex.intranet.d.dao.services.ServicesDao
import ru.yandex.intranet.d.datasource.dbSessionRetryable
import ru.yandex.intranet.d.datasource.model.YdbTableClient
import ru.yandex.intranet.d.i18n.Locales
import ru.yandex.intranet.d.kotlin.binding
import ru.yandex.intranet.d.kotlin.mono
import ru.yandex.intranet.d.loaders.providers.ProvidersLoader
import ru.yandex.intranet.d.loaders.resources.ResourcesLoader
import ru.yandex.intranet.d.loaders.users.UsersLoader
import ru.yandex.intranet.d.model.WithTenant
import ru.yandex.intranet.d.model.accounts.AccountModel
import ru.yandex.intranet.d.model.accounts.AccountSpaceModel
import ru.yandex.intranet.d.model.folders.FolderModel
import ru.yandex.intranet.d.model.providers.ProviderModel
import ru.yandex.intranet.d.model.resources.ResourceModel
import ru.yandex.intranet.d.model.services.ServiceMinimalModel
import ru.yandex.intranet.d.services.delivery.model.DeliveryDestinationDictionary
import ru.yandex.intranet.d.services.delivery.model.DeliveryDestinationValidationDictionary
import ru.yandex.intranet.d.services.delivery.model.PreValidatedDeliveryDestination
import ru.yandex.intranet.d.services.delivery.model.PreValidatedDeliveryDestinationRequest
import ru.yandex.intranet.d.services.delivery.model.ValidatedDeliveryDestination
import ru.yandex.intranet.d.services.delivery.model.ValidatedDeliveryDestinationRequest
import ru.yandex.intranet.d.services.security.SecurityManagerService
import ru.yandex.intranet.d.services.validators.AbcServiceValidator
import ru.yandex.intranet.d.util.Uuids
import ru.yandex.intranet.d.util.result.ErrorCollection
import ru.yandex.intranet.d.util.result.Result
import ru.yandex.intranet.d.util.result.TypedError
import ru.yandex.intranet.d.web.model.delivery.DeliveryAccountDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryAccountsSpaceDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryDestinationDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryDestinationProvidersDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryDestinationRequestDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryDestinationResourceDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryDestinationResponseDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryProviderDto
import ru.yandex.intranet.d.web.model.delivery.DeliveryResourcesAccountsDto
import ru.yandex.intranet.d.web.security.model.YaUserDetails
import java.util.*

/**
 * Delivery destination service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
class DeliveryDestinationService(@Qualifier("messageSource") private val messages: MessageSource,
                                 private val usersLoader: UsersLoader,
                                 private val resourcesLoader: ResourcesLoader,
                                 private val servicesDao: ServicesDao,
                                 private val securityManagerService: SecurityManagerService,
                                 private val providersLoader: ProvidersLoader,
                                 private val accountsSpacesDao: AccountsSpacesDao,
                                 private val accountsDao: AccountsDao,
                                 private val folderDao: FolderDao,
                                 private val ydbTableClient: YdbTableClient) {

    fun getDestinationsMono(request: DeliveryDestinationRequestDto,
                            locale: Locale): Mono<Result<DeliveryDestinationResponseDto>> {
        return mono { getDestinations(request, locale) }
    }

    suspend fun getDestinations(request: DeliveryDestinationRequestDto,
                                locale: Locale): Result<DeliveryDestinationResponseDto> = binding {
        val preValidatedRequest = preValidateRequest(request, locale).bind()!!
        val preDictionary = preLoadDictionaries(preValidatedRequest)
        val validatedRequest = validateRequestAndPermissions(preValidatedRequest, preDictionary, locale).bind()!!
        val dictionary = loadDictionaries(preDictionary)
        val result = findDestinations(validatedRequest, preDictionary, dictionary, locale)
        Result.success(result)
    }

    private suspend fun findDestinations(request: ValidatedDeliveryDestinationRequest,
                                         preDictionary: DeliveryDestinationValidationDictionary,
                                         dictionary: DeliveryDestinationDictionary,
                                         locale: Locale): DeliveryDestinationResponseDto {
        val serviceIds = request.deliverables.map { it.service.id }.distinct().toList()
        val folders = dbSessionRetryable(ydbTableClient) {
            serviceIds.chunked(500).flatMap { folderDao
                .getAllFoldersByServiceIds(roStaleSingleRetryableCommit(), it
                    .map { v -> WithTenant(Tenants.DEFAULT_TENANT_ID, v) }.toSet()).awaitSingle().get() }.toList()
        }!!.filterNot { it.isDeleted }
        val foldersByService = folders.groupBy { it.serviceId }
        val accountFilter = mutableSetOf<FolderProviderAccountsSpace>()
        request.deliverables.forEach { d -> foldersByService.getOrDefault(d.service.id, listOf())
            .forEach{ f -> d.resources.forEach { r -> accountFilter.add(FolderProviderAccountsSpace(
                Tenants.DEFAULT_TENANT_ID, f.id, r.providerId, r.accountsSpacesId)) } } }
        val accounts = dbSessionRetryable(ydbTableClient) {
            accountFilter.chunked(500).flatMap { accountsDao.getAllByFoldersProvidersAccountsSpaces(
                roStaleSingleRetryableCommit(), it.toSet(), false).awaitSingle() }.toList()
        }!!
        return buildResponse(request, preDictionary, dictionary, folders, accounts, locale)
    }

    private fun buildResponse(request: ValidatedDeliveryDestinationRequest,
                              preDictionary: DeliveryDestinationValidationDictionary,
                              dictionary: DeliveryDestinationDictionary,
                              folders: List<FolderModel>,
                              accounts: List<AccountModel>,
                              locale: Locale): DeliveryDestinationResponseDto {
        val resources = mutableSetOf<DeliveryDestinationResourceDto>()
        preDictionary.resources.map { r ->
            val eligible = !r.isReadOnly &&r.isManaged
            val ineligibilityReasons = mutableSetOf<String>()
            if (r.isReadOnly) {
                ineligibilityReasons.add(messages.getMessage("errors.resource.is.read.only", null, locale))
            }
            if (!r.isManaged) {
                ineligibilityReasons.add(messages.getMessage("errors.resource.not.managed", null, locale))
            }
            resources.add(DeliveryDestinationResourceDto(r.id, r.key, Locales.select(r.nameEn, r.nameRu, locale),
                eligible, ineligibilityReasons.toSet()))
        }
        val providersById = dictionary.providers.associateBy { it.id }
        val accountsSpacesById = dictionary.accountsSpaces.associateBy { it.id }
        val foldersById = folders.associateBy { it.id }
        val accountsByServiceProvider = accounts.groupBy { foldersById[it.folderId]!!.serviceId }
            .mapValues { e -> e.value.groupBy { a -> a.providerId } }
        val destinations = mutableSetOf<DeliveryDestinationProvidersDto>()
        request.deliverables.forEach { d ->
            val eligibleState = AbcServiceValidator.ALLOWED_SERVICE_STATES.contains(d.service.state)
                && (d.service.readOnlyState == null || AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES
                .contains(d.service.readOnlyState))
            val eligibleExportability = d.service.isExportable
            val serviceEligible = eligibleState && eligibleExportability
            val serviceIneligibilityReasons = mutableSetOf<String>()
            if (!eligibleState) {
                serviceIneligibilityReasons.add(messages.getMessage("errors.service.bad.status", null, locale))
            }
            if (!eligibleExportability) {
                serviceIneligibilityReasons.add(messages.getMessage("errors.service.is.non.exportable", null, locale))
            }
            val serviceAccountsByProvider = accountsByServiceProvider.getOrDefault(d.service.id, mapOf())
            val destinationProviders = mutableSetOf<DeliveryProviderDto>()
            d.resources.groupBy { it.providerId }.forEach { (providerId, resources) ->
                val provider = providersById[providerId]!!
                val providerAccounts = serviceAccountsByProvider.getOrDefault(providerId, listOf())
                val providerEligible = !provider.isReadOnly && provider.isManaged
                val providerIneligibilityReasons = mutableSetOf<String>()
                if (provider.isReadOnly) {
                    providerIneligibilityReasons.add(messages.getMessage("errors.provider.is.read.only", null, locale))
                }
                if (!provider.isManaged) {
                    providerIneligibilityReasons.add(messages
                        .getMessage("errors.provider.is.not.managed", null, locale))
                }
                val resourcesInAccountsSpaces = resources.filter { it.accountsSpacesId != null }
                val accountsInAccountsSpaces = providerAccounts.filter { it.accountsSpacesId.isPresent }
                val resourcesWithoutAccountsSpaces = resources.filter { it.accountsSpacesId == null }
                val accountsWithoutAccountsSpaces = providerAccounts.filter { it.accountsSpacesId.isEmpty }
                val destinationAccountsSpaces = mutableSetOf<DeliveryAccountsSpaceDto>()
                val destinationProviderAccounts = mutableSetOf<DeliveryAccountDto>()
                val destinationProviderResourceIds = mutableSetOf<String>()
                if (resourcesWithoutAccountsSpaces.isNotEmpty()) {
                    destinationProviderResourceIds.addAll(resourcesWithoutAccountsSpaces.map { it.id })
                    accountsWithoutAccountsSpaces.forEach { a -> destinationProviderAccounts.add(DeliveryAccountDto(
                        a.id, a.outerAccountIdInProvider, a.outerAccountKeyInProvider.orElse(null),
                        a.displayName.orElse(null), a.folderId, foldersById[a.folderId]!!.displayName)) }
                }
                if (resourcesInAccountsSpaces.isNotEmpty()) {
                    val accountsByAccountsSpace = accountsInAccountsSpaces.groupBy { it.accountsSpacesId.orElseThrow() }
                    resourcesInAccountsSpaces.groupBy { it.accountsSpacesId!! }
                        .forEach { (accountsSpaceId, accountsSpaceResources) ->
                            val accountSpace = accountsSpacesById[accountsSpaceId]!!
                            val accountsSpaceEligible = !accountSpace.isReadOnly
                            val accountsSpaceIneligibilityReasons = mutableSetOf<String>()
                            if (accountSpace.isReadOnly) {
                                accountsSpaceIneligibilityReasons.add(messages
                                    .getMessage("errors.provider.account.space.is.read.only", null, locale))
                            }
                            val destinationAccounts = mutableSetOf<DeliveryAccountDto>()
                            val destinationResourceIds = mutableSetOf<String>()
                            destinationResourceIds.addAll(accountsSpaceResources.map { it.id })
                            accountsByAccountsSpace.getOrDefault(accountSpace.id, listOf()).forEach { a ->
                                destinationAccounts.add(DeliveryAccountDto(a.id, a.outerAccountIdInProvider,
                                    a.outerAccountKeyInProvider.orElse(null), a.displayName.orElse(null),
                                    a.folderId, foldersById[a.folderId]!!.displayName)) }
                            destinationAccountsSpaces.add(DeliveryAccountsSpaceDto(accountSpace.id,
                                Locales.select(accountSpace.nameEn, accountSpace.nameRu, locale),
                                accountSpace.outerKeyInProvider, accountsSpaceEligible,
                                accountsSpaceIneligibilityReasons.toSet(), DeliveryResourcesAccountsDto(
                                    destinationResourceIds.toSet(), destinationAccounts.toSet())))
                        }
                }
                val resourcesAccounts = if (destinationProviderResourceIds.isNotEmpty()
                    || destinationProviderAccounts.isNotEmpty()) {
                    DeliveryResourcesAccountsDto(destinationProviderResourceIds.toSet(),
                        destinationProviderAccounts.toSet())
                } else {
                    null
                }
                destinationProviders.add(DeliveryProviderDto(provider.id, Locales.select(provider.nameEn,
                    provider.nameRu, locale), provider.key, providerEligible, providerIneligibilityReasons.toSet(),
                    destinationAccountsSpaces.toSet(), resourcesAccounts))
            }
            destinations.add(DeliveryDestinationProvidersDto(d.service.id, d.quotaRequestId, serviceEligible,
                serviceIneligibilityReasons.toSet(), destinationProviders.toSet()))
        }
        return DeliveryDestinationResponseDto(destinations.toSet(), resources.toSet())
    }

    private suspend fun validateRequestAndPermissions(
        preValidatedRequest: PreValidatedDeliveryDestinationRequest,
        dictionary: DeliveryDestinationValidationDictionary,
        locale: Locale): Result<ValidatedDeliveryDestinationRequest> = binding {
        if (dictionary.user != null) {
            val userDetails = YaUserDetails.fromUser(dictionary.user)
            securityManagerService.checkReadPermissions(userDetails, locale).awaitSingle().bind()
        }
        validateRequest(preValidatedRequest, dictionary, locale)
    }

    private fun validateRequest(preValidatedRequest: PreValidatedDeliveryDestinationRequest,
                                dictionary: DeliveryDestinationValidationDictionary,
                                locale: Locale): Result<ValidatedDeliveryDestinationRequest> {
        val errors = ErrorCollection.builder()
        val builder = ValidatedDeliveryDestinationRequest.Builder()
        if (dictionary.user == null || dictionary.user.staffDismissed.orElse(false)) {
            errors.addError("userUid", TypedError.invalid(messages
                .getMessage("errors.user.not.found", null, locale)))
        } else {
            builder.user(dictionary.user)
        }
        val servicesById = dictionary.services.associateBy { it.id }
        val resourcesById = dictionary.resources.associateBy { it.id }
        val deliverables = mutableListOf<ValidatedDeliveryDestination>()
        val deliverablesErrors = ErrorCollection.builder()
        for (index in preValidatedRequest.deliverables.indices) {
            val deliverable = preValidatedRequest.deliverables[index]
            validateDeliverable(deliverable, deliverables::add, servicesById, resourcesById,
                deliverablesErrors, "deliverables.$index", locale)
        }
        if (deliverablesErrors.hasAnyErrors()) {
            errors.add(deliverablesErrors)
        } else {
            builder.addDeliverables(deliverables)
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(builder.build())
    }

    private fun validateDeliverable(preValidated: PreValidatedDeliveryDestination,
                                    setter: (ValidatedDeliveryDestination) -> Unit,
                                    servicesById: Map<Long, ServiceMinimalModel>,
                                    resourcesById: Map<String, ResourceModel>,
                                    errors: ErrorCollection.Builder,
                                    field: String,
                                    locale: Locale) {
        val localErrors = ErrorCollection.builder()
        val builder = ValidatedDeliveryDestination.Builder()
        builder.quotaRequestId(preValidated.quotaRequestId)
        val service = servicesById[preValidated.serviceId]
        if (service == null) {
            localErrors.addError("$field.serviceId", TypedError.invalid(messages
                .getMessage("errors.service.not.found", null, locale)))
        } else {
            builder.service(service)
        }
        val resources = mutableListOf<ResourceModel>()
        val resourcesErrors = ErrorCollection.builder()
        for (index in preValidated.resourceIds.indices) {
            val resourceId = preValidated.resourceIds[index]
            val resource = resourcesById[resourceId]
            if (resource == null || resource.isDeleted) {
                resourcesErrors.addError("$field.resources.$index", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)))
            } else {
                resources.add(resource)
            }
        }
        if (resourcesErrors.hasAnyErrors()) {
            localErrors.add(resourcesErrors)
        } else {
            builder.addResources(resources)
        }
        if (localErrors.hasAnyErrors()) {
            errors.add(localErrors)
        } else {
            setter(builder.build())
        }
    }

    private suspend fun preLoadDictionaries(
        preValidatedRequest: PreValidatedDeliveryDestinationRequest): DeliveryDestinationValidationDictionary {
        val userO = usersLoader.getUserByPassportUidImmediate(preValidatedRequest.userUid, Tenants.DEFAULT_TENANT_ID)
            .awaitSingle()
        val resourceIds = preValidatedRequest.deliverables.flatMap { it.resourceIds }
            .distinct().toList()
        val resources = resourceIds.chunked(500).flatMap { a -> resourcesLoader.getResourcesByIdsImmediate(a
            .map { b -> Tuples.of(b, Tenants.DEFAULT_TENANT_ID) }).awaitSingle() }.toList()
        val serviceIds = preValidatedRequest.deliverables.map { it.serviceId }.distinct().toList()
        val services = dbSessionRetryable(ydbTableClient) {
            serviceIds.chunked(500).flatMap { servicesDao.getByIdsMinimal(roStaleSingleRetryableCommit(), it)
                .awaitSingle() }.toList()
        }!!
        return DeliveryDestinationValidationDictionary(userO.orElse(null), services, resources)
    }

    private suspend fun loadDictionaries(
        validationDictionary: DeliveryDestinationValidationDictionary
    ): DeliveryDestinationDictionary {
        val providerIds = validationDictionary.resources.map { it.providerId }.distinct().toList()
        val accountsSpacesIds = validationDictionary.resources.mapNotNull { it.accountsSpacesId }.distinct().toList()
        val providers = if (providerIds.isNotEmpty()) {
            providerIds.chunked(500).flatMap { providersLoader.getProvidersByIdsImmediate(it
                .map { id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID) }.toList()).awaitSingle() }.toList()
        } else {
            listOf<ProviderModel>()
        }
        val accountsSpaces = if (accountsSpacesIds.isNotEmpty()) {
            dbSessionRetryable(ydbTableClient) {
                accountsSpacesIds.chunked(500).flatMap { accountsSpacesDao
                    .getByIds(roStaleSingleRetryableCommit(), it, Tenants.DEFAULT_TENANT_ID).awaitSingle() }.toList()
            }!!
        } else {
            listOf<AccountSpaceModel>()
        }
        return DeliveryDestinationDictionary(providers, accountsSpaces)
    }

    private fun preValidateRequest(request: DeliveryDestinationRequestDto,
                                   locale: Locale): Result<PreValidatedDeliveryDestinationRequest> {
        val errors = ErrorCollection.builder()
        val builder = PreValidatedDeliveryDestinationRequest.Builder()
        preValidateUid(request.userUid, builder::userUid, errors, locale)
        preValidateDeliverables(request.deliverables, builder::addDeliverables, errors, locale)
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(builder.build())
    }

    private fun preValidateUid(
        uid: String?, setter: (String) -> Unit,
        errors: ErrorCollection.Builder, locale: Locale
    ) {
        if (uid == null) {
            errors.addError("userUid", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else if (uid.isBlank()) {
            errors.addError("userUid", TypedError.invalid(messages
                .getMessage("errors.non.blank.text.is.required", null, locale)))
        } else if (uid.length > 1024) {
            errors.addError("userUid", TypedError.invalid(messages
                .getMessage("errors.text.is.too.long", null, locale)))
        } else {
            setter(uid)
        }
    }

    private fun preValidateDeliverables(deliverables: List<DeliveryDestinationDto?>?,
                                        setter: (Collection<PreValidatedDeliveryDestination>) -> Unit,
                                        errors: ErrorCollection.Builder, locale: Locale
    ) {
        if (deliverables == null || deliverables.isEmpty()) {
            errors.addError("deliverables", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
            return
        }
        val preValidatedDeliverables = mutableListOf<PreValidatedDeliveryDestination>()
        val deliverablesErrors = ErrorCollection.builder()
        for (index in deliverables.indices) {
            val deliverable = deliverables[index]
            preValidateDeliverable(deliverable, preValidatedDeliverables::add, deliverablesErrors,
                "deliverables.$index", locale)
        }
        if (deliverablesErrors.hasAnyErrors()) {
            errors.add(deliverablesErrors)
        } else {
            setter(preValidatedDeliverables)
        }
    }

    private fun preValidateDeliverable(deliverable: DeliveryDestinationDto?,
                                       setter: (PreValidatedDeliveryDestination) -> Unit,
                                       errors: ErrorCollection.Builder, fieldKey: String, locale: Locale) {
        if (deliverable == null) {
            errors.addError(fieldKey, TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
            return
        }
        val localErrors = ErrorCollection.builder()
        val builder = PreValidatedDeliveryDestination.Builder()
        if (deliverable.serviceId == null) {
            localErrors.addError("$fieldKey.serviceId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            builder.serviceId(deliverable.serviceId)
        }
        if (deliverable.quotaRequestId == null) {
            localErrors.addError("$fieldKey.quotaRequestId", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            builder.quotaRequestId(deliverable.quotaRequestId)
        }
        if (deliverable.resourceIds == null || deliverable.resourceIds.isEmpty()) {
            localErrors.addError("$fieldKey.resourceIds", TypedError.invalid(messages
                .getMessage("errors.field.is.required", null, locale)))
        } else {
            val validatedResourceIds = mutableListOf<String>()
            val resourceIdsErrors = ErrorCollection.builder()
            for (resourceIndex in deliverable.resourceIds.indices) {
                val resourceId = deliverable.resourceIds[resourceIndex]
                if (resourceId == null) {
                    resourceIdsErrors.addError("$fieldKey.resourceIds.$resourceIndex",
                        TypedError.invalid(messages.getMessage("errors.field.is.required", null, locale)))
                    continue
                }
                if (!Uuids.isValidUuid(resourceId)) {
                    resourceIdsErrors.addError("$fieldKey.resourceIds.$resourceIndex",
                        TypedError.invalid(messages.getMessage("errors.resource.not.found", null, locale)))
                } else {
                    validatedResourceIds.add(resourceId)
                }
            }
            if (resourceIdsErrors.hasAnyErrors()) {
                localErrors.add(resourceIdsErrors)
            } else {
                builder.addResourceIds(validatedResourceIds)
            }
        }
        if (localErrors.hasAnyErrors()) {
            errors.add(localErrors)
        } else {
            setter(builder.build())
        }
    }

}
