package ru.yandex.intranet.d.services.delivery;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.Sets;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Mono;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.datasource.model.YdbTableClient;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.delivery.DeliverableFolderOperationModel;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaHistoryModel;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaResponseModel;
import ru.yandex.intranet.d.model.delivery.DeliverableRequestModel;
import ru.yandex.intranet.d.model.delivery.DeliverableResponseModel;
import ru.yandex.intranet.d.model.delivery.DeliveryResponseModel;
import ru.yandex.intranet.d.model.delivery.FinishedDeliveryModel;
import ru.yandex.intranet.d.model.folders.FolderHistoryFieldsModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.folders.FolderOperationType;
import ru.yandex.intranet.d.model.folders.QuotasByAccount;
import ru.yandex.intranet.d.model.folders.QuotasByResource;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.services.delivery.model.DeliverableRequestGroupKey;
import ru.yandex.intranet.d.services.delivery.model.DeliveryDictionary;
import ru.yandex.intranet.d.services.delivery.model.ValidatedDeliveryRequest;
import ru.yandex.intranet.d.services.delivery.model.ValidatedQuotas;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.delivery.DeliverableFolderOperationDto;
import ru.yandex.intranet.d.web.model.delivery.DeliverableMetaResponseDto;
import ru.yandex.intranet.d.web.model.delivery.DeliverableResponseDto;
import ru.yandex.intranet.d.web.model.delivery.DeliveryRequestDto;
import ru.yandex.intranet.d.web.model.delivery.DeliveryResponseDto;

/**
 * Delivery service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class DeliveryService {

    private final DeliveryValidationService validationService;
    private final DeliveryStoreService storeService;
    private final YdbTableClient tableClient;

    @SuppressWarnings("ParameterNumber")
    public DeliveryService(
            DeliveryValidationService validationService,
            DeliveryStoreService storeService,
            YdbTableClient tableClient
    ) {
        this.validationService = validationService;
        this.storeService = storeService;
        this.tableClient = tableClient;
    }

    public Mono<Result<DeliveryResponseDto>> deliver(DeliveryRequestDto request, Locale locale) {
        return validationService.preValidateRequest(request, locale).andThenMono(delivery -> inTx(ts ->
                storeService.getDeliveryById(ts, delivery.getDeliveryId()).flatMap(finishedDeliveryO -> {
                    if (finishedDeliveryO.isPresent()) {
                        return validationService.validateRequestMatch(delivery, finishedDeliveryO.get(), locale)
                                .andThenMono(v -> Mono.just(Result.success(toResponse(finishedDeliveryO.get()))));
                    }
                    return storeService.loadDictionaries(ts, delivery).flatMap(dict ->
                            validationService.validateRequest(delivery, dict, locale).andThenMono(validated ->
                                    storeService.loadQuotas(ts, validated, dict).flatMap(quotas ->
                                            validationService.validateQuotas(delivery, quotas, dict, validated, locale)
                                                    .applyMono(validatedQuotas -> applyDelivery(ts, dict, validated,
                                                            validatedQuotas)))));
        })));
    }

    private Mono<DeliveryResponseDto> applyDelivery(YdbTxSession session,
                                                    DeliveryDictionary dictionary,
                                                    ValidatedDeliveryRequest validatedRequest,
                                                    ValidatedQuotas quotas) {
        Instant now = Instant.now();
        List<FolderOperationLogModel> newFoldersOpLogs = prepareDefaultFolderCreationOpLogs(validatedRequest, now);
        Map<DeliverableRequestGroupKey, FolderOperationLogModel> opLogsByKey = prepareOpLogs(dictionary,
                validatedRequest, quotas, now);
        List<FolderOperationLogModel> opLogs = Stream.concat(opLogsByKey.values().stream(), newFoldersOpLogs.stream())
                .collect(Collectors.toList());
        List<QuotaModel> updatedQuotas = new ArrayList<>(quotas.getUpdatedQuotas().values());
        FinishedDeliveryModel finishedDelivery = prepareFinishedDelivery(validatedRequest, opLogsByKey);
        List<FolderModel> foldersToUpsert = updateNextOpLogOrder(validatedRequest.getNewDefaultFolders(),
                dictionary, opLogsByKey.values());
        return storeService.upsertFolders(session, foldersToUpsert)
                .then(Mono.defer(() -> storeService.upsertQuotas(session, updatedQuotas)))
                .then(Mono.defer(() -> storeService.upsertOperationLog(session, opLogs)))
                .then(Mono.defer(() -> storeService.upsertDelivery(session, finishedDelivery)))
                .thenReturn(toResponse(finishedDelivery));
    }

    private List<FolderModel> updateNextOpLogOrder(List<FolderModel> newDefaultFolders, DeliveryDictionary dictionary,
                                                   Collection<FolderOperationLogModel> deliveryOpLogs) {
        List<FolderModel> result = new ArrayList<>();
        Map<String, FolderModel> knownFolders = new HashMap<>();
        dictionary.getFolders().forEach(knownFolders::put);
        dictionary.getDefaultFolders().forEach((k, v) -> knownFolders.put(v.getId(), v));
        newDefaultFolders.forEach(v -> knownFolders.put(v.getId(), v));
        Map<String, Long> nextOpLogOrders = new HashMap<>();
        newDefaultFolders.forEach(v -> nextOpLogOrders.put(v.getId(), v.getNextOpLogOrder()));
        deliveryOpLogs.forEach(opLog -> {
            String folderId = opLog.getFolderId();
            long nextOpLogOrder = nextOpLogOrders.computeIfAbsent(folderId, id -> knownFolders.get(id)
                    .getNextOpLogOrder());
            nextOpLogOrders.put(folderId, nextOpLogOrder + 1L);
        });
        nextOpLogOrders.forEach((k, v) -> result.add(knownFolders.get(k).toBuilder().setNextOpLogOrder(v).build()));
        return result;
    }

    private List<FolderOperationLogModel> prepareDefaultFolderCreationOpLogs(
            ValidatedDeliveryRequest validatedRequest, Instant now) {
        List<FolderOperationLogModel> result = new ArrayList<>();
        validatedRequest.getNewDefaultFolders().forEach(newDefaultFolder -> {
            FolderOperationLogModel opLog = FolderOperationLogModel.builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setFolderId(newDefaultFolder.getId())
                    .setOperationDateTime(now)
                    .setId(UUID.randomUUID().toString())
                    .setProviderRequestId(null)
                    .setOperationType(FolderOperationType.FOLDER_CREATE)
                    .setAuthorUserId(validatedRequest.getAuthor().getId())
                    .setAuthorUserUid(validatedRequest.getAuthor().getPassportUid().orElse(null))
                    .setAuthorProviderId(null)
                    .setSourceFolderOperationsLogId(null)
                    .setDestinationFolderOperationsLogId(null)
                    .setOldFolderFields(null)
                    .setOldQuotas(new QuotasByResource(Map.of()))
                    .setOldBalance(new QuotasByResource(Map.of()))
                    .setOldProvisions(new QuotasByAccount(Map.of()))
                    .setOldAccounts(null)
                    .setNewFolderFields(FolderHistoryFieldsModel.builder()
                            .serviceId(newDefaultFolder.getServiceId())
                            .version(newDefaultFolder.getVersion())
                            .displayName(newDefaultFolder.getDisplayName())
                            .description(newDefaultFolder.getDescription().orElse(null))
                            .deleted(newDefaultFolder.isDeleted())
                            .folderType(newDefaultFolder.getFolderType())
                            .tags(newDefaultFolder.getTags())
                            .build())
                    .setNewQuotas(new QuotasByResource(Map.of()))
                    .setNewBalance(new QuotasByResource(Map.of()))
                    .setNewProvisions(new QuotasByAccount(Map.of()))
                    .setActuallyAppliedProvisions(null)
                    .setNewAccounts(null)
                    .setAccountsQuotasOperationsId(null)
                    .setQuotasDemandsId(null)
                    .setOperationPhase(null)
                    .setOrder(0L)
                    .setCommentId(null)
                    .build();
            result.add(opLog);
        });
        return result;
    }

    private Map<DeliverableRequestGroupKey, FolderOperationLogModel> prepareOpLogs(
            DeliveryDictionary dictionary, ValidatedDeliveryRequest validatedRequest, ValidatedQuotas quotas,
            Instant now) {
        Map<DeliverableRequestGroupKey, FolderOperationLogModel> result = new HashMap<>();
        Map<Long, FolderModel> newDefaultFoldersByServiceId = validatedRequest.getNewDefaultFolders().stream()
                .collect(Collectors.toMap(FolderModel::getServiceId, Function.identity()));
        Map<DeliverableRequestGroupKey, List<DeliverableRequestModel>> deliverablesByKey = validatedRequest
                .getDeliveryRequest().getDeliverables().stream()
                .collect(Collectors.groupingBy(DeliverableRequestGroupKey::new, Collectors.toList()));
        List<DeliverableRequestGroupKey> keys = deliverablesByKey.keySet().stream()
                .sorted(DeliverableRequestGroupKey.COMPARATOR).collect(Collectors.toList());
        Map<String, FolderModel> knownFolders = new HashMap<>();
        dictionary.getFolders().forEach(knownFolders::put);
        dictionary.getDefaultFolders().forEach((k, v) -> knownFolders.put(v.getId(), v));
        validatedRequest.getNewDefaultFolders().forEach(v -> knownFolders.put(v.getId(), v));
        Map<String, Long> nextOpLogOrders = new HashMap<>();
        Map<QuotaModel.Key, Long> currentQuotaAmounts = new HashMap<>();
        Map<QuotaModel.Key, Long> currentBalances = new HashMap<>();
        Map<QuotaModel.Key, Long> updatedQuotaAmounts = new HashMap<>();
        Map<QuotaModel.Key, Long> updatedBalances = new HashMap<>();
        keys.forEach(key -> {
            List<DeliverableRequestModel> deliverables = deliverablesByKey.get(key);
            Optional<String> targetFolderId = getTargetFolderId(key, dictionary, newDefaultFoldersByServiceId);
            if (targetFolderId.isEmpty()) {
                return;
            }
            deliverables.forEach(deliverable -> {
                ResourceModel resource = dictionary.getResources().get(deliverable.getResourceId());
                ProviderModel provider = dictionary.getProviders().get(resource.getProviderId());
                Optional<QuotaModel.Key> quotaKeyO = getDeliverableQuotaKey(deliverable, dictionary,
                        newDefaultFoldersByServiceId, resource, provider);
                if (quotaKeyO.isEmpty()) {
                    return;
                }
                QuotaModel.Key quotaKey = quotaKeyO.get();
                long currentQuotaAmount = updatedQuotaAmounts.computeIfAbsent(quotaKey, k -> {
                    if (quotas.getOriginalQuotas().containsKey(k)) {
                        QuotaModel quota = quotas.getOriginalQuotas().get(k);
                        return quota.getQuota() != null ? quota.getQuota() : 0L;
                    }
                    return 0L;
                });
                long currentBalance = updatedBalances.computeIfAbsent(quotaKey, k -> {
                    if (quotas.getOriginalQuotas().containsKey(k)) {
                        QuotaModel quota = quotas.getOriginalQuotas().get(k);
                        return quota.getBalance() != null ? quota.getBalance() : 0L;
                    }
                    return 0L;
                });
                UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
                UnitModel unit = unitsEnsemble.unitByKey(deliverable.getDelta().getUnitKey()).get();
                long deltaAmount = Units.convertFromApi(deliverable.getDelta().getAmount(), resource,
                        unitsEnsemble, unit).get();
                long updatedQuotaAmount = Units.add(currentQuotaAmount, deltaAmount).get();
                long updatedBalance = Units.add(currentBalance, deltaAmount).get();
                updatedQuotaAmounts.put(quotaKey, updatedQuotaAmount);
                updatedBalances.put(quotaKey, updatedBalance);
            });
            long nextOpLogOrder = nextOpLogOrders.computeIfAbsent(targetFolderId.get(),
                    id -> knownFolders.get(id).getNextOpLogOrder());
            nextOpLogOrders.put(targetFolderId.get(), nextOpLogOrder + 1L);
            Set<QuotaModel.Key> updatedQuotaKeys = Sets.union(updatedQuotaAmounts.keySet(), updatedBalances.keySet());
            updatedQuotaKeys.forEach(updatedQuotaKey -> {
                if (!currentQuotaAmounts.containsKey(updatedQuotaKey)) {
                    if (quotas.getOriginalQuotas().containsKey(updatedQuotaKey)) {
                        QuotaModel quota = quotas.getOriginalQuotas().get(updatedQuotaKey);
                        currentQuotaAmounts.put(updatedQuotaKey, quota.getQuota() != null ? quota.getQuota() : 0L);
                    } else {
                        currentQuotaAmounts.put(updatedQuotaKey, 0L);
                    }
                }
                if (!currentBalances.containsKey(updatedQuotaKey)) {
                    if (quotas.getOriginalQuotas().containsKey(updatedQuotaKey)) {
                        QuotaModel quota = quotas.getOriginalQuotas().get(updatedQuotaKey);
                        currentBalances.put(updatedQuotaKey, quota.getBalance() != null ? quota.getBalance() : 0L);
                    } else {
                        currentBalances.put(updatedQuotaKey, 0L);
                    }
                }
            });
            Map<String, Long> oldQuotas = new HashMap<>();
            Map<String, Long> newQuotas = new HashMap<>();
            Map<String, Long> oldBalances = new HashMap<>();
            Map<String, Long> newBalances = new HashMap<>();
            updatedQuotaKeys.forEach(quotaKey -> {
                long newQuota = updatedQuotaAmounts.get(quotaKey);
                long newBalance = updatedBalances.get(quotaKey);
                long oldQuota = currentQuotaAmounts.get(quotaKey);
                long oldBalance = currentBalances.get(quotaKey);
                if (newQuota != oldQuota) {
                    oldQuotas.put(quotaKey.getResourceId(), oldQuota);
                    newQuotas.put(quotaKey.getResourceId(), newQuota);
                }
                if (newBalance != oldBalance) {
                    oldBalances.put(quotaKey.getResourceId(), oldBalance);
                    newBalances.put(quotaKey.getResourceId(), newBalance);
                }
            });
            FolderOperationLogModel opLog = FolderOperationLogModel.builder()
                    .setTenantId(Tenants.DEFAULT_TENANT_ID)
                    .setFolderId(targetFolderId.get())
                    .setOperationDateTime(now)
                    .setId(UUID.randomUUID().toString())
                    .setProviderRequestId(null)
                    .setOperationType(FolderOperationType.QUOTA_DELIVERY)
                    .setAuthorUserId(validatedRequest.getAuthor().getId())
                    .setAuthorUserUid(validatedRequest.getAuthor().getPassportUid().orElse(null))
                    .setAuthorProviderId(null)
                    .setSourceFolderOperationsLogId(null)
                    .setDestinationFolderOperationsLogId(null)
                    .setOldFolderFields(null)
                    .setOldQuotas(new QuotasByResource(oldQuotas))
                    .setOldBalance(new QuotasByResource(oldBalances))
                    .setOldProvisions(new QuotasByAccount(Map.of()))
                    .setOldAccounts(null)
                    .setNewFolderFields(null)
                    .setNewQuotas(new QuotasByResource(newQuotas))
                    .setNewBalance(new QuotasByResource(newBalances))
                    .setNewProvisions(new QuotasByAccount(Map.of()))
                    .setActuallyAppliedProvisions(null)
                    .setNewAccounts(null)
                    .setAccountsQuotasOperationsId(null)
                    .setQuotasDemandsId(null)
                    .setOperationPhase(null)
                    .setOrder(nextOpLogOrder)
                    .setCommentId(null)
                    .setDeliveryMeta(DeliverableMetaHistoryModel.builder()
                            .deliveryId(validatedRequest.getDeliveryRequest().getDeliveryId())
                            .quotaRequestId(key.getMeta().getQuotaRequestId())
                            .campaignId(key.getMeta().getCampaignId())
                            .bigOrderIds(key.getMeta().getBigOrderId())
                            .build())
                    .build();
            result.put(key, opLog);
            updatedQuotaAmounts.forEach(currentQuotaAmounts::put);
            updatedBalances.forEach(currentBalances::put);
        });
        return result;
    }

    private Optional<QuotaModel.Key> getDeliverableQuotaKey(DeliverableRequestModel deliverable,
                                                            DeliveryDictionary dictionary,
                                                            Map<Long, FolderModel> newDefaultFoldersByServiceId,
                                                            ResourceModel resource,
                                                            ProviderModel provider) {
        if (deliverable.getServiceId().isPresent()) {
            if (dictionary.getDefaultFolders().containsKey(deliverable.getServiceId().get())) {
                FolderModel defaultFolder = dictionary.getDefaultFolders().get(deliverable.getServiceId().get());
                return Optional.of(new QuotaModel.Key(defaultFolder.getId(), provider.getId(), resource.getId()));
            } else {
                FolderModel newDefaultFolder = newDefaultFoldersByServiceId.get(deliverable.getServiceId().get());
                return Optional.of(new QuotaModel.Key(newDefaultFolder.getId(), provider.getId(), resource.getId()));
            }
        } else if (deliverable.getFolderId().isPresent()) {
            return Optional.of(new QuotaModel.Key(deliverable.getFolderId().get(), provider.getId(), resource.getId()));
        }
        return Optional.empty();
    }

    private Optional<String> getTargetFolderId(DeliverableRequestGroupKey key, DeliveryDictionary dictionary,
                                               Map<Long, FolderModel> newDefaultFoldersByServiceId) {
        if (key.getServiceId().isPresent()) {
            if (dictionary.getDefaultFolders().containsKey(key.getServiceId().get())) {
                FolderModel defaultFolder = dictionary.getDefaultFolders().get(key.getServiceId().get());
                return Optional.of(defaultFolder.getId());
            } else {
                FolderModel newDefaultFolder = newDefaultFoldersByServiceId.get(key.getServiceId().get());
                return Optional.of(newDefaultFolder.getId());
            }
        } else if (key.getFolderId().isPresent()) {
            return Optional.of(key.getFolderId().get());
        }
        return Optional.empty();
    }

    private FinishedDeliveryModel prepareFinishedDelivery(
            ValidatedDeliveryRequest validatedRequest,
            Map<DeliverableRequestGroupKey, FolderOperationLogModel> opLogsByKey) {
        DeliveryResponseModel.Builder deliveryResponseBuilder = DeliveryResponseModel.builder();
        Map<DeliverableRequestGroupKey, List<DeliverableRequestModel>> deliverablesByKey = validatedRequest
                .getDeliveryRequest().getDeliverables().stream()
                .collect(Collectors.groupingBy(DeliverableRequestGroupKey::new, Collectors.toList()));
        deliverablesByKey.forEach((key, deliverables) -> {
            FolderOperationLogModel opLog = opLogsByKey.get(key);
            Set<String> resourceIds = deliverables.stream()
                    .map(DeliverableRequestModel::getResourceId).collect(Collectors.toSet());
            DeliverableResponseModel.Builder deliverableResponseBuilder = DeliverableResponseModel.builder();
            deliverableResponseBuilder.addResourceIds(resourceIds);
            key.getServiceId().ifPresent(deliverableResponseBuilder::serviceId);
            key.getFolderId().ifPresent(deliverableResponseBuilder::folderId);
            deliverableResponseBuilder.meta(DeliverableMetaResponseModel.builder()
                    .quotaRequestId(key.getMeta().getQuotaRequestId())
                    .campaignId(key.getMeta().getCampaignId())
                    .bigOrderId(key.getMeta().getBigOrderId())
                    .build());
            deliverableResponseBuilder.folderOperationLog(DeliverableFolderOperationModel.builder()
                    .id(opLog.getId())
                    .folderId(opLog.getFolderId())
                    .operationDateTime(opLog.getOperationDateTime())
                    .build());
            deliveryResponseBuilder.addDeliverable(deliverableResponseBuilder.build());
        });
        return FinishedDeliveryModel.builder()
                .id(validatedRequest.getDeliveryRequest().getDeliveryId())
                .tenantId(Tenants.DEFAULT_TENANT_ID)
                .request(validatedRequest.getDeliveryRequest())
                .response(deliveryResponseBuilder.build())
                .build();
    }

    private <T> Mono<T> inTx(Function<YdbTxSession, Mono<T>> inTx) {
        return tableClient.usingSessionMonoRetryable(session ->
                session.usingTxMonoRetryable(TransactionMode.SERIALIZABLE_READ_WRITE, inTx));
    }

    private DeliveryResponseDto toResponse(FinishedDeliveryModel delivery) {
        DeliveryResponseDto.Builder builder = DeliveryResponseDto.builder();
        delivery.getResponse().getDeliverables().forEach(deliverable -> {
            deliverable.getResourceIds().forEach(resourceId -> {
                DeliverableResponseDto.Builder deliverableBuilder = DeliverableResponseDto.builder();
                deliverable.getServiceId().ifPresent(deliverableBuilder::serviceId);
                deliverable.getFolderId().ifPresent(deliverableBuilder::folderId);
                deliverableBuilder.resourceId(resourceId);
                deliverableBuilder.folderOperationLog(DeliverableFolderOperationDto.builder()
                        .id(deliverable.getFolderOperationLog().getId())
                        .folderId(deliverable.getFolderOperationLog().getFolderId())
                        .timestamp(deliverable.getFolderOperationLog().getOperationDateTime())
                        .build());
                deliverableBuilder.meta(DeliverableMetaResponseDto.builder()
                        .setQuotaRequestId(deliverable.getMeta().getQuotaRequestId())
                        .setCampaignId(deliverable.getMeta().getCampaignId())
                        .setBigOrderId(deliverable.getMeta().getBigOrderId())
                        .build());
                builder.addDeliverable(deliverableBuilder.build());
            });
        });
        return builder.build();
    }
}
