package ru.yandex.intranet.d.services.delivery;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.dao.delivery.FinishedDeliveriesDao;
import ru.yandex.intranet.d.dao.folders.FolderDao;
import ru.yandex.intranet.d.dao.folders.FolderOperationLogDao;
import ru.yandex.intranet.d.dao.providers.ProvidersDao;
import ru.yandex.intranet.d.dao.quotas.QuotasDao;
import ru.yandex.intranet.d.dao.resources.ResourcesDao;
import ru.yandex.intranet.d.dao.services.ServicesDao;
import ru.yandex.intranet.d.dao.units.UnitsEnsemblesDao;
import ru.yandex.intranet.d.dao.users.UsersDao;
import ru.yandex.intranet.d.datasource.model.YdbTxSession;
import ru.yandex.intranet.d.model.TenantId;
import ru.yandex.intranet.d.model.WithTenant;
import ru.yandex.intranet.d.model.delivery.FinishedDeliveryModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderOperationLogModel;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.delivery.model.DeliveryDictionary;
import ru.yandex.intranet.d.services.delivery.model.DeliveryQuotas;
import ru.yandex.intranet.d.services.delivery.model.DeliveryRequest;
import ru.yandex.intranet.d.services.delivery.model.ValidatedDeliveryRequest;

/**
 * Delivery store service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class DeliveryStoreService {

    private final FolderDao folderDao;
    private final ResourcesDao resourcesDao;
    private final ServicesDao servicesDao;
    private final UnitsEnsemblesDao unitsEnsemblesDao;
    private final UsersDao usersDao;
    private final ProvidersDao providersDao;
    private final QuotasDao quotasDao;
    private final FolderOperationLogDao folderOperationLogDao;
    private final FinishedDeliveriesDao finishedDeliveriesDao;

    @SuppressWarnings("ParameterNumber")
    public DeliveryStoreService(FolderDao folderDao, ResourcesDao resourcesDao, ServicesDao servicesDao,
                                UnitsEnsemblesDao unitsEnsemblesDao, UsersDao usersDao, ProvidersDao providersDao,
                                QuotasDao quotasDao, FolderOperationLogDao folderOperationLogDao,
                                FinishedDeliveriesDao finishedDeliveriesDao) {
        this.folderDao = folderDao;
        this.resourcesDao = resourcesDao;
        this.servicesDao = servicesDao;
        this.unitsEnsemblesDao = unitsEnsemblesDao;
        this.usersDao = usersDao;
        this.providersDao = providersDao;
        this.quotasDao = quotasDao;
        this.folderOperationLogDao = folderOperationLogDao;
        this.finishedDeliveriesDao = finishedDeliveriesDao;
    }

    public Mono<Optional<FinishedDeliveryModel>> getDeliveryById(YdbTxSession session, String deliveryId) {
        return finishedDeliveriesDao.getById(session, deliveryId, Tenants.DEFAULT_TENANT_ID);
    }

    public Mono<Void> upsertDelivery(YdbTxSession session, FinishedDeliveryModel delivery) {
        return finishedDeliveriesDao.upsertOneRetryable(session, delivery).then();
    }

    public Mono<Void> upsertFolders(YdbTxSession session, List<FolderModel> folders) {
        if (folders.isEmpty()) {
            return Mono.empty();
        }
        return folderDao.upsertAllRetryable(session, folders);
    }

    public Mono<Void> upsertQuotas(YdbTxSession session, List<QuotaModel> quotas) {
        if (quotas.isEmpty()) {
            return Mono.empty();
        }
        return quotasDao.upsertAllRetryable(session, quotas);
    }

    public Mono<Void> upsertOperationLog(YdbTxSession session, List<FolderOperationLogModel> logs) {
        if (logs.isEmpty()) {
            return Mono.empty();
        }
        return folderOperationLogDao.upsertAllRetryable(session, logs);
    }

    public Mono<DeliveryDictionary> loadDictionaries(YdbTxSession session, DeliveryRequest request) {
        String authorUid = request.getAuthorUid();
        Set<Long> serviceIds = new HashSet<>();
        Set<String> folderIds = new HashSet<>();
        Set<String> resourceIds = new HashSet<>();
        collectIds(request, serviceIds, folderIds, resourceIds);
        return loadFolders(session, folderIds)
                .flatMap(folders -> loadResources(session, resourceIds)
                .flatMap(resources -> loadServices(session, serviceIds)
                .flatMap(services -> loadUnitsEnsembles(session, resources)
                .flatMap(unitsEnsembles -> loadUser(session, authorUid)
                .flatMap(users -> loadDefaultFolders(session, serviceIds)
                .flatMap(defaultFolders -> loadProviders(session, resources)
                .map(providers -> buildDictionaries(folders, resources, services, unitsEnsembles, users,
                        defaultFolders, providers))))))));
    }

    public Mono<DeliveryQuotas> loadQuotas(YdbTxSession session, ValidatedDeliveryRequest request,
                                           DeliveryDictionary dictionary) {
        Set<QuotaModel.Key> keySet = new HashSet<>();
        request.getDeliveryRequest().getDeliverables().forEach(deliverable -> {
            ResourceModel resource = dictionary.getResources().get(deliverable.getResourceId());
            ProviderModel provider = dictionary.getProviders().get(resource.getProviderId());
            if (deliverable.getServiceId().isPresent()) {
                if (dictionary.getDefaultFolders().containsKey(deliverable.getServiceId().get())) {
                    FolderModel defaultFolder = dictionary.getDefaultFolders().get(deliverable.getServiceId().get());
                    keySet.add(new QuotaModel.Key(defaultFolder.getId(), provider.getId(), resource.getId()));
                }
            } else if (deliverable.getFolderId().isPresent()) {
                keySet.add(new QuotaModel.Key(deliverable.getFolderId().get(), provider.getId(), resource.getId()));
            }
        });
        List<WithTenant<QuotaModel.Key>> keys = keySet.stream()
                .map(k -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, k)).collect(Collectors.toList());
        if (keys.isEmpty()) {
            return Mono.just(new DeliveryQuotas(Map.of()));
        }
        return Flux.fromIterable(Lists.partition(keys, 500))
                .concatMap(v -> quotasDao.getByKeys(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()))
                .map(l -> new DeliveryQuotas(l.stream()
                        .collect(Collectors.toMap(QuotaModel::toKey, Function.identity()))));
    }

    private DeliveryDictionary buildDictionaries(List<FolderModel> folders,
                                                 List<ResourceModel> resources,
                                                 List<ServiceWithStatesModel> services,
                                                 List<UnitsEnsembleModel> unitsEnsembles,
                                                 List<UserModel> users,
                                                 List<FolderModel> defaultFolders,
                                                 List<ProviderModel> providers) {
        Map<String, ResourceModel> resourcesById = resources.stream()
                .collect(Collectors.toMap(ResourceModel::getId, Function.identity()));
        Map<String, UnitsEnsembleModel> unitsEnsemblesById = unitsEnsembles.stream()
                .collect(Collectors.toMap(UnitsEnsembleModel::getId, Function.identity()));
        Map<String, FolderModel> foldersById = folders.stream()
                .collect(Collectors.toMap(FolderModel::getId, Function.identity()));
        Map<Long, ServiceWithStatesModel> servicesById = services.stream()
                .collect(Collectors.toMap(ServiceWithStatesModel::getId, Function.identity()));
        Map<Long, FolderModel> defaultFoldersByServiceId = defaultFolders.stream()
                .collect(Collectors.toMap(FolderModel::getServiceId, Function.identity()));
        Map<String, UserModel> usersByUid = users.stream().filter(u -> u.getPassportUid().isPresent())
                .collect(Collectors.toMap(r -> r.getPassportUid().get(), Function.identity()));
        Map<String, ProviderModel> providersById = providers.stream()
                .collect(Collectors.toMap(ProviderModel::getId, Function.identity()));
        return new DeliveryDictionary(resourcesById, unitsEnsemblesById, foldersById, servicesById,
                defaultFoldersByServiceId, usersByUid, providersById);
    }

    private void collectIds(DeliveryRequest request, Set<Long> serviceIds, Set<String> folderIds,
                            Set<String> resourceIds) {
        request.getDeliverables().forEach(deliverable -> {
            deliverable.getServiceId().ifPresent(serviceIds::add);
            deliverable.getFolderId().ifPresent(folderIds::add);
            resourceIds.add(deliverable.getResourceId());
        });
    }

    public Mono<List<FolderModel>> loadFolders(YdbTxSession session, Set<String> folderIds) {
        if (folderIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(folderIds), 500))
                .concatMap(v -> folderDao.getByIds(session, v, Tenants.DEFAULT_TENANT_ID))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ResourceModel>> loadResources(YdbTxSession session, Set<String> resourceIds) {
        if (resourceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> ids = resourceIds.stream().map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID))
                .collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> resourcesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ServiceWithStatesModel>> loadServices(YdbTxSession session, Set<Long> serviceIds) {
        if (serviceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        return Flux.fromIterable(Lists.partition(new ArrayList<>(serviceIds), 500))
                .concatMap(v -> servicesDao.getServiceStatesByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<UnitsEnsembleModel>> loadUnitsEnsembles(YdbTxSession session, List<ResourceModel> resources) {
        if (resources.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> ids = resources.stream().map(ResourceModel::getUnitsEnsembleId)
                .distinct().map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> unitsEnsemblesDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<UserModel>> loadUser(YdbTxSession session, String uid) {
        return usersDao.getByPassportUid(session, uid, Tenants.DEFAULT_TENANT_ID)
                .map(u -> u.stream().collect(Collectors.toList()));
    }

    public Mono<List<FolderModel>> loadDefaultFolders(YdbTxSession session, Set<Long> serviceIds) {
        if (serviceIds.isEmpty()) {
            return Mono.just(List.of());
        }
        List<WithTenant<Long>> ids = serviceIds.stream()
                .map(id -> new WithTenant<>(Tenants.DEFAULT_TENANT_ID, id)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> folderDao.getAllDefaultFoldersByServiceIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

    public Mono<List<ProviderModel>> loadProviders(YdbTxSession session, List<ResourceModel> resources) {
        if (resources.isEmpty()) {
            return Mono.just(List.of());
        }
        List<Tuple2<String, TenantId>> ids = resources.stream().map(ResourceModel::getProviderId)
                .distinct().map(id -> Tuples.of(id, Tenants.DEFAULT_TENANT_ID)).collect(Collectors.toList());
        return Flux.fromIterable(Lists.partition(ids, 500))
                .concatMap(v -> providersDao.getByIds(session, v))
                .collectList().map(l -> l.stream().flatMap(Collection::stream).collect(Collectors.toList()));
    }

}
