package ru.yandex.intranet.d.services.delivery;

import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import ru.yandex.intranet.d.dao.Tenants;
import ru.yandex.intranet.d.model.delivery.DeliverableDeltaModel;
import ru.yandex.intranet.d.model.delivery.DeliverableMetaRequestModel;
import ru.yandex.intranet.d.model.delivery.DeliverableRequestModel;
import ru.yandex.intranet.d.model.delivery.DeliveryRequestModel;
import ru.yandex.intranet.d.model.delivery.FinishedDeliveryModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.folders.FolderType;
import ru.yandex.intranet.d.model.providers.ProviderModel;
import ru.yandex.intranet.d.model.quotas.QuotaModel;
import ru.yandex.intranet.d.model.resources.ResourceModel;
import ru.yandex.intranet.d.model.services.ServiceWithStatesModel;
import ru.yandex.intranet.d.model.units.UnitModel;
import ru.yandex.intranet.d.model.units.UnitsEnsembleModel;
import ru.yandex.intranet.d.model.users.UserModel;
import ru.yandex.intranet.d.services.delivery.model.DeliverableDelta;
import ru.yandex.intranet.d.services.delivery.model.DeliverableMetaRequest;
import ru.yandex.intranet.d.services.delivery.model.DeliverableRequest;
import ru.yandex.intranet.d.services.delivery.model.DeliveryDictionary;
import ru.yandex.intranet.d.services.delivery.model.DeliveryQuotas;
import ru.yandex.intranet.d.services.delivery.model.DeliveryRequest;
import ru.yandex.intranet.d.services.delivery.model.ValidatedDeliveryRequest;
import ru.yandex.intranet.d.services.delivery.model.ValidatedQuotas;
import ru.yandex.intranet.d.services.validators.AbcServiceValidator;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.util.units.Units;
import ru.yandex.intranet.d.web.model.delivery.DeliverableDeltaDto;
import ru.yandex.intranet.d.web.model.delivery.DeliverableMetaRequestDto;
import ru.yandex.intranet.d.web.model.delivery.DeliverableRequestDto;
import ru.yandex.intranet.d.web.model.delivery.DeliveryRequestDto;

/**
 * Delivery validation service implementation.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
@Component
public class DeliveryValidationService {

    private static final int MAX_DELIVERY_ID_LENGTH = 256;
    private static final int MAX_TEXT_LENGTH = 1024;
    private static final int MAX_DELIVERABLES = 10000;

    private final MessageSource messages;

    public DeliveryValidationService(@Qualifier("messageSource") MessageSource messages) {
        this.messages = messages;
    }

    public Result<DeliveryRequest> preValidateRequest(DeliveryRequestDto request, Locale locale) {
        DeliveryRequest.Builder builder = DeliveryRequest.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateText(request::getDeliveryId, builder::deliveryId, errors,
                "deliveryId", MAX_DELIVERY_ID_LENGTH, locale);
        validateText(request::getAuthorUid, builder::authorUid, errors, "authorUid", MAX_TEXT_LENGTH, locale);
        if (request.getDeliverables().isEmpty() || request.getDeliverables().get().isEmpty()) {
            errors.addError("deliverables", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            for (int i = 0; i < request.getDeliverables().get().size(); i++) {
                DeliverableRequestDto deliverable = request.getDeliverables().get().get(i);
                preValidateDeliverable(builder, errors, deliverable, i, locale);
            }
            if (request.getDeliverables().get().size() > MAX_DELIVERABLES) {
                errors.addError("deliverables", TypedError.invalid(messages
                        .getMessage("errors.too.many.list.elements", null, locale)));
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(builder.build());
    }

    public Result<Void> validateRequestMatch(DeliveryRequest request, FinishedDeliveryModel delivery, Locale locale) {
        DeliveryRequestModel existingRequest = delivery.getRequest();
        DeliveryRequestModel newRequest = fromRequest(request);
        if (!existingRequest.equals(newRequest)) {
            return Result.failure(ErrorCollection.builder().addError(TypedError.invalid(messages
                    .getMessage("errors.delivery.request.mismatch", null, locale))).build());
        }
        return Result.success(null);
    }

    public Result<ValidatedDeliveryRequest> validateRequest(DeliveryRequest request, DeliveryDictionary dictionary,
                                                            Locale locale) {
        ValidatedDeliveryRequest.Builder builder = ValidatedDeliveryRequest.builder();
        DeliveryRequestModel.Builder requestBuilder = DeliveryRequestModel.builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        requestBuilder.deliveryId(request.getDeliveryId());
        if (!dictionary.getUsersByUid().containsKey(request.getAuthorUid())) {
            errors.addError("authorUid", TypedError.invalid(messages
                    .getMessage("errors.user.not.found", null, locale)));
        } else {
            UserModel author = dictionary.getUsersByUid().get(request.getAuthorUid());
            if (author.isDeleted() || author.getStaffDismissed().orElse(false)) {
                errors.addError("authorUid", TypedError.invalid(messages
                        .getMessage("errors.user.not.found", null, locale)));
            } else {
                requestBuilder.authorUid(request.getAuthorUid());
                builder.author(author);
            }
        }
        for (int i = 0; i < request.getDeliverables().size(); i++) {
            DeliverableRequest deliverable = request.getDeliverables().get(i);
            validateDeliverable(deliverable, dictionary, requestBuilder, builder, errors, i, locale);
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        builder.deliveryRequest(requestBuilder.build());
        return Result.success(builder.build());
    }

    public Result<ValidatedQuotas> validateQuotas(DeliveryRequest request,
                                                  DeliveryQuotas originalQuotas,
                                                  DeliveryDictionary dictionary,
                                                  ValidatedDeliveryRequest validatedDeliveryRequest,
                                                  Locale locale) {
        Map<QuotaModel.Key, QuotaModel> updatedQuotas = new HashMap<>();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        Map<Long, FolderModel> newDefaultFoldersByServiceId = validatedDeliveryRequest.getNewDefaultFolders().stream()
                .collect(Collectors.toMap(FolderModel::getServiceId, Function.identity()));
        Map<QuotaModel.Key, Long> updatedQuotaAmounts = new HashMap<>();
        Map<QuotaModel.Key, Long> updatedBalances = new HashMap<>();
        for (int i = 0; i < request.getDeliverables().size(); i++) {
            DeliverableRequest deliverable = request.getDeliverables().get(i);
            ResourceModel resource = dictionary.getResources().get(deliverable.getResourceId());
            ProviderModel provider = dictionary.getProviders().get(resource.getProviderId());
            Optional<QuotaModel.Key> quotaKeyO = getDeliverableQuotaKey(deliverable, dictionary,
                    newDefaultFoldersByServiceId, resource, provider);
            if (quotaKeyO.isEmpty()) {
                continue;
            }
            QuotaModel.Key quotaKey = quotaKeyO.get();
            long currentQuotaAmount = updatedQuotaAmounts.computeIfAbsent(quotaKey, k -> {
                if (originalQuotas.getQuotas().containsKey(k)) {
                    QuotaModel quota = originalQuotas.getQuotas().get(k);
                    return quota.getQuota() != null ? quota.getQuota() : 0L;
                }
                return 0L;
            });
            long currentBalance = updatedBalances.computeIfAbsent(quotaKey, k -> {
                if (originalQuotas.getQuotas().containsKey(k)) {
                    QuotaModel quota = originalQuotas.getQuotas().get(k);
                    return quota.getBalance() != null ? quota.getBalance() : 0L;
                }
                return 0L;
            });
            UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
            UnitModel unit = unitsEnsemble.unitByKey(deliverable.getDelta().getUnitKey()).get();
            Optional<Long> deltaAmount = Units.convertFromApi(deliverable.getDelta().getAmount(), resource,
                    unitsEnsemble, unit);
            if (deltaAmount.isEmpty()) {
                errors.addError("deliverables." + i + ".delta.amount", TypedError.invalid(messages
                        .getMessage("errors.value.can.not.be.converted.to.base.unit", null, locale)));
            } else {
                Optional<Long> updatedQuotaAmount = Units.add(currentQuotaAmount, deltaAmount.get());
                Optional<Long> updatedBalance = Units.add(currentBalance, deltaAmount.get());
                if (updatedQuotaAmount.isEmpty() || updatedBalance.isEmpty()) {
                    errors.addError("deliverables." + i + ".delta.amount", TypedError.invalid(messages
                            .getMessage("errors.number.out.of.range", null, locale)));
                } else {
                    updatedQuotaAmounts.put(quotaKey, updatedQuotaAmount.get());
                    updatedBalances.put(quotaKey, updatedBalance.get());
                }
            }
        }
        Set<QuotaModel.Key> keys = Sets.union(updatedQuotaAmounts.keySet(), updatedBalances.keySet());
        keys.forEach(key -> {
            long updatedQuotaAmount = updatedQuotaAmounts.getOrDefault(key, 0L);
            long updatedBalance = updatedBalances.getOrDefault(key, 0L);
            if (originalQuotas.getQuotas().containsKey(key)) {
                QuotaModel originalQuota = originalQuotas.getQuotas().get(key);
                QuotaModel updatedQuota = QuotaModel.builder(originalQuota)
                        .quota(updatedQuotaAmount)
                        .balance(updatedBalance)
                        .build();
                updatedQuotas.put(key, updatedQuota);
            } else {
                QuotaModel updatedQuota = QuotaModel.builder()
                        .tenantId(Tenants.DEFAULT_TENANT_ID)
                        .folderId(key.getFolderId())
                        .providerId(key.getProviderId())
                        .resourceId(key.getResourceId())
                        .quota(updatedQuotaAmount)
                        .balance(updatedBalance)
                        .frozenQuota(0L)
                        .build();
                updatedQuotas.put(key, updatedQuota);
            }
        });
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(new ValidatedQuotas(originalQuotas.getQuotas(), updatedQuotas));
    }

    private Optional<QuotaModel.Key> getDeliverableQuotaKey(DeliverableRequest deliverable,
                                                            DeliveryDictionary dictionary,
                                                            Map<Long, FolderModel> newDefaultFoldersByServiceId,
                                                            ResourceModel resource,
                                                            ProviderModel provider) {
        if (deliverable.getServiceId().isPresent()) {
            if (dictionary.getDefaultFolders().containsKey(deliverable.getServiceId().get())) {
                FolderModel defaultFolder = dictionary.getDefaultFolders().get(deliverable.getServiceId().get());
                return Optional.of(new QuotaModel.Key(defaultFolder.getId(), provider.getId(), resource.getId()));
            } else {
                FolderModel newDefaultFolder = newDefaultFoldersByServiceId.get(deliverable.getServiceId().get());
                return Optional.of(new QuotaModel.Key(newDefaultFolder.getId(), provider.getId(), resource.getId()));
            }
        } else if (deliverable.getFolderId().isPresent()) {
            return Optional.of(new QuotaModel.Key(deliverable.getFolderId().get(), provider.getId(), resource.getId()));
        }
        return Optional.empty();
    }

    private void validateDeliverable(DeliverableRequest deliverable, DeliveryDictionary dictionary,
                                     DeliveryRequestModel.Builder requestBuilder,
                                     ValidatedDeliveryRequest.Builder builder, ErrorCollection.Builder errors,
                                     int i, Locale locale) {
        DeliverableRequestModel.Builder deliverableBuilder = DeliverableRequestModel.builder();
        ErrorCollection.Builder deliverableErrors = ErrorCollection.builder();
        if (!dictionary.getResources().containsKey(deliverable.getResourceId())) {
            deliverableErrors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                    .getMessage("errors.resource.not.found", null, locale)));
        } else {
            ResourceModel resource = dictionary.getResources().get(deliverable.getResourceId());
            ProviderModel provider = dictionary.getProviders().get(resource.getProviderId());
            if (resource.isDeleted() || provider.isDeleted()) {
                deliverableErrors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.resource.not.found", null, locale)));
            } else if (!resource.isManaged()) {
                deliverableErrors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.resource.not.managed", null, locale)));
            } else if (!provider.isManaged()) {
                deliverableErrors.addError("deliverables." + i + ".resourceId", TypedError.invalid(messages
                        .getMessage("errors.provider.is.not.managed", null, locale)));
            } else {
                deliverableBuilder.resourceId(deliverable.getResourceId());
            }
            UnitsEnsembleModel unitsEnsemble = dictionary.getUnitsEnsembles().get(resource.getUnitsEnsembleId());
            Optional<UnitModel> unitO = unitsEnsemble.unitByKey(deliverable.getDelta().getUnitKey());
            if (unitO.isEmpty()) {
                deliverableErrors.addError("deliverables." + i + ".delta.unitKey", TypedError.invalid(messages
                        .getMessage("errors.unit.not.found", null, locale)));
            } else {
                deliverableBuilder.delta(DeliverableDeltaModel.builder()
                        .amount(deliverable.getDelta().getAmount())
                        .unitKey(deliverable.getDelta().getUnitKey())
                        .build());
            }
        }
        if (deliverable.getServiceId().isPresent()) {
            if (!dictionary.getServices().containsKey(deliverable.getServiceId().get())) {
                deliverableErrors.addError("deliverables." + i + ".serviceId", TypedError.invalid(messages
                        .getMessage("errors.service.not.found", null, locale)));
            } else {
                ServiceWithStatesModel service = dictionary.getServices().get(deliverable.getServiceId().get());
                if (!service.isExportable()) {
                    deliverableErrors.addError("deliverables." + i + ".serviceId", TypedError.invalid(messages
                            .getMessage("errors.service.is.non.exportable", null, locale)));
                } else if (!isValidServiceState(service)) {
                    deliverableErrors.addError("deliverables." + i + ".serviceId", TypedError.invalid(messages
                            .getMessage("errors.service.bad.status", null, locale)));
                } else {
                    deliverableBuilder.serviceId(deliverable.getServiceId().get());
                    if (!dictionary.getDefaultFolders().containsKey(deliverable.getServiceId().get())
                            && !builder.hasNewDefaultFolder(deliverable.getServiceId().get())) {
                        builder.addNewDefaultFolder(FolderModel.newBuilder()
                                .setTenantId(Tenants.DEFAULT_TENANT_ID)
                                .setId(UUID.randomUUID().toString())
                                .setServiceId(deliverable.getServiceId().get())
                                .setFolderType(FolderType.COMMON_DEFAULT_FOR_SERVICE)
                                .setDisplayName("default")
                                .setNextOpLogOrder(1L)
                                .build());
                    }
                }
            }
        } else if (deliverable.getFolderId().isPresent()) {
            if (!dictionary.getFolders().containsKey(deliverable.getFolderId().get())) {
                deliverableErrors.addError("deliverables." + i + ".folderId", TypedError.invalid(messages
                        .getMessage("errors.folder.not.found", null, locale)));
            } else {
                FolderModel folder = dictionary.getFolders().get(deliverable.getFolderId().get());
                if (folder.isDeleted()) {
                    deliverableErrors.addError("deliverables." + i + ".folderId", TypedError.invalid(messages
                            .getMessage("errors.folder.not.found", null, locale)));
                } else {
                    deliverableBuilder.folderId(deliverable.getFolderId().get());
                }
            }
        }
        deliverableBuilder.meta(DeliverableMetaRequestModel.builder()
                .quotaRequestId(deliverable.getMeta().getQuotaRequestId())
                .campaignId(deliverable.getMeta().getCampaignId())
                .bigOrderId(deliverable.getMeta().getBigOrderId())
                .build());
        if (deliverableErrors.hasAnyErrors()) {
            errors.add(deliverableErrors);
        } else {
            requestBuilder.addDeliverable(deliverableBuilder.build());
        }
    }

    private boolean isValidServiceState(ServiceWithStatesModel service) {
        boolean isValidState = AbcServiceValidator.ALLOWED_SERVICE_STATES.contains(service.getState());
        boolean isValidReadOnlyState = service.getReadOnlyState() == null ||
                AbcServiceValidator.ALLOWED_SERVICE_READONLY_STATES.contains(service.getReadOnlyState());
        return isValidState && isValidReadOnlyState;
    }

    private DeliveryRequestModel fromRequest(DeliveryRequest request) {
        DeliveryRequestModel.Builder builder = DeliveryRequestModel.builder();
        builder.deliveryId(request.getDeliveryId());
        builder.authorUid(request.getAuthorUid());
        request.getDeliverables().forEach(deliverable -> {
            DeliverableRequestModel.Builder deliverableBuilder = DeliverableRequestModel.builder();
            deliverable.getServiceId().ifPresent(deliverableBuilder::serviceId);
            deliverable.getFolderId().ifPresent(deliverableBuilder::folderId);
            deliverableBuilder.resourceId(deliverable.getResourceId());
            deliverableBuilder.delta(DeliverableDeltaModel.builder()
                    .amount(deliverable.getDelta().getAmount())
                    .unitKey(deliverable.getDelta().getUnitKey())
                    .build());
            deliverableBuilder.meta(DeliverableMetaRequestModel.builder()
                    .quotaRequestId(deliverable.getMeta().getQuotaRequestId())
                    .campaignId(deliverable.getMeta().getCampaignId())
                    .bigOrderId(deliverable.getMeta().getBigOrderId())
                    .build());
            builder.addDeliverable(deliverableBuilder.build());
        });
        return builder.build();
    }

    private void preValidateDeliverable(DeliveryRequest.Builder builder, ErrorCollection.Builder errors,
                                        DeliverableRequestDto deliverable, int index, Locale locale) {
        if (deliverable == null) {
            errors.addError("deliverables." + index, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        DeliverableRequest.Builder deliverableBuilder = DeliverableRequest.builder();
        ErrorCollection.Builder deliverableErrors = ErrorCollection.builder();
        deliverable.getServiceId().ifPresent(deliverableBuilder::serviceId);
        validateFolderId(deliverable::getFolderId, deliverableBuilder::folderId,
                deliverableErrors, "deliverables." + index + ".folderId", locale);
        validateResourceId(deliverable::getResourceId, deliverableBuilder::resourceId,
                deliverableErrors, "deliverables." + index + ".resourceId", locale);
        preValidateDelta(deliverableBuilder, deliverableErrors,
                deliverable::getDelta, "deliverables." + index + ".delta", locale);
        preValidateMeta(deliverableBuilder, deliverableErrors,
                deliverable::getMeta, "deliverables." + index + ".meta", locale);
        if (deliverable.getServiceId().isEmpty() && deliverable.getFolderId().isEmpty()) {
            deliverableErrors.addError("deliverables." + index, TypedError.invalid(messages
                    .getMessage("errors.folder.id.or.service.id.is.required", null, locale)));
        } else if (deliverable.getServiceId().isPresent() && deliverable.getFolderId().isPresent()) {
            deliverableErrors.addError("deliverables." + index, TypedError.invalid(messages
                    .getMessage("errors.folder.id.and.service.id.can.not.be.both.supplied", null, locale)));
        }
        if (deliverableErrors.hasAnyErrors()) {
            errors.add(deliverableErrors);
            return;
        }
        builder.addDeliverable(deliverableBuilder.build());
    }

    private void preValidateDelta(DeliverableRequest.Builder builder, ErrorCollection.Builder errors,
                                  Supplier<Optional<DeliverableDeltaDto>> deltaSupplier, String fieldPrefix,
                                  Locale locale) {
        Optional<DeliverableDeltaDto> deltaO = deltaSupplier.get();
        if (deltaO.isEmpty()) {
            errors.addError(fieldPrefix, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        DeliverableDeltaDto delta = deltaO.get();
        DeliverableDelta.Builder deltaBuilder = DeliverableDelta.builder();
        ErrorCollection.Builder deltaErrors = ErrorCollection.builder();
        validateAmount(delta::getAmount, deltaBuilder::amount, deltaErrors, fieldPrefix + ".amount", locale);
        validateText(delta::getUnitKey, deltaBuilder::unitKey, deltaErrors,
                fieldPrefix + ".unitKey", MAX_TEXT_LENGTH, locale);
        if (deltaErrors.hasAnyErrors()) {
            errors.add(deltaErrors);
            return;
        }
        builder.delta(deltaBuilder.build());
    }

    private void preValidateMeta(DeliverableRequest.Builder builder, ErrorCollection.Builder errors,
                                 Supplier<Optional<DeliverableMetaRequestDto>> metaSupplier, String fieldPrefix,
                                 Locale locale) {
        Optional<DeliverableMetaRequestDto> metaO = metaSupplier.get();
        if (metaO.isEmpty()) {
            errors.addError(fieldPrefix, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        DeliverableMetaRequestDto meta = metaO.get();
        DeliverableMetaRequest.Builder metaBuilder = DeliverableMetaRequest.builder();
        ErrorCollection.Builder metaErrors = ErrorCollection.builder();
        validateRequired(meta::getQuotaRequestId, metaBuilder::quotaRequestId, metaErrors,
                fieldPrefix + ".quotaRequestId", locale);
        validateRequired(meta::getCampaignId, metaBuilder::campaignId, metaErrors,
                fieldPrefix + ".campaignId", locale);
        validateRequired(meta::getBigOrderId, metaBuilder::bigOrderId, metaErrors,
                fieldPrefix + ".bigOrderId", locale);
        if (metaErrors.hasAnyErrors()) {
            errors.add(metaErrors);
            return;
        }
        builder.meta(metaBuilder.build());
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    private <T> void validateRequired(Supplier<Optional<T>> getter, Consumer<T> setter,
                                      ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<T> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private void validateResourceId(Supplier<Optional<String>> getter, Consumer<String> setter,
                                    ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> id = getter.get();
        if (id.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Uuids.isValidUuid(id.get())) {
            errors.addError(fieldKey, TypedError.notFound(messages
                    .getMessage("errors.resource.not.found", null, locale)));
        } else {
            setter.accept(id.get());
        }
    }

    private void validateFolderId(Supplier<Optional<String>> getter, Consumer<String> setter,
                                  ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> id = getter.get();
        if (id.isEmpty()) {
            return;
        }
        if (!Uuids.isValidUuid(id.get())) {
            errors.addError(fieldKey, TypedError.notFound(messages
                    .getMessage("errors.folder.not.found", null, locale)));
        } else {
            setter.accept(id.get());
        }
    }

    private void validateAmount(Supplier<Optional<Long>> getter, Consumer<Long> setter,
                                ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<Long> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (value.get() <= 0L) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }
}
