package ru.yandex.intranet.d.services.delivery.model;

import java.util.Objects;
import java.util.Optional;

import com.google.common.base.Preconditions;

/**
 * Deliverable request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class DeliverableRequest {

    private final Long serviceId;
    private final String folderId;
    private final String resourceId;
    private final DeliverableDelta delta;
    private final DeliverableMetaRequest meta;

    public DeliverableRequest(Long serviceId, String folderId, String resourceId, DeliverableDelta delta,
                              DeliverableMetaRequest meta) {
        this.serviceId = serviceId;
        this.folderId = folderId;
        this.resourceId = resourceId;
        this.delta = delta;
        this.meta = meta;
    }

    public static Builder builder() {
        return new Builder();
    }

    public Optional<Long> getServiceId() {
        return Optional.ofNullable(serviceId);
    }

    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    public String getResourceId() {
        return resourceId;
    }

    public DeliverableDelta getDelta() {
        return delta;
    }

    public DeliverableMetaRequest getMeta() {
        return meta;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableRequest that = (DeliverableRequest) o;
        return Objects.equals(serviceId, that.serviceId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(delta, that.delta) &&
                Objects.equals(meta, that.meta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(serviceId, folderId, resourceId, delta, meta);
    }

    @Override
    public String toString() {
        return "DeliverableRequest{" +
                "serviceId=" + serviceId +
                ", folderId='" + folderId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", delta=" + delta +
                ", meta=" + meta +
                '}';
    }

    public static final class Builder {

        private Long serviceId;
        private String folderId;
        private String resourceId;
        private DeliverableDelta delta;
        private DeliverableMetaRequest meta;

        private Builder() {
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder delta(DeliverableDelta delta) {
            this.delta = delta;
            return this;
        }

        public Builder meta(DeliverableMetaRequest meta) {
            this.meta = meta;
            return this;
        }

        public DeliverableRequest build() {
            Preconditions.checkNotNull(resourceId, "ResourceId is required");
            Preconditions.checkNotNull(delta, "Delta is required");
            Preconditions.checkNotNull(meta, "Meta is required");
            Preconditions.checkArgument(serviceId != null || folderId != null,
                    "Either ServiceId or FolderId is required");
            return new DeliverableRequest(serviceId, folderId, resourceId, delta, meta);
        }

    }

}
