package ru.yandex.intranet.d.services.delivery.model;

import java.util.Comparator;
import java.util.Objects;
import java.util.Optional;

import org.jetbrains.annotations.NotNull;

import ru.yandex.intranet.d.model.delivery.DeliverableMetaRequestModel;
import ru.yandex.intranet.d.model.delivery.DeliverableRequestModel;
import ru.yandex.intranet.d.model.delivery.provide.DeliveryAndProvideDestinationModel;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideDestination;

/**
 * Deliverable request group key.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class DeliverableRequestGroupKey implements Comparable<DeliverableRequestGroupKey> {

    public static final Comparator<DeliverableRequestGroupKey> COMPARATOR = new DeliverableRequestGroupKeyComparator();

    private final DeliverableMetaRequestModel meta;
    private final Long serviceId;
    private final String folderId;

    public DeliverableRequestGroupKey(DeliverableRequestModel request) {
        this.meta = request.getMeta();
        this.serviceId = request.getServiceId().orElse(null);
        this.folderId = request.getFolderId().orElse(null);
    }

    public DeliverableRequestGroupKey(DeliveryAndProvideDestinationModel destination) {
        this.meta = DeliverableMetaRequestModel.builder()
                .quotaRequestId(destination.getMeta().getQuotaRequestId())
                .bigOrderId(destination.getMeta().getBigOrderId())
                .campaignId(destination.getMeta().getCampaignId())
                .build();
        this.serviceId = destination.getServiceId();
        this.folderId = destination.getFolderId();
    }

    public DeliverableRequestGroupKey(DeliveryAndProvideDestination destination) {
        this.meta = DeliverableMetaRequestModel.builder()
                .quotaRequestId(destination.getMeta().getQuotaRequestId())
                .bigOrderId(destination.getMeta().getBigOrderId())
                .campaignId(destination.getMeta().getCampaignId())
                .build();
        this.serviceId = destination.getServiceId();
        this.folderId = destination.getFolderId();
    }

    public DeliverableMetaRequestModel getMeta() {
        return meta;
    }

    public Optional<Long> getServiceId() {
        return Optional.ofNullable(serviceId);
    }

    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableRequestGroupKey that = (DeliverableRequestGroupKey) o;
        return Objects.equals(meta, that.meta) &&
                Objects.equals(serviceId, that.serviceId) &&
                Objects.equals(folderId, that.folderId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(meta, serviceId, folderId);
    }

    @Override
    public String toString() {
        return "DeliverableRequestGroupKey{" +
                "meta=" + meta +
                ", serviceId=" + serviceId +
                ", folderId='" + folderId + '\'' +
                '}';
    }

    @Override
    public int compareTo(@NotNull DeliverableRequestGroupKey o) {
        if (this.serviceId != null && o.serviceId == null) {
            return 1;
        }
        if (this.serviceId == null && o.serviceId != null) {
            return -1;
        }
        if (this.serviceId != null) {
            int compare = this.serviceId.compareTo(o.serviceId);
            if (compare != 0) {
                return compare;
            }
        }
        if (this.folderId != null && o.folderId == null) {
            return 1;
        }
        if (this.folderId == null && o.folderId != null) {
            return -1;
        }
        if (this.folderId != null) {
            int compare = this.folderId.compareTo(o.folderId);
            if (compare != 0) {
                return compare;
            }
        }
        return this.meta.compareTo(o.meta);
    }

    public static final class DeliverableRequestGroupKeyComparator implements Comparator<DeliverableRequestGroupKey> {

        private DeliverableRequestGroupKeyComparator() {
        }

        @Override
        public int compare(DeliverableRequestGroupKey l, DeliverableRequestGroupKey r) {
            return l.compareTo(r);
        }

    }

}
