package ru.yandex.intranet.d.services.delivery.model;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import com.google.common.base.Preconditions;

import ru.yandex.intranet.d.model.delivery.DeliveryRequestModel;
import ru.yandex.intranet.d.model.folders.FolderModel;
import ru.yandex.intranet.d.model.users.UserModel;

/**
 * Validated delivery request.
 *
 * @author Dmitriy Timashov <dm-tim@yandex-team.ru>
 */
public final class ValidatedDeliveryRequest {

    private final DeliveryRequestModel deliveryRequest;
    private final List<FolderModel> newDefaultFolders;
    private final UserModel author;

    public ValidatedDeliveryRequest(DeliveryRequestModel deliveryRequest, List<FolderModel> newDefaultFolders,
                                    UserModel author) {
        this.deliveryRequest = deliveryRequest;
        this.newDefaultFolders = newDefaultFolders;
        this.author = author;
    }

    public static Builder builder() {
        return new Builder();
    }

    public DeliveryRequestModel getDeliveryRequest() {
        return deliveryRequest;
    }

    public List<FolderModel> getNewDefaultFolders() {
        return newDefaultFolders;
    }

    public UserModel getAuthor() {
        return author;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ValidatedDeliveryRequest that = (ValidatedDeliveryRequest) o;
        return Objects.equals(deliveryRequest, that.deliveryRequest) &&
                Objects.equals(newDefaultFolders, that.newDefaultFolders) && Objects.equals(author, that.author);
    }

    @Override
    public int hashCode() {
        return Objects.hash(deliveryRequest, newDefaultFolders, author);
    }

    @Override
    public String toString() {
        return "ValidatedDeliveryRequest{" +
                "deliveryRequest=" + deliveryRequest +
                ", newDefaultFolders=" + newDefaultFolders +
                ", author=" + author +
                '}';
    }

    public static final class Builder {

        private final Map<Long, FolderModel> newDefaultFolders = new HashMap<>();

        private DeliveryRequestModel deliveryRequest;
        private UserModel author;

        private Builder() {
        }

        public Builder deliveryRequest(DeliveryRequestModel deliveryRequest) {
            this.deliveryRequest = deliveryRequest;
            return this;
        }

        public Builder addNewDefaultFolder(FolderModel folder) {
            this.newDefaultFolders.put(folder.getServiceId(), folder);
            return this;
        }

        public Builder author(UserModel author) {
            this.author = author;
            return this;
        }

        public boolean hasNewDefaultFolder(long serviceId) {
            return newDefaultFolders.containsKey(serviceId);
        }

        public ValidatedDeliveryRequest build() {
            Preconditions.checkNotNull(deliveryRequest, "DeliveryRequest is required");
            Preconditions.checkNotNull(author, "Author is required");
            return new ValidatedDeliveryRequest(deliveryRequest, new ArrayList<>(newDefaultFolders.values()), author);
        }

    }

}
