package ru.yandex.intranet.d.services.delivery.provide;

import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.LongConsumer;
import java.util.function.Supplier;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import reactor.util.function.Tuple2;
import reactor.util.function.Tuples;

import ru.yandex.intranet.d.services.delivery.model.DeliverableDelta;
import ru.yandex.intranet.d.services.delivery.model.DeliverableMetaRequest;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideDestination;
import ru.yandex.intranet.d.services.delivery.model.provide.DeliveryAndProvideRequest;
import ru.yandex.intranet.d.util.Uuids;
import ru.yandex.intranet.d.util.result.ErrorCollection;
import ru.yandex.intranet.d.util.result.Result;
import ru.yandex.intranet.d.util.result.TypedError;
import ru.yandex.intranet.d.web.model.delivery.DeliverableDeltaDto;
import ru.yandex.intranet.d.web.model.delivery.DeliverableMetaRequestDto;
import ru.yandex.intranet.d.web.model.delivery.provide.DeliveryAndProvideDestinationDto;
import ru.yandex.intranet.d.web.model.delivery.provide.DeliveryAndProvideRequestDto;

/**
 * Delivery pre validation DTO service
 *
 * @author Evgenii Serov <evserov@yandex-team.ru>
 */
@Component
public class DeliveryAndProvidePreValidationService {
    private static final int MAX_TEXT_LENGTH = 1024;
    private static final int MAX_DELIVERABLES = 1000;

    private final MessageSource messages;

    public DeliveryAndProvidePreValidationService(@Qualifier("messageSource") MessageSource messages) {
        this.messages = messages;
    }

    public Result<DeliveryAndProvideRequest> preValidateProvideRequest(DeliveryAndProvideRequestDto request,
                                                                       Locale locale) {
        DeliveryAndProvideRequest.Builder builder = new DeliveryAndProvideRequest.Builder();
        ErrorCollection.Builder errors = ErrorCollection.builder();
        validateRequiredAndValidUuid(request::getDeliveryIdOptional, builder::deliveryId, errors, "deliveryId", locale);
        validateText(request::getAuthorUidOptional, builder::authorUid, errors, "authorUid", MAX_TEXT_LENGTH, locale);
        if (request.getDeliverablesOptional().isEmpty() || request.getDeliverablesOptional().get().isEmpty()) {
            errors.addError("deliverables", TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            List<DeliveryAndProvideDestinationDto> deliverables = Objects.requireNonNull(request.getDeliverables());

            Set<Tuple2<Tuple2<String, Long>, String>> resourceIdAccountIdSet = new HashSet<>();
            Long campaignId = null;
            Long requestId = null;
            int size = deliverables.size();
            for (int i = 0; i < size; i++) {
                DeliveryAndProvideDestinationDto destination = deliverables.get(i);
                preValidateDeliveryAndProvideDestination(builder, errors, destination, i, locale);

                if (destination.getResourceIdOptional().isPresent()
                        && destination.getAccountIdOptional().isPresent()
                        && destination.getMetaOptional().isPresent()
                        && destination.getMetaOptional().get().getBigOrderId().isPresent()
                        && !resourceIdAccountIdSet.add(Tuples.of(
                        Tuples.of(destination.getResourceIdOptional().orElseThrow(),
                                destination.getMetaOptional().get().getBigOrderId().get()),
                        destination.getAccountIdOptional().orElseThrow()))) {
                    errors.addError("deliverables." + i, TypedError.invalid(messages
                            .getMessage("errors.resource.id.duplicate.account.pair.are.not.allowed",
                                    new Object[]{destination.getAccountIdOptional().orElseThrow(),
                                            destination.getResourceIdOptional().orElseThrow(),
                                            destination.getMetaOptional().get().getBigOrderId().get()}, locale)));
                }

                if (destination.getMetaOptional().isPresent()
                        && destination.getMetaOptional().get().getCampaignId().isPresent()) {
                    if (campaignId == null) {
                        campaignId = destination.getMetaOptional().get().getCampaignId().get();
                    } else if (!campaignId.equals(destination.getMetaOptional().get().getCampaignId().get())) {
                        errors.addError("deliverables." + i + ".meta.campaignId", TypedError.invalid(messages
                                .getMessage("errors.different.campaign.id.not.allowed",
                                        new Object[]{destination.getMetaOptional().get().getCampaignId().get()},
                                        locale)));
                    }
                }

                if (destination.getMetaOptional().isPresent()
                        && destination.getMetaOptional().get().getQuotaRequestId().isPresent()) {
                    if (requestId == null) {
                        requestId = destination.getMetaOptional().get().getQuotaRequestId().get();
                    } else if (!requestId.equals(destination.getMetaOptional().get().getQuotaRequestId().get())) {
                        errors.addError("deliverables." + i + ".meta.quotaRequestId", TypedError.invalid(messages
                                .getMessage("errors.different.request.id.not.allowed",
                                        new Object[]{destination.getMetaOptional().get().getQuotaRequestId().get()},
                                        locale)));
                    }
                }
            }
            if (size > MAX_DELIVERABLES) {
                errors.addError("deliverables", TypedError.invalid(messages
                        .getMessage("errors.too.many.list.elements", null, locale)));
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(builder.build());
    }

    private void preValidateDeliveryAndProvideDestination(DeliveryAndProvideRequest.Builder builder,
                                                          ErrorCollection.Builder errors,
                                                          DeliveryAndProvideDestinationDto destination,
                                                          int index, Locale locale) {
        if (destination == null) {
            errors.addError("deliverables." + index, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        DeliveryAndProvideDestination.Builder destinationBuilder = new DeliveryAndProvideDestination.Builder();
        ErrorCollection.Builder destinationErrors = ErrorCollection.builder();
        validateRequired(destination::getServiceIdOptional, destinationBuilder::serviceId,
                destinationErrors, "deliverables." + index + ".serviceId", locale);
        validateRequiredAndValidUuid(destination::getProviderIdOptional, destinationBuilder::providerId,
                destinationErrors, "deliverables." + index + ".providerId", locale);
        validateRequiredAndValidUuid(destination::getFolderIdOptional, destinationBuilder::folderId,
                destinationErrors, "deliverables." + index + ".folderId", locale);
        validateRequiredAndValidUuid(destination::getAccountIdOptional, destinationBuilder::accountId,
                destinationErrors, "deliverables." + index + ".accountId", locale);
        validateRequiredAndValidUuid(destination::getResourceIdOptional, destinationBuilder::resourceId,
                destinationErrors, "deliverables." + index + ".resourceId", locale);
        validateDelta(destination::getDeltaOptional, destinationBuilder::delta, destinationErrors,
                "deliverables." + index + ".delta", locale);
        validateMeta(destination::getMetaOptional, destinationBuilder::meta, destinationErrors,
                "deliverables." + index + ".meta", locale);
        if (destinationErrors.hasAnyErrors()) {
            errors.add(destinationErrors);
            return;
        }
        builder.addDeliverable(destinationBuilder.build());
    }

    private <T> void validateRequired(Supplier<Optional<T>> getter, Consumer<T> setter,
                                      ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<T> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private void validateRequiredAndValidUuid(Supplier<Optional<String>> getter, Consumer<String> setter,
                                              ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<String> id = getter.get();
        if (id.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (!Uuids.isValidUuid(id.get())) {
            errors.addError(fieldKey, TypedError.notFound(messages
                    .getMessage("errors.invalid.uuid", new Object[]{id.get()}, locale)));
        } else {
            setter.accept(id.get());
        }
    }

    private void validateDelta(Supplier<Optional<DeliverableDeltaDto>> deltaSupplier,
                               Consumer<DeliverableDelta> setter, ErrorCollection.Builder errors,
                               String fieldPrefix, Locale locale) {
        Optional<DeliverableDeltaDto> deltaO = deltaSupplier.get();
        if (deltaO.isEmpty()) {
            errors.addError(fieldPrefix, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        DeliverableDeltaDto delta = deltaO.get();
        DeliverableDelta.Builder deltaBuilder = DeliverableDelta.builder();
        ErrorCollection.Builder deltaErrors = ErrorCollection.builder();
        validateAmount(delta::getAmount, deltaBuilder::amount, deltaErrors, fieldPrefix + ".amount", locale);
        validateText(delta::getUnitKey, deltaBuilder::unitKey, deltaErrors,
                fieldPrefix + ".unitKey", MAX_TEXT_LENGTH, locale);
        if (deltaErrors.hasAnyErrors()) {
            errors.add(deltaErrors);
            return;
        }
        setter.accept(deltaBuilder.build());
    }

    private void validateMeta(Supplier<Optional<DeliverableMetaRequestDto>> metaSupplier,
                              Consumer<DeliverableMetaRequest> setter, ErrorCollection.Builder errors,
                              String fieldPrefix, Locale locale) {
        Optional<DeliverableMetaRequestDto> metaO = metaSupplier.get();
        if (metaO.isEmpty()) {
            errors.addError(fieldPrefix, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
            return;
        }
        DeliverableMetaRequestDto meta = metaO.get();
        DeliverableMetaRequest.Builder metaBuilder = DeliverableMetaRequest.builder();
        ErrorCollection.Builder metaErrors = ErrorCollection.builder();
        validateRequired(meta::getQuotaRequestId, metaBuilder::quotaRequestId, metaErrors,
                fieldPrefix + ".quotaRequestId", locale);
        validateRequired(meta::getCampaignId, metaBuilder::campaignId, metaErrors,
                fieldPrefix + ".campaignId", locale);
        validateRequired(meta::getBigOrderId, metaBuilder::bigOrderId, metaErrors,
                fieldPrefix + ".bigOrderId", locale);
        if (metaErrors.hasAnyErrors()) {
            errors.add(metaErrors);
            return;
        }
        setter.accept(metaBuilder.build());
    }

    private void validateAmount(Supplier<Optional<Long>> getter, LongConsumer setter,
                                ErrorCollection.Builder errors, String fieldKey, Locale locale) {
        Optional<Long> value = getter.get();
        if (value.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (value.get() <= 0L) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.number.must.be.positive", null, locale)));
        } else {
            setter.accept(value.get());
        }
    }

    private void validateText(Supplier<Optional<String>> getter, Consumer<String> setter,
                              ErrorCollection.Builder errors, String fieldKey, int maxLength, Locale locale) {
        Optional<String> text = getter.get();
        if (text.isEmpty()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.field.is.required", null, locale)));
        } else if (text.get().isBlank()) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.non.blank.text.is.required", null, locale)));
        } else if (text.get().length() > maxLength) {
            errors.addError(fieldKey, TypedError.invalid(messages
                    .getMessage("errors.text.is.too.long", null, locale)));
        } else {
            setter.accept(text.get());
        }
    }

    public Result<Void> validateRequiredListUuids(List<String> ids, Locale locale) {
        if (ids.isEmpty()) {
            return Result.failure(ErrorCollection.builder()
                    .addError(TypedError.badRequest(messages
                            .getMessage("errors.non.empty.list.is.required", null, locale)))
                    .build());
        }
        if (ids.size() > 1000) {
            return Result.failure(ErrorCollection.builder()
                    .addError(TypedError.badRequest(messages
                            .getMessage("errors.limit.is.too.large", null, locale)))
                    .build());
        }
        var errors = ErrorCollection.builder();
        for (String id : ids) {
            if (!Uuids.isValidUuid(id)) {
                errors.addError(TypedError.badRequest(
                        messages.getMessage("errors.invalid.uuid", new Object[]{id}, locale))
                );
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }
}
